package ru.yandex.msearch.proxy.api.async.mail.relevance.search;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.logging.Level;

import ru.yandex.json.parser.JsonException;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.proxy.api.async.mail.SearchSession;
import ru.yandex.msearch.proxy.api.async.mail.documents.Documents;
import ru.yandex.msearch.proxy.api.async.mail.documents.DocumentsGroup;

import ru.yandex.msearch.proxy.api.async.mail.relevance.RelevanceException;

import ru.yandex.msearch.proxy.api.async.mail.rules.AbstractSessionCallback;
import ru.yandex.msearch.proxy.api.async.mail.rules.RankingRule;

import ru.yandex.parser.uri.CgiParams;
import ru.yandex.util.string.StringUtils;

public class RankingSession extends AbstractSessionCallback<Documents> {
    protected final CgiParams params;
    protected final PrefixedLogger logger;
    protected final AbstractMailSearchRelevanceSorter sorter;

    public RankingSession(
        final SearchSession session,
        final AbstractMailSearchRelevanceSorter sorter)
    {
        super(session);

        this.logger = session.httpSession().logger()
            .addPrefix(RankingRule.LOG_PREFIX);

        this.params = session.params();
        this.sorter = sorter;
    }

    protected int calculateTopResultsSize(
        final List<RankedDocument> serp,
        final int rankedSize)
    {
        int pos = Integer.MAX_VALUE;
        if (serp.size() > rankedSize) {
            pos = serp.get(rankedSize).nonRankedPos();
        }

        for (int i = rankedSize - 1; i >= 0; i--) {
            if (serp.get(i).nonRankedPos() > pos) {
                return i + 1;
            }

            pos = serp.get(i).nonRankedPos();
        }

        return 0;
    }

    protected Documents rank(final Documents documents) {
        if (sorter == null
            || documents.size() < sorter.minSerpSize()
            || documents.size() == 0
            || sorter.rankedPositions() == 0)
        {
            return documents;
        }

        try {
            return rankDocuments(documents);
        } catch (JsonException | RelevanceException | NumberFormatException e) {
            logger.log(Level.WARNING, "Unable to rank documents", e);
            return documents;
        }
    }

    protected RankingDocuments buildTopRelevant(
        final List<RankedDocument> documents)
        throws JsonException, RelevanceException
    {
        int rankedPositions = sorter.rankedPositions();

        boolean falbackDateOrder = false;
        String request = session.requestInfo().options().request();
        if (request != null
            && !sorter.rankEmailRequests()
            && request.contains("@"))
        {
            falbackDateOrder = true;
        }

        if (rankedPositions < 0 || documents.size() < sorter.minSerpSize()) {
            falbackDateOrder = true;
        }

        if (falbackDateOrder) {
            Collections.sort(documents, RankedDocument.DATE_COMPARATOR);
            return new RankingDocuments(documents);
        }

        int topRelevantSize;
        if (rankedPositions < documents.size()) {
            Collections.sort(
                documents.subList(rankedPositions, documents.size()),
                RankedDocument.DATE_COMPARATOR);
             topRelevantSize =
                 calculateTopResultsSize(documents, rankedPositions);
        } else {
            topRelevantSize =
                calculateTopResultsSize(documents, documents.size());
        }

        logger.info("Top relevant size " + topRelevantSize);

        return new RankingDocuments(documents, topRelevantSize);
    }

    protected RankingDocuments buildPlain(
        final List<RankedDocument> documents)
        throws JsonException, RelevanceException
    {
        int rankedPositions = sorter.rankedPositions();

        logger.info("Plain, Ranked positions " + sorter.rankedPositions());

        if (rankedPositions < 0) {
            return new RankingDocuments(documents);
        }

        RankingDocuments result;

        if (rankedPositions >= documents.size()) {
            result = new RankingDocuments(documents);
        } else {
            result = new RankingDocuments(documents.size(), 0);

            RankedDocument[] sourceSorted =
                new RankedDocument[documents.size() - rankedPositions];
            for (int i = rankedPositions; i < documents.size(); i++) {
                sourceSorted[i - rankedPositions] = documents.get(i);
            }

            Arrays.sort(sourceSorted, RankedDocument.DATE_COMPARATOR);
            for (int i = 0; i < rankedPositions; i++) {
                result.add(documents.get(i));
            }

            for (int i = 0; i < sourceSorted.length; i++) {
                result.add(sourceSorted[i]);
            }

            logger.info(
                "Ranked by model: "
                    + String.valueOf(documents.size() - sourceSorted.length));
        }

        return result;
    }

    protected RankingDocuments rankDocuments(
        final Documents documents)
        throws JsonException, RelevanceException
    {
        List<? extends DocumentsGroup> docList = documents.sort();
        long startRanking = System.currentTimeMillis();

        String request = session.requestInfo().options().request();
        if (request != null) {
            request = request.toLowerCase(Locale.ENGLISH);
        }

        // gather documents
        List<RankedDocument> ranked = sorter.sort(docList, request);

        long scoreTime = System.currentTimeMillis() - startRanking;
        logger.info("Ranking by model took " + scoreTime + "ms");

        RankingDocuments result;
        if (sorter.topRelevant()) {
            result = buildTopRelevant(ranked);
        } else {
            result = buildPlain(ranked);
        }

        result.total(documents.total());

        long totalRankingMs = System.currentTimeMillis() - startRanking;
        logger.info(
            StringUtils.concat(
                "Total ranking took ",
                String.valueOf(totalRankingMs),
                " ms"));

        return result;
    }

    @Override
    public void completed(final Documents documents) {
        session.callback().completed(rank(documents));
    }
}
