package ru.yandex.msearch.proxy.api.async.mail.relevance.search.factors;

import java.util.ArrayList;
import java.util.List;

import ru.yandex.json.parser.JsonException;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.proxy.api.async.mail.PerfieldScorer;
import ru.yandex.msearch.proxy.api.async.mail.documents.Document;

import ru.yandex.msearch.proxy.api.async.mail.relevance.FactorGroup;
import ru.yandex.msearch.proxy.api.async.mail.relevance.FactorGroupFactory;
import ru.yandex.msearch.proxy.api.async.mail.relevance.NormalizedFactorGroup;
import ru.yandex.msearch.proxy.api.async.mail.relevance.RelevanceException;
import ru.yandex.parser.uri.CgiParams;

import ru.yandex.search.result.SearchDocument;

import ru.yandex.util.string.StringUtils;

public class LuceneFieldFactorFactory implements FactorGroupFactory {
    private static final int FACTORS_NUM = 6;

    private static final String[] NAME_POSTFIXES =
        {
            "_freq_e", "_hits_e", "_score_e",
            "_freq_ne", "_hits_ne", "_score_ne"
        };

    private final String name;
    private final String field;
    private final List<String> fields;
    private final List<String> names;

    private final int startIndex;

    public LuceneFieldFactorFactory(final String name, final int startIndex) {
        this(name, name, startIndex);
    }

    public LuceneFieldFactorFactory(
        final String name,
        final String field,
        final int startIndex)
    {
        this.name = name;
        this.field = field;
        this.startIndex = startIndex;

        this.names = new ArrayList<>(FACTORS_NUM);
        this.fields = new ArrayList<>(FACTORS_NUM);

        for (int i = 0; i < FACTORS_NUM; i++) {
            this.fields.add(StringUtils.concat(
                "#",
                field,
                PerfieldScorer.FIELD_POSTFIXES[i]));

            this.names.add(StringUtils.concat(field, NAME_POSTFIXES[i]));
        }
    }

    @Override
    public FactorGroup create(
        final CgiParams params,
        final PrefixedLogger logger)
    {
        return new LuceneFieldFactor(fields, startIndex);
    }

    @Override
    public int size() {
        return FACTORS_NUM;
    }

    @Override
    public int startIndex() {
        return startIndex;
    }

    @Override
    public List<String> names() {
        return names;
    }

    @Override
    public String groupName() {
        return name;
    }

    public String field() {
        return field;
    }

    @Override
    public String toString() {
        return name + ",";
    }

    private final class LuceneFieldFactor
        implements NormalizedFactorGroup<Document>
    {
        private final List<String> fields;
        private final int startIndex;
        private double[] result;

        public LuceneFieldFactor(
            final List<String> fields,
            final int startIndex)
        {
            this.fields = fields;
            this.startIndex = startIndex;
        }

        private double retrieveValue(
            final SearchDocument doc,
            final String field)
            throws RelevanceException
        {
            String value = doc.attrs().getOrDefault(field, "0");
            if (value == null) {
                return 0;
            }

            try {
                return Double.parseDouble(value);
            } catch (NumberFormatException nfe) {
                throw new RelevanceException(
                    "Invalid " + field + " value " + value);
            }
        }

        @Override
        public void calc(
            final Document doc,
            final String request,
            final double[] res)
            throws RelevanceException, JsonException
        {
            result = new double[fields.size()];
            for (int i = 0; i < fields.size(); i++) {
                result[i] = this.retrieveValue(doc.doc(), fields.get(i));
                if (res != null) {
                    res[startIndex + i] = result[i];
                }
            }
        }
    }
}