package ru.yandex.msearch.proxy.api.async.mail.result;

import java.io.IOException;
import java.util.List;

import ru.yandex.dbfields.FilterSearchFields;
import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;

import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonWriter;

import ru.yandex.msearch.proxy.api.async.mail.documents.Document;
import ru.yandex.msearch.proxy.highlight.Highlighter;

import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.util.string.StringUtils;

public class HighlightingAllFieldsPrinter implements MailFieldsPrinter {
    private static final String HIGHLIGHT = "_highlighted";
    private static final String DISPLAYNAME = "displayName";
    private static final String HID0 = "0";

    private final JsonWriter writer;
    private final List<String> request;
    private final Highlighter highlighter;

    public HighlightingAllFieldsPrinter(
        final JsonWriter writer,
        final Highlighter highlighter,
        final String request)
    {
        this.writer = writer;
        this.highlighter = highlighter;
        this.request = highlighter.prepareRequest(request);
    }

    protected void highlightAddress(
        final String field,
        final JsonMap doc)
        throws JsonException
    {
        boolean marked = false;
        JsonList addresses = doc.getListOrNull(field);
        if (addresses == null || addresses.size() <= 0) {
            return;
        }

        for (JsonObject addressObj: addresses) {
            JsonMap address = addressObj.asMap();
            String displayName = address.getString(DISPLAYNAME, "");
            JsonObject highlight =
                highlighter.highlight(displayName, request, true);
            if (highlight != null) {
                address.put(DISPLAYNAME + HIGHLIGHT, highlight);
                doc.put(field + HIGHLIGHT, JsonBoolean.TRUE);
                return;
            } else {
                String local =
                    address.getString(FilterSearchFields.ADDR_LOCAL, "");
                String domain =
                    address.getString(FilterSearchFields.ADDR_DOMAIN, "");
                highlight =
                    highlighter.highlight(
                        StringUtils.concat(local, '@', domain),
                        request,
                        false,
                        true);
                if (highlight != null && !marked) {
                    doc.put(field + HIGHLIGHT, JsonBoolean.TRUE);
                    marked = true;
                }
            }
        }
    }

    @Override
    public void accept(final Document doc) throws IOException, JsonException {
        JsonMap envelope = doc.envelope();

        String mailString =
            envelope.getString(MailMetaInfo.FIRSTLINE, "");
        JsonObject highlight =
            highlighter.highlight(mailString, request, true);
        if (highlight != null) {
            envelope.put(
                MailMetaInfo.FIRSTLINE + HIGHLIGHT,
                highlight);
        }

        JsonMap subjectInfo =
            envelope.getMapOrNull(FilterSearchFields.SUBJECT_INFO);
        if (subjectInfo != null) {
            mailString = subjectInfo.getString(MailMetaInfo.SUBJECT, null);
            if (mailString != null && !mailString.isEmpty()) {
                highlight =
                    highlighter.highlight(mailString, request, true);
                if (highlight != null) {
                    subjectInfo.put(
                        MailMetaInfo.SUBJECT + HIGHLIGHT,
                        highlight);
                }
            }
        }
        mailString = envelope.getString(MailMetaInfo.SUBJECT, "");
        highlight = highlighter.highlight(mailString, request, true);

        if (highlight != null) {
            envelope.put(
                MailMetaInfo.SUBJECT + HIGHLIGHT,
                highlight);
        }

        highlightAddress(MailMetaInfo.FROM, envelope);
        highlightAddress(MailMetaInfo.TO, envelope);
        highlightAddress(MailMetaInfo.CC, envelope);
        highlightAddress(MailMetaInfo.BCC, envelope);

        String hid =
            doc.doc().attrs().getOrDefault(MailIndexFields.HID, HID0);

        boolean hitInAttach = false;
        if (!HID0.equalsIgnoreCase(hid)) {
            // check if it is really hit in attach instead of hit in meta
            try {
                hitInAttach =
                    Double.parseDouble(
                        doc.doc().attrs().getOrDefault(
                            "#body_text_exact_hits",
                            "0.0")) > 0;
                hitInAttach |=
                    Double.parseDouble(
                        doc.doc().attrs().getOrDefault(
                            "#body_text_non_exact_hits",
                            "0.0")) > 0;
            } catch (NumberFormatException nfe) {
                hitInAttach = false;
            }
        }

        JsonList attachments =
            envelope.getListOrNull(MailMetaInfo.ATTACHMENTS);

        for (JsonObject attachmentObj: attachments) {
            JsonMap attach = attachmentObj.asMap();
            if (attach == null) {
                continue;
            }

            if (hid.equalsIgnoreCase(
                    attach.getString(FilterSearchFields.ATTACH_HID, HID0)))
            {
                if (hitInAttach) {
                    attach.put(
                        FilterSearchFields.ATTACH_HID + "_content" + HIGHLIGHT,
                        JsonBoolean.TRUE);
                }
            }

            mailString =
                attach.getString(FilterSearchFields.ATTACH_FILENAME, "");
            highlight = highlighter.highlight(
                mailString,
                request,
                false,
                false);
            if (highlight != null) {
                attach.put(
                    FilterSearchFields.ATTACH_FILENAME + HIGHLIGHT,
                    highlight);
            }
        }

        envelope.writeValue(writer);
    }
}
