package ru.yandex.msearch.proxy.api.async.mail.result;

import java.io.IOException;
import java.io.OutputStreamWriter;

import java.net.SocketTimeoutException;
import java.nio.charset.CodingErrorAction;

import java.util.List;

import java.util.logging.Level;

import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NByteArrayEntity;

import ru.yandex.http.util.BadGatewayException;
import ru.yandex.http.util.GatewayTimeoutException;

import ru.yandex.http.util.nio.NByteArrayEntityFactory;
import ru.yandex.io.DecodableByteArrayOutputStream;

import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.SearchProxyAccessLoggerConfigDefaults;

import ru.yandex.msearch.proxy.api.async.Session;

import ru.yandex.msearch.proxy.api.async.mail.RequestInfo;
import ru.yandex.msearch.proxy.api.async.mail.documents.Document;
import ru.yandex.msearch.proxy.api.async.mail.documents.Documents;
import ru.yandex.msearch.proxy.api.async.mail.documents.DocumentsGroup;

public class ResultPrinter implements FutureCallback<Documents> {
    protected final RequestInfo requestInfo;
    protected final Session session;
    protected final HttpRequest request;

    public ResultPrinter(
        final RequestInfo requestInfo)
    {
        this.requestInfo = requestInfo;
        session = requestInfo.session();
        request = requestInfo.request();
    }

    @Override
    public void cancelled() {
        session.logger().warning("Request cancelled: " + request);
    }

    // CSOFF: FinalParameters
    @Override
    public void failed(Exception e) {
        session.logger().log(
            Level.WARNING,
            "Request failed: " + request + '\n'
            + session.requestsListener().details(),
            e);
        if (e instanceof IOException) {
            if (e instanceof SocketTimeoutException) {
                e = new GatewayTimeoutException(e);
            } else {
                e = new BadGatewayException(e);
            }
        }
        session.handleException(e);
    }
    // CSON: FinalParameters

    protected void prologue(final JsonWriter writer, final Documents docs)
        throws IOException
    {
        writer.startObject();
        writeDetails(writer, docs);
        writer.key("envelopes");
        writer.startArray();
    }

    protected void writeDocuments(
        final JsonWriter writer,
        final MailFieldsPrinter printer,
        final Documents docs)
        throws JsonException, IOException
    {
        prologue(writer, docs);
        writeDocuments(printer, docs.sort());
        epilogue(writer);
    }

    protected void writeDocuments(
        final MailFieldsPrinter printer,
        final List<? extends DocumentsGroup> docs)
        throws JsonException, IOException
    {
        int pos = 0;

        int threshold = requestInfo.offset() + requestInfo.length();
        for (DocumentsGroup group: docs) {
            for (Document doc: group) {
                if (pos >= threshold) {
                    return;
                }

                if (pos >= requestInfo.offset()) {
                    printer.accept(doc);
                }
                ++pos;
            }
        }
    }

    protected void epilogue(final JsonWriter writer) throws IOException {
        writer.endArray();
        writer.endObject();
    }

    protected MailFieldsPrinter extract(final JsonWriter writer) {
        return MailFieldsPrinterExtractor.extract(writer, requestInfo);
    }

    @Override
    public void completed(final Documents docs) {
        session.logger().info(
            "Request completed: " + request
                + ", SERP size: " + docs.size());

        session.setSessionInfo(
            SearchProxyAccessLoggerConfigDefaults.HITS_COUNT,
            Long.toString(docs.size()));
        DecodableByteArrayOutputStream out =
            new DecodableByteArrayOutputStream();
        try (OutputStreamWriter outWriter =
                new OutputStreamWriter(
                    out,
                    requestInfo.acceptedCharset().newEncoder()
                        .onMalformedInput(CodingErrorAction.REPLACE)
                        .onUnmappableCharacter(CodingErrorAction.REPLACE));
            JsonWriter writer = requestInfo.jsonType().create(outWriter))
        {
            MailFieldsPrinter printer = extract(writer);
            // TODO: make group.sort() return Collection<Collection<Documents>>
            // or something
            writeDocuments(writer, printer, docs);
        } catch (IOException | JsonException e) {
            failed(e);
            return;
        }
        NByteArrayEntity entity =
            out.processWith(NByteArrayEntityFactory.INSTANCE);
        entity.setContentType(
            ContentType.APPLICATION_JSON.withCharset(
                requestInfo.acceptedCharset())
                .toString());
        session.response(HttpStatus.SC_OK, entity);
    }

    protected void writeDetails(final JsonWriter writer, final Documents docs)
        throws IOException
    {
        writer.key("details");
        writer.startObject();

        writer.key("crc32");
        writer.value("0");

        if (requestInfo.hasPage()) {
            writer.key("pager");
            writer.startObject();
            writer.key("items-per-page");
            writer.value(requestInfo.length());
            writer.key("prev");
            if (requestInfo.offset() > 0) {
                writer.startObject();
                writer.key("n");
                writer.value(requestInfo.offset() / requestInfo.length());
                writer.endObject();
            } else {
                writer.value("");
            }
            writer.key("next");
            if (docs.size() >= requestInfo.offset() + requestInfo.length()) {
                writer.startObject();
                writer.key("n");
                writer.value(requestInfo.offset() / requestInfo.length() + 2);
                writer.endObject();
            } else {
                writer.value("");
            }
            writer.endObject();
        } else {
            writer.key("search-limits");
            writer.startObject();
            writer.key("offset");
            writer.value(requestInfo.offset());
            writer.key("length");
            writer.value(requestInfo.length());
            writer.endObject();
        }

        writer.key("search-options");
        writer.value(requestInfo.options());
        writer.key("total-found");
        writer.value(docs.total());
        if (docs.relevant() > 0) {
            writer.key("top-relevant");
            writer.value(docs.relevant());
        }

        writer.endObject();
    }

}

