package ru.yandex.msearch.proxy.api.async.mail.rules;

import java.net.URISyntaxException;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.message.BasicHeader;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.parser.JsonException;
import ru.yandex.msearch.proxy.api.async.SubrequestsCounter;
import ru.yandex.msearch.proxy.api.async.mail.SearchRequest;
import ru.yandex.msearch.proxy.api.async.mail.SearchSession;
import ru.yandex.msearch.proxy.api.async.mail.documents.BasicDocument;
import ru.yandex.msearch.proxy.api.async.mail.documents.Document;
import ru.yandex.msearch.proxy.api.async.mail.documents.Documents;
import ru.yandex.msearch.proxy.api.async.mail.searcher.AbstractPlainSearcher;
import ru.yandex.msearch.proxy.api.async.mail.searcher.PlainSearcher;
import ru.yandex.msearch.proxy.api.async.mail.searcher.ProducerParallelSearcher;
import ru.yandex.msearch.proxy.document.MailSearchDocument;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;
import ru.yandex.stater.RequestInfo;
import ru.yandex.stater.RequestsStater;
import ru.yandex.util.timesource.TimeSource;

public class FilterPlainSearchCallback extends AbstractPlainSearchCallback {
    private final SubrequestsCounter subrequests = new SubrequestsCounter();
    private final RequestsStater filterSearchResponsesStater;
    private final PlainSearcher searcher;
    private final RuleContext context;
    //private final String ticket;
    protected final Collector collector;
    protected final AsyncClient filterSearchClient;
    protected boolean hasMoreData = false;

    public FilterPlainSearchCallback(
        final SearchSession session,
        final RuleContext context,
        final SearchRequest request)
        throws BadRequestException
    {
        this(
            session,
            context,
            request,
            new Collector(
                session.requestInfo(),
                request,
                context.documentsFactory()),
            new ProducerParallelSearcher(
                session.httpSession(),
                session.params(),
                request.user()));
    }

    public FilterPlainSearchCallback(
        final SearchSession session,
        final RuleContext context,
        final SearchRequest request,
        final Collector collector,
        final PlainSearcher searcher)
        throws BadRequestException
    {
        super(session, context, request);

        filterSearchResponsesStater =
            context.server().filterSearchResponsesStater();
        filterSearchClient =
            context.server().filterSearchClient(request.corp());
        this.collector = collector;
        this.searcher = searcher;
        this.context = context;
    }

    public RequestsStater filterSearchResponsesStater() {
        return filterSearchResponsesStater;
    }

    @Override
    protected Function<? super HttpHost, ? extends HttpAsyncRequestProducer> nextRequest() {
        return new BasicAsyncRequestProducerGenerator(collector.nextRequest());
    }

    @Override
    protected PlainSearcher searcher() {
        return searcher;
    }

    protected boolean filter(final Map<String, SearchDocument> docs) {
        StringBuilder sb =
            new StringBuilder(request.filterSearchRequest());
        for (String mid: docs.keySet()) {
            sb.append("&mids=");
            sb.append(mid);
        }
        try {
            filterSearchClient.execute(
                new HeaderAsyncRequestProducerSupplier(
                new AsyncGetURIRequestProducerSupplier(new String(sb)),
                    new BasicHeader(YandexHeaders.X_YA_SERVICE_TICKET,
                        context.server().filterSearchTvm2Ticket(request.corp()))),
                JsonAsyncTypesafeDomConsumerFactory.INTERNING_OK,
                session.httpSession().requestsListener()
                    .createContextGeneratorFor(filterSearchClient),
                new FilterSearchCallback(session, docs, this));
            return true;
        } catch (URISyntaxException e) {
            failed(e);
            return false;
        }
    }

    private void addDoc(
        final Map<String, SearchDocument> docs,
        final SearchDocument doc)
    {
        String mid = doc.attrs().get("mid");
        if (mid != null) {
            docs.put(mid, new MailSearchDocument(doc, mid));
        }
    }

    @Override
    public synchronized void completed(final SearchResult result) {
        if (done) {
            return;
        }
        stater.accept(
            new RequestInfo(
                TimeSource.INSTANCE.currentTimeMillis(),
                HttpStatus.SC_OK,
                start,
                start,
                0L,
                result.hitsArray().size()));
        hasMoreData = result.hitsArray().size() >= collector.length();

        int sent = sendSubrequests(result);
        if (sent < 0) {
            return;
        }

        if (collector.requests() <= 1) {
            collector.documents().total(result.hitsCount());
        }

        if (subrequests.sent(sent)) {
            iterationCompleted();
        }
    }

    protected int sendSubrequests(final SearchResult result) {
        List<SearchDocument> hitsArray = result.hitsArray();
        int hitsCount = hitsArray.size();
        int batchSize = request.filterSearchConfig().batchSize();

        int subrequests = 0;
        int pos = 0;
        while (pos < hitsCount) {
            Map<String, SearchDocument> docs =
                new LinkedHashMap<>(batchSize << 1);
            while (pos < hitsCount && docs.size() < batchSize) {
                addDoc(docs, hitsArray.get(pos++));
            }

            if (hitsCount - pos < batchSize) {
                while (pos < hitsCount) {
                    addDoc(docs, hitsArray.get(pos++));
                }
            }

            if (!docs.isEmpty()) {
                if (filter(docs)) {
                    ++subrequests;
                } else {
                    return -1;
                }
            }
        }
        return subrequests;
    }

    public void subrequestDone() {
        if (subrequests.received()) {
            iterationCompleted();
        }
    }

    protected synchronized void iterationCompleted() {
        if (done) {
            return;
        }
        if (hasMoreData && collector.needMoreDocuments()) {
            sendNextRequest();
        } else {
            done = true;
            session.callback().completed(buildResultDocuments());
        }
    }

    protected Documents buildResultDocuments() {
        Documents documents = collector.documents();
        if (documents.size() < collector.needDocuments()
            && documents.size() < documents.total())
        {
            documents.total(documents.size());
        }

        return collector.documents();
    }

    public void document(
        final String mid,
        final SearchDocument doc,
        final JsonMap envelope)
        throws JsonException
    {
        Document groupDoc = new BasicDocument(mid, doc, envelope);
        synchronized (collector) {
            collector.documents().add(mid, groupDoc);
        }
    }

    public void reduceTotalDocs(final int reducement) {
        synchronized (collector) {
            Documents docs = collector.documents();
            docs.total(docs.total() - reducement);
        }
    }
}

