package ru.yandex.msearch.proxy.api.async.mail.rules;

import java.text.ParseException;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.joda.time.DateTime;
import org.joda.time.DateTimeConstants;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.client.wmi.Folders;
import ru.yandex.client.wmi.Labels;

import ru.yandex.msearch.proxy.api.async.suggest.lang.SuggestLanguagePack;
import ru.yandex.msearch.proxy.api.async.suggest.lang.SuggestMultiLanguagePack;
import ru.yandex.msearch.proxy.mail.SearchFilter;

import ru.yandex.parser.email.MailAliases;

import ru.yandex.parser.query.FieldQuery;
import ru.yandex.parser.query.QueryAtom;
import ru.yandex.parser.query.QueryParser;
import ru.yandex.parser.query.QueryToken;
import ru.yandex.parser.query.QuotedQuery;

import ru.yandex.parser.string.BooleanParser;
import ru.yandex.parser.string.DurationParser;

import ru.yandex.search.request.util.SearchRequestText;

import ru.yandex.util.string.StringUtils;

public class LuceneQueryConstructor extends AbstractLuceneQueryConstructor {
    private static final DateTimeFormatter DATE_FORMATTERS[] =
        new DateTimeFormatter[] {
            DateTimeFormat.forPattern("dd.MM.yy"),
            DateTimeFormat.forPattern("dd.MM.yyyy"),
            DateTimeFormat.forPattern("yyyy-MM-dd"),
            DateTimeFormat.forPattern("yyyy/MM/dd"),
            DateTimeFormat.forPattern("dd-MM-yyyy")
        };
    // Yandex.Mail was launched June 26, 2000
    private static final int TILL_YEAR = 2000;

    public static final Map<String, Integer> MONTHS;

    static {
        MONTHS = new HashMap<>();
        for (SuggestLanguagePack languagePack
            : SuggestMultiLanguagePack.LANGUAGES.values())
        {
            List<String> months = languagePack.months();
            for (int i = 0; i < months.size(); ++i) {
                MONTHS.put(months.get(i), i + 1);
            }
            months = languagePack.longMonths();
            for (int i = 0; i < months.size(); ++i) {
                MONTHS.put(months.get(i), i + 1);
            }
        }
    }

    private final Deque<Collection<String>> scopes = new ArrayDeque<>();
    private final Labels labels;
    private final Folders folders;

    public LuceneQueryConstructor(
        final LuceneQueryContext context,
        final Labels labels,
        final Folders folders)
    {
        super(context);
        this.labels = labels;
        this.folders = folders;
        scopes.push(context.defaultScope());
    }

    public String request() {
        return new String(request);
    }

    private List<String> luceneFieldsNames(final String field)
        throws ParseException
    {
        List<String> luceneFields;
        switch (field.toLowerCase(Locale.ROOT)) {
            case "from":
            case "от":
                luceneFields = Arrays.asList("hdr_from", "reply_to");
                break;
            case "to":
            case "кому":
                luceneFields = Collections.singletonList("hdr_to");
                break;
            case "cc":
            case "сс": // cyrillic 'с's for smartest users
            case "копия":
                luceneFields = Collections.singletonList("hdr_cc");
                break;
            case "bcc":
            case "скрытая-копия":
                luceneFields = Collections.singletonList("hdr_bcc");
                break;
            case "subject":
            case "тема":
                luceneFields = Collections.singletonList("hdr_subject");
                break;
            case "text":
            case "текст":
                luceneFields = Collections.singletonList("pure_body");
                break;
            case "attachment":
            case "filename":
            case "вложение":
                luceneFields = Arrays.asList("attachname", "attachments");
                break;
            case "type":
            case "тип":
                luceneFields = Collections.singletonList("message_type");
                break;
            case "mid":
                luceneFields = Collections.singletonList("mid_p");
                break;
            default:
                luceneFields = null;
                break;
        }
        return luceneFields;
    }

    private long parseTimestamp(final String str) throws ParseException {
        for (int i = 0; i < DATE_FORMATTERS.length; ++i) {
            try {
                return
                    DATE_FORMATTERS[i].withZone(context.timezone())
                        .parseMillis(str)
                    / DateTimeConstants.MILLIS_PER_SECOND;
            } catch (Throwable t) {
                // Keep calm and try next pattern
            }
        }
        throw new ParseException("Failed to parse timestamp from: " + str, 0);
    }

    private static long parseDuration(final String str) throws ParseException {
        try {
            return DurationParser.LONG.apply(str.toLowerCase(Locale.ROOT));
        } catch (Throwable t) {
            ParseException e =
                new ParseException("Failed to parse duration: " + str, 0);
            e.initCause(t);
            throw e;
        }
    }

    private static boolean parseFlag(final String str) throws ParseException {
        try {
            return BooleanParser.INSTANCE.apply(str);
        } catch (Throwable t) {
            throw new ParseException("Failed to parse flag: " + str, 0);
        }
    }

    public static String templatizeEmail(final String email) {
        final StringBuilder sb = new StringBuilder(email.length() + 1);
        int at = email.indexOf('@');
        if (at == -1) {
            sb.append(email);
        } else {
            String mailbox = email.substring(0, at);
            sb.append(mailbox);
            sb.append('*');
            sb.append('@');
            sb.append(MailAliases.INSTANCE.extractAndNormalizeDomain(email));
        }
        return new String(sb);
    }

    @Override
    public Void visit(final FieldQuery query) throws ParseException {
        List<String> fields = query.fields();
        QueryAtom subquery = query.query();
        if (fields.size() == 1) {
            boolean processed = true;
            switch (fields.get(0)) {
                case "folder":
                case "папка":
                    new FolderQueryConstructor().visit(subquery);
                    break;
                case "label":
                case "метка":
                    new LabelQueryConstructor().visit(subquery);
                    break;
                case "date-end":
                case "заканчивая-датой":
                    new DateEndQueryConstructor().visit(subquery);
                    break;
                case "before":
                case "before-date":
                case "date-before":
                case "до-даты":
                case "older":
                case "старее":
                    new DateBeforeQueryConstructor().visit(subquery);
                    break;
                case "date-begin":
                case "с-даты":
                case "after":
                case "после":
                case "newer":
                case "новее":
                    new DateBeginQueryConstructor().visit(subquery);
                    break;
                case "month":
                case "месяц":
                    new MonthQueryConstructor().visit(subquery);
                    break;
                case "year":
                case "год":
                    new YearQueryConstructor().visit(subquery);
                    break;
                case "older-than":
                case "старее-чем":
                    new OlderThanQueryConstructor().visit(subquery);
                    break;
                case "newer-than":
                case "новее-чем":
                    new NewerThanQueryConstructor().visit(subquery);
                    break;
                case "has-attachments":
                case "с-вложениями":
                    new HasAttachmentsQueryConstructor().visit(subquery);
                    break;
                case "unread":
                case "непрочитанные":
                    new UnreadQueryConstructor().visit(subquery);
                    break;
                case "filter":
                case "фильтр":
                    new FilterQueryConstructor().visit(subquery);
                    break;
                case "subscription-email":
                    new SubscriptionEmailQueryConstructor().visit(subquery);
                    break;
                case "shared":
                case "общие":
                    new SharedQueryConstructor().visit(subquery);
                    break;
                default:
                    processed = false;
                    break;
            }
            if (processed) {
                context.nonTrivial();
                return null;
            }
        }

        List<String> luceneFields = new ArrayList<>(fields.size() << 1);
        int unknownField = -1;
        for (int i = 0; i < fields.size(); ++i) {
            List<String> currentLuceneFields =
                luceneFieldsNames(fields.get(i));
            if (currentLuceneFields == null) {
                unknownField = i;
            } else {
                luceneFields.addAll(currentLuceneFields);
            }
        }

        if (unknownField == -1) {
            context.nonTrivial();
            scopes.push(luceneFields);
            visit(subquery);
            scopes.pop();
        } else if (subquery instanceof QueryToken) {
            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < fields.size(); ++i) {
                if (i != 0) {
                    sb.append(',');
                }
                sb.append(fields.get(i));
            }
            sb.append(':');
            sb.append(((QueryToken) subquery).text());
            processQueryToken(new String(sb));
        } else {
            throw new ParseException(
                "Unknown field: " + fields.get(unknownField),
                unknownField);
        }
        return null;
    }

    @Override
    public Void visit(final QuotedQuery query) throws ParseException {
        Collection<String> scope = scopes.peek();
        boolean empty = true;
        request.append('(');
        for (String field: scope) {
            if (empty) {
                empty = false;
            } else {
                request.append(OR);
            }
            request.append(field);
            request.append(':');
            request.append('"');
            boolean first = true;
            for (QueryToken token: query.tokens()) {
                if (first) {
                    first = false;
                } else {
                    request.append(' ');
                }
                request.append(
                    token.text().replace('"', ' ').replace('\\', ' '));
            }
            request.append('"');
        }
        request.append(')');
        return null;
    }

    private void processQueryToken(final String token) throws ParseException {
        if (token.chars().anyMatch(Character::isLetterOrDigit)) {
            String text = token.toLowerCase(Locale.ROOT);
            Collection<String> scope = scopes.peek();
            boolean empty = true;
            request.append('(');
            int sep = MailAliases.emailSeparatorPos(text);
            if (sep == -1) {
                text = SearchRequestText.fullEscape(text, false);
                String[] synonyms = context.synonyms().synonyms(text);
                if (synonyms != null) {
                    text = StringUtils.join(synonyms, OR, "" + '(', "" + ')');
                }
                for (String field: scope) {
                    if (empty) {
                        empty = false;
                    } else {
                        request.append(OR);
                    }
                    request.append(field);
                    request.append(':');
                    request.append(text);
                }
            } else {
                Set<String> emails =
                    MailAliases.INSTANCE.equivalentEmails(text, sep);
                String normalized = SearchRequestText.fullEscape(
                    MailAliases.INSTANCE.normalizeEmail(text, sep),
                    false);
                String equivalentEmails;
                if (emails.size() > 1) {
                    StringBuilder sb = new StringBuilder();
                    for (String email: emails) {
                        if (sb.length() == 0) {
                            sb.append('(');
                        } else {
                            sb.append(OR);
                        }
                        sb.append(SearchRequestText.fullEscape(email, false));
                    }
                    sb.append(')');
                    equivalentEmails = new String(sb);
                } else {
                    equivalentEmails =
                        SearchRequestText.fullEscape(text, false);
                }
                for (String field: scope) {
                    if (empty) {
                        empty = false;
                    } else {
                        request.append(OR);
                    }
                    switch (field) {
                        case "body_text":
                        case "pure_body":
                        case "hdr_subject":
                            request.append(field);
                            request.append(':');
                            request.append(equivalentEmails);
                            break;
                        case "hdr_from":
                        case "hdr_to":
                        case "hdr_cc":
                        case "hdr_bcc":
                            request.append(field);
                            request.append("_normalized:");
                            request.append(normalized);
                            break;
                        case "hdr_from_normalized":
                        case "hdr_to_normalized":
                        case "hdr_cc_normalized":
                        case "hdr_bcc_normalized":
                            request.append(field);
                            request.append(':');
                            request.append(normalized);
                            break;
                        default:
                            request.append(field);
                            request.append(':');
                            request.append(text);
                            break;
                    }
                }
            }
            request.append(')');
        } else {
            request.append(context.selectAll());
        }
    }

    @Override
    public Void visit(final QueryToken query) throws ParseException {
        processQueryToken(query.text());
        return null;
    }

    private static String join(final List<QueryToken> tokens) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < tokens.size(); ++i) {
            if (i != 0) {
                sb.append(' ');
            }
            sb.append(tokens.get(i).text());
        }
        return new String(sb);
    }

    private abstract class SpecialQueryConstructor
        extends AbstractLuceneQueryConstructor
    {
        SpecialQueryConstructor(final LuceneQueryContext context) {
            super(context);
        }

        protected abstract void process(final String text)
            throws ParseException;

        @Override
        public Void visit(final FieldQuery query) throws ParseException {
            LuceneQueryConstructor.this.visit(query);
            return null;
        }

        @Override
        public Void visit(final QueryToken query) throws ParseException {
            process(query.text());
            return null;
        }

        @Override
        public Void visit(final QuotedQuery query) throws ParseException {
            process(join(query.tokens()));
            return null;
        }
    }

    private class FolderQueryConstructor extends SpecialQueryConstructor {
        FolderQueryConstructor() {
            super(LuceneQueryConstructor.this.context);
        }

        @Override
        protected void process(String text) throws ParseException {
            text = text.toLowerCase(Locale.ROOT);
            String folderType = null;
            switch (text) {
                case "входящие":
                    folderType = "inbox";
                    break;
                case "спам":
                    folderType = "spam";
                    break;
                case "удалённые":
                case "удаленные":
                    folderType = "trash";
                    break;
                case "отправленные":
                    folderType = "sent";
                    break;
                case "исходящие":
                    folderType = "outbox";
                    break;
                case "черновики":
                    folderType = "draft";
                    break;
                case "архив":
                    folderType = "archive";
                    break;
                case "шаблоны":
                    folderType = "template";
                    break;
                default:
                    break;
            }
            List<String> fids = folders.fids().get(text);
            if (fids == null && folderType == null) {
                throw new ParseException("Unknown folder: " + text, 0);
            }
            request.append('(');
            if (fids != null) {
                for (int i = 0; i < fids.size(); ++i) {
                    if (i != 0) {
                        request.append(OR);
                    }
                    request.append("fid:");
                    request.append(fids.get(i));
                }
                if (folderType != null) {
                    request.append(OR);
                }
            }
            if (folderType != null) {
                request.append("folder_type:");
                request.append(folderType);
            }
            request.append(')');
        }
    }

    private class LabelQueryConstructor extends SpecialQueryConstructor {
        LabelQueryConstructor() {
            super(LuceneQueryConstructor.this.context);
        }

        @Override
        protected void process(String text) throws ParseException {
            text = text.toLowerCase(Locale.ROOT);
            boolean processed = false;
            switch (text) {
                case "read":
                case "прочитанные":
                    processed = true;
                    request.append('(');
                    request.append(context.selectAll());
                    request.append(" AND NOT unread:1)");
                    break;
                case "unread":
                case "непрочитанные":
                    processed = true;
                    request.append("unread:1");
                    break;
                case "flagged":
                case "urgent":
                case "важные":
                case "важное":
                    processed = true;
                    request.append("lids:");
                    request.append(labels.importantLid());
                    break;
                default:
                    break;
            }
            if (!processed) {
                List<String> lids = labels.lids().get(text);
                if (lids == null) {
                    throw new ParseException("Unknown label: " + text, 0);
                }
                request.append('(');
                for (int i = 0; i < lids.size(); ++i) {
                    if (i != 0) {
                        request.append(OR);
                    }
                    request.append("lids:");
                    request.append(lids.get(i));
                }
                request.append(')');
            }
        }
    }

    private class DateEndQueryConstructor extends SpecialQueryConstructor {
        DateEndQueryConstructor() {
            super(LuceneQueryConstructor.this.context);
        }

        @Override
        protected void process(String text) throws ParseException {
            long timestamp = parseTimestamp(text);
            request.append("received_date:[0 TO ");
            request.append(timestamp + DateTimeConstants.SECONDS_PER_DAY - 1);
            request.append(']');
        }
    }

    private class DateBeforeQueryConstructor extends SpecialQueryConstructor {
        DateBeforeQueryConstructor() {
            super(LuceneQueryConstructor.this.context);
        }

        @Override
        protected void process(String text) throws ParseException {
            long timestamp = parseTimestamp(text);
            request.append("received_date:[0 TO ");
            request.append(timestamp);
            request.append(']');
        }
    }

    private class DateBeginQueryConstructor extends SpecialQueryConstructor {
        DateBeginQueryConstructor() {
            super(LuceneQueryConstructor.this.context);
        }

        @Override
        protected void process(String text) throws ParseException {
            long timestamp = parseTimestamp(text);
            request.append("received_date:[");
            request.append(timestamp);
            request.append(" TO 9999999999]");
        }
    }

    private class MonthQueryConstructor extends SpecialQueryConstructor {
        MonthQueryConstructor() {
            super(LuceneQueryConstructor.this.context);
        }

        @Override
        protected void process(String text) throws ParseException {
            Integer month = MONTHS.get(text.toLowerCase(Locale.ROOT));
            if (month == null) {
                throw new ParseException("Unknown month: " + text, 0);
            }
            DateTime now = DateTime.now(context.timezone());
            request.append("received_date:(");
            for (int year = now.year().get(); year >= TILL_YEAR; --year) {
                DateTime start = new DateTime(
                    year,
                    month,
                    1,
                    0,
                    0,
                    0,
                    0,
                    context.timezone());
                DateTime end = start.plusMonths(1);
                request.append('[');
                request.append(
                    start.getMillis() / DateTimeConstants.MILLIS_PER_SECOND);
                request.append(" TO ");
                request.append(
                    (end.getMillis() - 1L)
                    / DateTimeConstants.MILLIS_PER_SECOND);
                request.append(']');
                if (year != TILL_YEAR) {
                    request.append(OR);
                }
            }
            request.append(')');
        }
    }

    private class YearQueryConstructor extends SpecialQueryConstructor {
        YearQueryConstructor() {
            super(LuceneQueryConstructor.this.context);
        }

        @Override
        protected void process(String text) throws ParseException {
            DateTime start;
            try {
                start = new DateTime(
                    Integer.parseInt(text),
                    1,
                    1,
                    0,
                    0,
                    0,
                    0,
                    context.timezone());
            } catch (Throwable t) {
                ParseException e =
                    new ParseException("Failed to parse year: " + text, 0);
                e.initCause(t);
                throw e;
            }
            DateTime end = start.plusYears(1);
            request.append("received_date:[");
            request.append(
                start.getMillis() / DateTimeConstants.MILLIS_PER_SECOND);
            request.append(" TO ");
            request.append(
                (end.getMillis() - 1L)
                / DateTimeConstants.MILLIS_PER_SECOND);
            request.append(']');
        }
    }

    private class OlderThanQueryConstructor extends SpecialQueryConstructor {
        OlderThanQueryConstructor() {
            super(LuceneQueryConstructor.this.context);
        }

        @Override
        protected void process(String text) throws ParseException {
            long duration = parseDuration(text);
            request.append("received_date:[0 TO ");
            request.append(
                (System.currentTimeMillis() - duration)
                / DateTimeConstants.MILLIS_PER_SECOND);
            request.append(']');
        }
    }

    private class NewerThanQueryConstructor extends SpecialQueryConstructor {
        NewerThanQueryConstructor() {
            super(LuceneQueryConstructor.this.context);
        }

        @Override
        protected void process(String text) throws ParseException {
            long duration = parseDuration(text);
            request.append("received_date:[");
            request.append(
                (System.currentTimeMillis() - duration)
                / DateTimeConstants.MILLIS_PER_SECOND);
            request.append(" TO 9999999999]");
        }
    }

    private class HasAttachmentsQueryConstructor
        extends SpecialQueryConstructor
    {
        HasAttachmentsQueryConstructor() {
            super(LuceneQueryConstructor.this.context);
        }

        @Override
        protected void process(String text) throws ParseException {
            if (parseFlag(text)) {
                request.append("has_attachments:1");
            } else {
                request.append('(');
                request.append(context.selectAll());
                request.append(" AND NOT has_attachments:1)");
            }
        }
    }

    private class UnreadQueryConstructor extends SpecialQueryConstructor {
        UnreadQueryConstructor() {
            super(LuceneQueryConstructor.this.context);
        }

        @Override
        protected void process(String text) throws ParseException {
            if (parseFlag(text)) {
                request.append("unread:1");
            } else {
                request.append('(');
                request.append(context.selectAll());
                request.append(" AND NOT unread:1)");
            }
        }
    }

    private class FilterQueryConstructor extends SpecialQueryConstructor {
        FilterQueryConstructor() {
            super(LuceneQueryConstructor.this.context);
        }

        @Override
        protected void process(String text) throws ParseException {
            SearchFilter filter =
                context.filters().get(text.toLowerCase(Locale.ROOT));
            if (filter == null) {
                throw new ParseException("Unknown filter: " + text, 0);
            } else {
                filter.apply(request, labels);
            }
        }
    }

    private class SubscriptionEmailQueryConstructor
        extends SpecialQueryConstructor
    {
        SubscriptionEmailQueryConstructor() {
            super(LuceneQueryConstructor.this.context);
        }

        @Override
        protected void process(String text) throws ParseException {
            request.append("hdr_from_normalized:(");
            request.append(templatizeEmail(text));
            request.append(')');
        }
    }

    private class SharedQueryConstructor extends SpecialQueryConstructor {
        SharedQueryConstructor() {
            super(LuceneQueryConstructor.this.context);
        }

        @Override
        protected void process(String text) throws ParseException {
            if (parseFlag(text)) {
                request.append("lids:FAKE_SYNCED_LBL");
            } else {
                request.append('(');
                request.append(context.selectAll());
                request.append(" AND NOT lids:FAKE_SYNCED_LBL)");
            }
        }
    }
}

