package ru.yandex.msearch.proxy.api.async.mail.rules;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

import org.apache.http.HttpException;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.msearch.proxy.api.async.mail.SearchSession;
import ru.yandex.msearch.proxy.api.async.mail.SingleFieldSortOrder;
import ru.yandex.msearch.proxy.api.async.mail.SortOrder;

import ru.yandex.msearch.proxy.api.async.mail.documents.Document;
import ru.yandex.msearch.proxy.api.async.mail.documents.Documents;
import ru.yandex.msearch.proxy.api.async.mail.documents.DocumentsGroup;
import ru.yandex.msearch.proxy.api.async.mail.documents.UnorderedDocuments;

public class MailGroupsSearchRule implements SearchRule {
    private static final String MESSAGE_TYPE = "message_type";
    private static final String ONLY_ATTACHMENTS = "only_attachments";
    private static final String REQUEST = "request";
    private static final String SCOPE = "scope";

    private final SearchRule next;

    public MailGroupsSearchRule(final SearchRule next) {
        this.next = next;
    }

    @Override
    public void execute(final SearchSession session) throws HttpException {
        new MultiRequestCallback(next, session).sendRequests();
    }

    public static class MultiRequestCallback {
        private final SearchRule next;
        private final SearchSession session;
        private final SortOrder sort;
        private final AtomicBoolean done;
        private final AtomicReference<Documents> sendersResult;
        private final AtomicReference<Documents> addressResult;
        private final AtomicReference<Documents> subjectResult;
        private final AtomicReference<Documents> pureExactMatchResult;
        private final AtomicReference<Documents> purePeopleResult;
        private final AtomicReference<Documents> pureResult;
        private final AtomicReference<Documents> attachNameResult;
        private final AtomicReference<Documents> attachBodyResult;
        private final AtomicReference<Documents> quotationResult;
        private final List<AtomicReference<Documents>> results;

        public MultiRequestCallback(
            final SearchRule next,
            final SearchSession session)
            throws BadRequestException
        {
            this.next = next;
            this.session = session;
            this.sort = session.params().getEnum(
                SingleFieldSortOrder.class,
                "order",
                SingleFieldSortOrder.DEFAULT);
            this.done = new AtomicBoolean(false);
            List<AtomicReference<Documents>> results = new ArrayList<>();
            this.sendersResult = createResult(results);
            this.addressResult = createResult(results);
            this.subjectResult = createResult(results);
            this.pureExactMatchResult = createResult(results);
            this.purePeopleResult = createResult(results);
            this.pureResult = createResult(results);
            this.attachNameResult = createResult(results);
            this.attachBodyResult = createResult(results);
            this.quotationResult = createResult(results);
            this.results = Collections.unmodifiableList(results);
        }

        private static AtomicReference<Documents> createResult(
            final List<AtomicReference<Documents>> results)
        {
            AtomicReference<Documents> result = new AtomicReference<>();
            results.add(result);
            return result;
        }

        public SearchSession session() {
            return session;
        }

        public void sendRequests() throws HttpException {
            SearchSession clean = session.copy();
            clean.params().remove(SCOPE);
            clean.params().remove(MESSAGE_TYPE);
            clean.params().remove(ONLY_ATTACHMENTS);
            // senders
            SearchSession sendersSession = clean.copy();
            sendersSession.params().add(SCOPE, "hdr_from");
            sendRequest(sendersSession, "senders", sendersResult);
            // other address fields
            SearchSession addressSession = clean.copy();
            addressSession.params().add(
                SCOPE,
                "hdr_to,hdr_cc,hdr_bcc,hdr_reply_to");
            sendRequest(addressSession, "address", addressResult);
            // subject
            SearchSession subjectSession = clean.copy();
            subjectSession.params().add(SCOPE, "hdr_subject");
            sendRequest(subjectSession, "subject", subjectResult);
            // pure body, exact match
            SearchSession pureExactMatchSession = clean.copy();
            String request =
                pureExactMatchSession.params().getString(REQUEST, null);
            if (request != null) {
                pureExactMatchSession.params().replace(
                    REQUEST,
                    '"' + request.replace('"', ' ') + '"');
            }
            pureExactMatchSession.params().add(SCOPE, "pure_body");
            sendRequest(
                pureExactMatchSession,
                "body_exactmatch",
                pureExactMatchResult);
            // pure body, only people
            SearchSession purePeopleSession = clean.copy();
            purePeopleSession.params().add(SCOPE, "pure_body");
            purePeopleSession.params().add(MESSAGE_TYPE, "4");
            sendRequest(purePeopleSession, "body_people", purePeopleResult);
            // pure body, all
            SearchSession pureSession = clean.copy();
            pureSession.params().add(SCOPE, "pure_body");
            sendRequest(pureSession, "body", pureResult);
            // attachment name
            SearchSession attachNameSession = clean.copy();
            attachNameSession.params().add(SCOPE, "attachname");
            sendRequest(attachNameSession, "attach_name", attachNameResult);
            // attachment body
            SearchSession attachBodySession = clean.copy();
            attachBodySession.params().add(SCOPE, "body_text");
            attachBodySession.params().add(ONLY_ATTACHMENTS, "1");
            sendRequest(attachBodySession, "attach_text", attachBodyResult);
            // quotation
            SearchSession quotationSession = clean.copy();
            sendRequest(quotationSession, "citation", quotationResult);
        }

        public boolean isDone() {
            for (AtomicReference<Documents> result: results) {
                if (result.get() == null) {
                    return false;
                }
            }
            return true;
        }

        public void responseReceived() {
            if (!done.get() && isDone() && done.compareAndSet(false, true)) {
                Documents result = new UnorderedDocuments(sort.comparator());
                for (AtomicReference<Documents> ref: results) {
                    Documents newResult = ref.get();
                    for (DocumentsGroup group: newResult) {
                        String groupId = group.id();
                        for (Document doc: group) {
                            result.add(groupId, doc);
                        }
                    }
                }
                session.callback().completed(result);
            }
        }

        private void sendRequest(
            final SearchSession session,
            final String group,
            final AtomicReference<Documents> result)
            throws HttpException
        {
            SearchSession sessionWithCallback = session.withCallback(
                new SingleRequestCallback(
                    session,
                    new SingleRequestContext(this, group, result)));
            next.execute(sessionWithCallback);
        }
    }

    private static class SingleRequestContext {
        private final MultiRequestCallback callback;
        private final String group;
        private final AtomicReference<Documents> result;

        public SingleRequestContext(
            final MultiRequestCallback callback,
            final String group,
            final AtomicReference<Documents> result)
        {
            this.callback = callback;
            this.group = group;
            this.result = result;
        }

        public void completed(final Documents docs) {
            // put group
            result.set(docs);
            callback.responseReceived();
        }
    }

    private static class SingleRequestCallback
        extends AbstractSessionCallback<Documents>
    {
        private final SingleRequestContext context;

        public SingleRequestCallback(
            final SearchSession session,
            final SingleRequestContext context)
        {
            super(session);
            this.context = context;
        }

        @Override
        public void completed(final Documents docs) {
            synchronized (this) {
                if (done) {
                    return;
                }
                done = true;
            }
            context.completed(docs);
        }
    }
}

