package ru.yandex.msearch.proxy.api.async.mail.rules;

import java.io.IOException;

import java.nio.charset.CharacterCodingException;

import java.text.ParseException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.erratum.EmptyErratumResult;
import ru.yandex.erratum.ErratumClient;
import ru.yandex.erratum.ErratumResult;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.proxy.api.async.mail.SearchSession;
import ru.yandex.msearch.proxy.api.async.mail.documents.Documents;
import ru.yandex.msearch.proxy.api.mail.MailSearchOptions;

import ru.yandex.parser.query.EmptyQueryVisitor;
import ru.yandex.parser.query.QueryParser;
import ru.yandex.parser.query.QueryToken;

import ru.yandex.parser.uri.CgiParams;

import ru.yandex.util.string.StringUtils;

public class MisspellRule implements SearchRule {
    private final SearchRule next;
    private final ErratumClient client;

    public MisspellRule(final SearchRule next, final ErratumClient client) {
        this.next = next;
        this.client = client;
    }

    private void erratum(
        final SearchSession session,
        final MailSearchOptions options,
        final MultiFutureCallback<FieldErratumResult> callback)
        throws CharacterCodingException
    {
        String request = options.request().trim();
        if (!request.isEmpty() && request.indexOf('@') == -1) {
            session.httpSession().subscribeForCancellation(
                client.execute(
                    request,
                    session.httpSession().requestsListener()
                        .createContextGeneratorFor(client),
                    new ErrorSuppressingFutureCallback<>(
                        new FieldErratumCallback(
                            options,
                            callback.newCallback()),
                        EmptyErratumResult.INSTANCE)));
        }
        for (List<MailSearchOptions> requests: options.requests().values()) {
            for (MailSearchOptions suboptions: requests) {
                erratum(session, suboptions, callback);
            }
        }
    }

    private List<QueryToken> queryLanguageTokens(final SearchSession session)
        throws HttpException
    {
        if (session.requestInfo().options().requests().isEmpty()
            && session.params().getBoolean(
                "query-language",
                session.httpSession().server().config().queryLanguage()))
        {
            try {
                final List<QueryToken> tokens = new ArrayList<>();
                new QueryParser(session.requestInfo().options().request())
                    .parse()
                    .accept(
                        new EmptyQueryVisitor<RuntimeException>() {
                            @Override
                            public Void visit(final QueryToken token) {
                                if (token.text().indexOf('@') == -1) {
                                    tokens.add(token);
                                }
                                return null;
                            }
                        });
                return tokens;
            } catch (IOException | ParseException e) {
                // Not a valid query
            }
        }
        return null;
    }

    @Override
    public void execute(final SearchSession session) throws HttpException {
        CgiParams params = session.params();
        if (params.getBoolean("imap", false)
            || params.getBoolean("force", false))
        {
            next.execute(session);
        } else {
            List<QueryToken> tokens = queryLanguageTokens(session);
            try {
                if (tokens == null) {
                    Step1Callback callback = new Step1Callback(session);
                    next.execute(session.withCallback(callback));
                    MultiFutureCallback<FieldErratumResult> erratumCallback =
                        new MultiFutureCallback<>(
                            new ErratumCallback(callback));
                    erratum(
                        session,
                        session.requestInfo().options(),
                        erratumCallback);
                    erratumCallback.done();
                } else if (tokens.isEmpty()) {
                    next.execute(session);
                } else {
                    DoubleFutureCallback<Documents, FixedDocuments> callback =
                        new DoubleFutureCallback<>(
                            new QueryLanguageCallback(session));
                    next.execute(session.withCallback(callback.first()));
                    MultiFutureCallback<SubstringErratumResult>
                    erratumCallback =
                        new MultiFutureCallback<>(
                            new QueryLanguageErratumCallback(
                                session,
                                next,
                                callback.second()));
                    for (QueryToken token: tokens) {
                        session.httpSession().subscribeForCancellation(
                            client.execute(
                                token.text(),
                                session.httpSession().requestsListener()
                                    .createContextGeneratorFor(client),
                                new ErrorSuppressingFutureCallback<>(
                                    new SubstringErratumCallback(
                                        erratumCallback.newCallback(),
                                        token.pos(),
                                        token.length()),
                                    EmptyErratumResult.INSTANCE)));
                    }
                    erratumCallback.done();
                }
            } catch (CharacterCodingException e) {
                throw new BadRequestException(e);
            }
        }
    }

    private class Step1Callback extends AbstractSessionCallback<Documents> {
        private final PrefixedLogger logger;
        private List<FieldErratumResult> erratumResult = null;
        private Documents documents = null;

        public Step1Callback(final SearchSession session) {
            super(session);
            logger = session.httpSession().logger().addPrefix("erratum");
        }

        public SearchSession session() {
            return session;
        }

        public PrefixedLogger logger() {
            return logger;
        }

        public List<FieldErratumResult> erratumResult() {
            return erratumResult;
        }

        public Documents documents() {
            return documents;
        }

        private void checkCompleted() {
            if (done || erratumResult == null || documents == null) {
                return;
            }
            done = true;
            if (erratumResult.isEmpty()) {
                session.callback().completed(documents);
            } else {
                SearchSession session = this.session.copy();
                CgiParams params = session.params();
                for (FieldErratumResult result: erratumResult) {
                    params.add(
                        result.options().field(),
                        result.erratumResult().text());
                }
                try {
                    next.execute(
                        session.withCallback(new Step2Callback(this)));
                } catch (HttpException e) {
                    logger.log(Level.WARNING, "Failed to fix request", e);
                    session.callback().completed(documents);
                }
            }
        }

        public synchronized void erratumResult(
            final List<FieldErratumResult> erratumResult)
        {
            this.erratumResult = erratumResult;
            logger.info("Erratum result: " + erratumResult);
            checkCompleted();
        }

        @Override
        public synchronized void completed(final Documents documents) {
            this.documents = documents;
            logger.info("Search result received");
            checkCompleted();
        }
    }

    private static class Step2Callback
        extends AbstractSessionCallback<Documents>
    {
        private final Step1Callback step1;

        public Step2Callback(final Step1Callback step1) {
            super(step1.session());
            this.step1 = step1;
        }

        @Override
        public synchronized void completed(final Documents documents) {
            if (!done) {
                done = true;
                int count = step1.documents().size();
                int fixedCount = documents.size();
                step1.logger().fine(
                    "Original request results count: " + count
                    + ", fixed count: " + fixedCount);
                if (step1.documents().equals(documents)) {
                    step1.logger().info("Keep original request unchanged");
                    session.callback().completed(step1.documents());
                } else if (count == 0) {
                    step1.logger().info("Switching to fixed results");
                    for (FieldErratumResult result: step1.erratumResult()) {
                        result.options().request(
                            result.erratumResult().text(),
                            result.erratumResult().rule());
                    }
                    session.callback().completed(documents);
                } else {
                    step1.logger().info("Adding misspell suggest");
                    for (FieldErratumResult result: step1.erratumResult()) {
                        result.options().suggest(
                            result.erratumResult().text(),
                            result.erratumResult().rule());
                    }
                    session.callback().completed(step1.documents());
                }
            }
        }
    }

    private static class ErratumCallback
        implements FutureCallback<List<FieldErratumResult>>
    {
        private final Step1Callback callback;

        public ErratumCallback(final Step1Callback callback) {
            this.callback = callback;
        }

        @Override
        public void completed(final List<FieldErratumResult> erratumResult) {
            List<FieldErratumResult> correctedResults =
                new ArrayList<>(erratumResult);
            correctedResults.removeIf(
                r -> r.erratumResult().code() != ErratumResult.CODE_CORRECTED);
            callback.erratumResult(correctedResults);
        }

        @Override
        public void failed(final Exception e) {
            callback.logger().log(Level.WARNING, "Request failed", e);
            callback.erratumResult(Collections.emptyList());
        }

        @Override
        public void cancelled() {
            callback.logger().warning("Request cancelled");
            callback.erratumResult(Collections.emptyList());
        }
    }

    private static class FieldErratumResult {
        private final MailSearchOptions options;
        private final ErratumResult erratumResult;

        public FieldErratumResult(
            final MailSearchOptions options,
            final ErratumResult erratumResult)
        {
            this.options = options;
            this.erratumResult = erratumResult;
        }

        public MailSearchOptions options() {
            return options;
        }

        public ErratumResult erratumResult() {
            return erratumResult;
        }

        @Override
        public String toString() {
            return options.field() + ':' + erratumResult;
        }
    }

    private static class FieldErratumCallback
        implements FutureCallback<ErratumResult>
    {
        private final MailSearchOptions options;
        private final FutureCallback<? super FieldErratumResult> callback;

        public FieldErratumCallback(
            final MailSearchOptions options,
            final FutureCallback<? super FieldErratumResult> callback)
        {
            this.options = options;
            this.callback = callback;
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }

        @Override
        public void completed(final ErratumResult erratumResult) {
            callback.completed(new FieldErratumResult(options, erratumResult));
        }

        @Override
        public void failed(final Exception e) {
            callback.failed(e);
        }
    }

    private static class SubstringErratumResult {
        private final int pos;
        private final int length;
        private final ErratumResult erratumResult;

        public SubstringErratumResult(
            final int pos,
            final int length,
            final ErratumResult erratumResult)
        {
            this.pos = pos;
            this.length = length;
            this.erratumResult = erratumResult;
        }

        public int pos() {
            return pos;
        }

        public int length() {
            return length;
        }

        public ErratumResult erratumResult() {
            return erratumResult;
        }
    }

    private static class SubstringErratumCallback
        extends AbstractFilterFutureCallback<
            ErratumResult,
            SubstringErratumResult>
    {
        private final int pos;
        private final int length;

        public SubstringErratumCallback(
            final FutureCallback<SubstringErratumResult> callback,
            final int pos,
            final int length)
        {
            super(callback);
            this.pos = pos;
            this.length = length;
        }

        @Override
        public void completed(final ErratumResult result) {
            callback.completed(
                new SubstringErratumResult(pos, length, result));
        }
    }

    private class QueryLanguageErratumCallback
        extends AbstractSessionCallback<List<SubstringErratumResult>>
    {
        private final SearchRule next;
        private final FutureCallback<FixedDocuments> callback;

        public QueryLanguageErratumCallback(
            final SearchSession session,
            final SearchRule next,
            final FutureCallback<FixedDocuments> callback)
        {
            super(session);
            this.next = next;
            this.callback = callback;
        }

        private int wordCount(final CharSequence text) {
            return wordCount(text, 0, text.length());
        }

        private int wordCount(
            final CharSequence text,
            final int offset,
            final int len)
        {
            int count = 0;
            for (int i = offset; i < len; i++) {
                if (Character.isWhitespace(text.charAt(i))) {
                    count += 1;
                }
            }

            return count;
        }

        @Override
        public void completed(final List<SubstringErratumResult> result) {
            StringBuilder request =
                new StringBuilder(session.requestInfo().options().request());
            String rule = null;
            for (int i = result.size(); i-- > 0;) {
                SubstringErratumResult replacement = result.get(i);
                ErratumResult erratumResult = replacement.erratumResult();
                if (erratumResult.code() == ErratumResult.CODE_CORRECTED) {
                    if (replacement.pos() == 0
                        && replacement.length() == request.length())
                    {
                        // hook if we have simple request
                        request = new StringBuilder(erratumResult.text());
                    } else {
                        int sourceWC = wordCount(
                            request,
                            replacement.pos(),
                            replacement.length());

                        String replacementText = erratumResult.text();
                        if (sourceWC < wordCount(erratumResult.text())) {
                            replacementText =
                                StringUtils.concat(
                                    "(",
                                    erratumResult.text(),
                                    ")");
                        }

                        request.replace(
                            replacement.pos,
                            replacement.pos() + replacement.length(),
                            replacementText);
                    }

                    rule = erratumResult.rule();
                }
            }

            if (rule == null) {
                callback.completed(null);
            } else {
                SearchSession session = this.session.copy();
                String fixedRequest = new String(request);
                session.params().replace("request", fixedRequest);
                try {
                    next.execute(
                        session.withCallback(
                            new FixedCallback(callback, fixedRequest, rule)));
                } catch (HttpException e) {
                    session.httpSession().logger().addPrefix("erratum")
                        .log(Level.WARNING, "Failed to fix request", e);
                    callback.completed(null);
                }
            }
        }
    }

    private static class FixedCallback
        extends AbstractFilterFutureCallback<Documents, FixedDocuments>
    {
        private final String request;
        private final String rule;

        public FixedCallback(
            final FutureCallback<FixedDocuments> callback,
            final String request,
            final String rule)
        {
            super(callback);
            this.request = request;
            this.rule = rule;
        }

        @Override
        public void completed(final Documents documents) {
            callback.completed(new FixedDocuments(documents, request, rule));
        }
    }

    private static class FixedDocuments {
        private final Documents documents;
        private final String request;
        private final String rule;

        public FixedDocuments(
            final Documents documents,
            final String request,
            final String rule)
        {
            this.documents = documents;
            this.request = request;
            this.rule = rule;
        }

        public Documents documents() {
            return documents;
        }

        public String request() {
            return request;
        }

        public String rule() {
            return rule;
        }
    }

    private static class QueryLanguageCallback
        extends AbstractSessionCallback<Map.Entry<Documents, FixedDocuments>>
    {
        public QueryLanguageCallback(final SearchSession session) {
            super(session);
        }

        @Override
        public void completed(
            final Map.Entry<Documents, FixedDocuments> result)
        {
            PrefixedLogger logger =
                session.httpSession().logger().addPrefix("erratum");
            Documents documents = result.getKey();
            FixedDocuments fixed = result.getValue();
            if (fixed == null) {
                logger.info("No misspell suggestions");
                session.callback().completed(documents);
            } else {
                int count = documents.size();
                Documents fixedDocuments = fixed.documents();
                int fixedCount = fixedDocuments.size();
                if (fixedCount == 0) {
                    logger.info("No results on fixed request");
                    session.callback().completed(documents);
                } else if (count == 0) {
                    logger.info("Switching to fixed results");
                    session.requestInfo().options().request(
                        fixed.request(),
                        fixed.rule());
                    session.callback().completed(fixedDocuments);
                } else {
                    logger.info("Adding misspell suggest");
                    session.requestInfo().options().suggest(
                        fixed.request(),
                        fixed.rule());
                    session.callback().completed(documents);
                }
            }
        }
    }
}

