package ru.yandex.msearch.proxy.api.async.mail.rules;

import java.util.ArrayList;

import java.util.Arrays;
import java.util.List;

import java.util.logging.Level;

import org.apache.http.HttpException;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.json.parser.JsonException;

import ru.yandex.msearch.proxy.AsyncHttpServer;

import ru.yandex.msearch.proxy.api.async.ProxyParams;

import ru.yandex.msearch.proxy.api.async.mail.SearchSession;
import ru.yandex.msearch.proxy.api.async.mail.documents.Document;
import ru.yandex.msearch.proxy.api.async.mail.documents.Documents;
import ru.yandex.msearch.proxy.api.async.mail.documents.DocumentsGroup;

import ru.yandex.msearch.proxy.api.async.mail.relevance.PersonalFactorsProcesor;
import ru.yandex.msearch.proxy.api.async.mail.relevance.PersonalFactorsProcessorFactory;

import ru.yandex.msearch.proxy.api.async.mail.searcher.PlainSearcher;
import ru.yandex.msearch.proxy.api.async.mail.searcher
    .ProducerSequentialSearcher;

import ru.yandex.parser.searchmap.User;

import ru.yandex.parser.uri.CgiParams;

import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.PrefixType;

import ru.yandex.search.result.SearchResult;

public class PersonalFactorsGatherRule implements SearchRule {
    private static final String LOG_PREFIX = "personal-factors";

    private final SearchRule next;
    private final AsyncHttpServer server;
    private final List<String> scorerFields;
    private final boolean ruleEnabled;
    private final List<PersonalFactorsProcessorFactory> factories;

    public PersonalFactorsGatherRule(
        final SearchRule next,
        final AsyncHttpServer server,
        final PersonalFactorsProcessorFactory... factories)
    {
        this.next = next;
        this.server = server;

        this.ruleEnabled =
            (server.searchClient() != null)
                && (server.config().tskvLogConfig() != null)
                && server.config().tskvLogConfig().personal();

        this.scorerFields =
            new ArrayList<>(server.relevance().textFields());
        this.factories = Arrays.asList(factories);
    }

    protected boolean enabled(
        final CgiParams params)
        throws BadRequestException
    {
        return ruleEnabled
            && !params.getBoolean("imap", false)
            && !params.containsKey("scope");
    }

    private User buildUser(final CgiParams params) throws BadRequestException {
        String mdb = params.getString(ProxyParams.MDB);
        PrefixType prefixType = server.searchMap().prefixType(mdb);
        Prefix prefix;
        if (mdb.equals("pg")) {
            prefix = params.get(ProxyParams.UID, prefixType);
        } else {
            prefix = params.get(ProxyParams.SUID, prefixType);
        }

        return new User(server.resolveService(mdb, prefix), prefix);
    }

    protected void adjustParams(final CgiParams params) {
        if (!scorerFields.isEmpty() && !params.containsKey("scorer")) {
            params.add("scorer", "perfield");
            params.put("scorer_fields", scorerFields);
        }
    }


    @Override
    public void execute(final SearchSession session) throws HttpException {
        CgiParams params = session.params();

        if (!enabled(params)) {
            session.httpSession().logger().info(
                "Personal factors gather disabled");

            next.execute(session);
            return;
        }

        User user = buildUser(params);

        PlainSearcher<SearchResult> searcher =
            new ProducerSequentialSearcher(
                server,
                session,
                server.searchClient(),
                null,
                user);

        List<PersonalFactorsProcesor> procesors
            = new ArrayList<>(factories.size());

        for (PersonalFactorsProcessorFactory factory: factories) {
            PersonalFactorsProcesor procesor
                = factory.create(session, user, searcher);
            procesor.execute();
            procesors.add(procesor);
        }

        adjustParams(params);

        next.execute(
            session.withCallback(
                new PersonalFactorsSession(session, procesors)));
    }

    private static final class PersonalFactorsSession
        extends AbstractSessionCallback<Documents>
    {
        private final List<PersonalFactorsProcesor> procesors;

        public PersonalFactorsSession(
            final SearchSession session,
            final List<PersonalFactorsProcesor> procesors)
        {
            super(session);
            this.procesors = procesors;
        }

        private void updateDocs(final Documents documents)
            throws JsonException
        {
            for (DocumentsGroup dg: documents) {
                for (Document doc: dg) {
                    for (PersonalFactorsProcesor procesor: procesors) {
                        procesor.accept(doc);
                    }
                }
            }
        }

        @Override
        public void completed(final Documents documents) {
            try {
                updateDocs(documents);
            } catch (JsonException je) {
                session.httpSession().logger()
                    .log(Level.WARNING, "Failed to parse doc mtype", je);
            } finally {
                session.callback().completed(documents);
            }
        }
    }
}
