package ru.yandex.msearch.proxy.api.async.mail.rules;

import java.util.logging.Level;

import org.apache.http.Header;
import org.apache.http.HttpException;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.proxy.api.async.mail.MailSearchHandler;
import ru.yandex.msearch.proxy.api.async.mail.RequestInfo;
import ru.yandex.msearch.proxy.api.async.mail.SearchSession;

import ru.yandex.msearch.proxy.api.async.mail.documents.Documents;
import ru.yandex.msearch.proxy.api.async.mail.relevance.search.LoggingSession;
import ru.yandex.msearch.proxy.api.async.mail.relevance.search.RankingSession;

import ru.yandex.msearch.proxy.api.async.mail.relevance.search.MailSearchRelevanceFactory;
import ru.yandex.msearch.proxy.api.async.mail.relevance.search.MailSearchSorterFactory;

import ru.yandex.msearch.proxy.api.mail.MailSearchOptions;

import ru.yandex.msearch.proxy.experiment.UserSplit;
import ru.yandex.msearch.proxy.logger.ProxyTskvLogger;

import ru.yandex.parser.uri.CgiParams;

public class RankingRule implements SearchRule {
    public static final String LOG_PREFIX = "ranking";
    private final SearchRule next;
    private final MailSearchRelevanceFactory relevanceFactory;

    public RankingRule(
        final SearchRule next,
        final MailSearchRelevanceFactory relevanceFactory)
    {
        this.relevanceFactory = relevanceFactory;
        this.next = next;
    }

    @Override
    public void execute(final SearchSession session) throws HttpException {
        PrefixedLogger logger =
            session.httpSession().logger().addPrefix(LOG_PREFIX);

        boolean skipping = false;
        // first check that ranking or logging enabled
        try {
            if (!isRequestLoggable(session, logger)) {
                skipping = true;
                logger.info("Request is not loggable, skipping");
            }
        } catch (BadRequestException bre) {
            skipping = true;
            logger.log(Level.WARNING, "Failed process request params", bre);
        }

        if (skipping) {
            this.next.execute(session);
            return;
        }

        RankingSession rankingSession;
        UserSplit userSplit = session.requestInfo().options().userSplit();
        if (userSplit != null) {
            MailSearchSorterFactory sorterFactory =
                relevanceFactory.create(
                    session,
                    session.requestInfo().options().userSplit());
            logger.info(
                "User experiments got, relevance model "
                    + sorterFactory.name());

            rankingSession = new LoggingSession(
                session,
                sorterFactory.build(session));
            this.next.execute(session.withCallback(rankingSession));
        } else {
            //waiting for usersplit
            this.next.execute(
                session.withCallback(
                    new RankingCallback(session)));
        }
    }

    private final class RankingCallback
        extends AbstractSessionCallback<Documents>
    {
        public RankingCallback(final SearchSession session) {
            super(session);
        }

        @Override
        public void completed(final Documents documents) {
            UserSplit userSplit =
                session.requestInfo().options().userSplit();

            MailSearchSorterFactory sorterFactory =
                relevanceFactory.create(session, userSplit);
            if (sorterFactory != null) {
                session.httpSession().logger().info(
                    "Using relevance model " + sorterFactory.name());

                new LoggingSession(
                    session,
                    sorterFactory.build(session)).completed(documents);
            } else {
                session.httpSession().logger().info(
                    "No relevance got for request");
                session.callback().completed(documents);
            }
        }
    }

    protected boolean isRequestLoggable(
        final SearchSession session,
        final PrefixedLogger logger)
        throws BadRequestException
    {
        ProxyTskvLogger tskvLogger = relevanceFactory.factorsLogger();

        if (tskvLogger == null) {
            return false;
        }

        CgiParams params = session.params();
        String xRequestId = extractRequestId(session.requestInfo());
        String reqId = params.getOrNull("reqid");
        MailSearchOptions options = session.requestInfo().options();
        String request = options.request();

        if (request == null) {
            logger.info("factors logging disabled: request null");
            return false;
        }

        if (xRequestId == null && reqId == null) {
            logger.info("factors logging disabled: xrequest and reqid null");
            return false;
        }

        if (params.getBoolean("imap", false)
            || params.containsKey("scope")
            || params.containsKey(MailSearchHandler.SEARCH_FILTER)
            || params.containsKey("message_type"))
        {
            return false;
        }

        return true;
    }

    private String extractRequestId(final RequestInfo requestInfo) {
        Header header = requestInfo.request()
            .getLastHeader(YandexHeaders.X_REQUEST_ID);
        if (header == null) {
            return null;
        }

        return header.getValue();
    }
}
