package ru.yandex.msearch.proxy.api.async.mail.rules;

import java.io.IOException;
import java.net.URISyntaxException;
import java.text.ParseException;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.UnaryOperator;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.message.BasicHeader;
import org.joda.time.Chronology;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.chrono.ISOChronology;

import ru.yandex.client.wmi.Folders;
import ru.yandex.client.wmi.FoldersConsumerFactory;
import ru.yandex.client.wmi.Labels;
import ru.yandex.client.wmi.LabelsConsumerFactory;
import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.ProxyParams;
import ru.yandex.msearch.proxy.api.async.mail.MailSearchHandler;
import ru.yandex.msearch.proxy.api.async.mail.SearchRequest;
import ru.yandex.msearch.proxy.api.async.mail.SearchSession;
import ru.yandex.msearch.proxy.api.mail.rules.MailSearchRule;
import ru.yandex.msearch.proxy.mail.SearchFilter;
import ru.yandex.parser.email.MailAliases;
import ru.yandex.parser.query.QueryAtom;
import ru.yandex.parser.query.QueryParser;
import ru.yandex.parser.query.QueryPrintingVisitor;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.PrefixType;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.util.string.StringUtils;

public class RewriteRequestRule implements SearchRule {
    private static final CollectionParser<String, List<String>, Exception>
        SCOPE_PARSER = new CollectionParser<>(
            NonEmptyValidator.INSTANCE,
            ArrayList::new);

    public static final List<String> REQUEST_TEXT_PARAMS = Arrays.asList(
        "body_text",
        "hdr_subject",
        "hdr_from",
        "hdr_to",
        "hdr_cc",
        "hdr_bcc",
        "hdr_to_cc_bcc");

    private static final List<String> PURE_FIELDS = Arrays.asList(
        "pure_body",
        "attachname",
        "attachtype",
        "attachments");

    private static final List<String> ALL_BODY_FIELDS;
    static {
        ALL_BODY_FIELDS = new ArrayList<>(PURE_FIELDS);
        ALL_BODY_FIELDS.addAll(MailSearchRule.EXTENDED_SCOPE);
    }

    private static final String AND = " AND (";
    public static final String ATTACH =
        "(disposition_type:attachment OR attachsize_b:*)";

    private final AsyncHttpServer server;
    private final SearchRule next;

    public RewriteRequestRule(
        final AsyncHttpServer server,
        final SearchRule next)
    {
        this.server = server;
        this.next = next;
    }

    public AsyncHttpServer server() {
        return server;
    }

    public SearchRule next() {
        return next;
    }

    private static void and(final StringBuilder request) {
        if (request.length() != 0) {
            request.append(" AND ");
        }
    }

    private static void or(
        final StringBuilder request,
        final boolean needDelimiter)
    {
        if (needDelimiter) {
            request.append(" OR ");
        }
    }

    private static void addCondition(
        final StringBuilder request,
        final String condition)
    {
        and(request);
        request.append(condition);
    }

    public static List<String> scope(
        final CgiParams params,
        final List<String> defaultScope)
        throws BadRequestException
    {
        return params.getAll("scope", defaultScope, SCOPE_PARSER);
    }

    // CSOFF: ParameterNumber
    public static void checkFlag(
        final StringBuilder request,
        final CgiParams params,
        final String flagName,
        final String condition)
        throws BadRequestException
    {
        if (params.getBoolean(flagName, false)) {
            addCondition(request, condition);
        }
    }
    // CSON: ParameterNumber

    private static void filterAll(
        final StringBuilder request,
        final CgiParams params,
        final String name)
    {
        filterAll(request, params, name, name);
    }

    // CSOFF: ParameterNumber
    private static void filterAll(
        final StringBuilder request,
        final CgiParams params,
        final String paramName,
        final String fieldName)
    {
        List<String> values = params.getAll(paramName);
        int size = values.size();
        if (size != 0) {
            and(request);
            request.append(fieldName);
            request.append(':');
            request.append('(');
            for (int i = 0; i < size; ++i) {
                or(request, i != 0);
                request.append(values.get(i));
            }
            request.append(')');
        }
    }
    // CSON: ParameterNumber

    // CSOFF: ReturnCount
    protected static List<String> scopeFor(
        final String param,
        final CgiParams params)
        throws BadRequestException
    {
        switch (param) {
            case "body_text":
                return ALL_BODY_FIELDS;
            case "pure_body":
                return PURE_FIELDS;
            case "hdr_from":
                return Arrays.asList(
                    "hdr_from",
                    "hdr_from_normalized",
                    "reply_to",
                    "reply_to_normalized");
            case "hdr_to":
            case "hdr_cc":
            case "hdr_bcc":
                return Arrays.asList(
                    param,
                    param + MailIndexFields.NORMALIZED);
            case "hdr_to_cc_bcc":
                return Arrays.asList(
                    "hdr_to",
                    "hdr_to_normalized",
                    "hdr_cc_normalized",
                    "hdr_bcc_normalized");
            default:
                return Collections.singletonList(param);
        }
    }
    // CSON: ReturnCount

    public static boolean applyUnixtimeTimerangeFilter(
        final StringBuilder request,
        final LuceneQueryContext context,
        final CgiParams params,
        final String fromParamName,
        final String toParamName)
        throws BadRequestException
    {
        long from = params.get(fromParamName, 0, NonNegativeIntegerValidator.INSTANCE);
        long to = params.get(
            toParamName,
            Integer.MAX_VALUE,
            NonNegativeIntegerValidator.INSTANCE);
        if (from > to || from < 0) {
            throw new BadRequestException("Invalid date " + from + " " + to);
        }

        if (from > 0 || to != Integer.MAX_VALUE) {
            and(request);
            request.append("received_date:[");
            request.append(from);
            request.append(" TO ");
            request.append(to);
            request.append(']');
            return true;
        }

        return false;
    }

    public static void applyTimerangeFilter(
        final StringBuilder request,
        final LuceneQueryContext context,
        final CgiParams params)
        throws BadRequestException
    {
        int from = params.get("from", 0, NonNegativeIntegerValidator.INSTANCE);
        int to = params.get(
            "to",
            Integer.MAX_VALUE,
            NonNegativeIntegerValidator.INSTANCE);
        if ((from > 0 || to != Integer.MAX_VALUE) && to >= from) {
            Chronology chronology =
                ISOChronology.getInstance(context.timezone());
            long fromEpoch;
            if (from == 0) {
                fromEpoch = 0L;
            } else {
                fromEpoch = new DateTime(
                    from / 10000,
                    (from / 100) % 100,
                    from % 100,
                    0,
                    0,
                    chronology).getMillis() / 1000L;
            }
            long toEpoch;
            if (to == Integer.MAX_VALUE) {
                toEpoch = Long.MAX_VALUE;
            } else {
                toEpoch = new DateTime(
                    to / 10000,
                    (to / 100) % 100,
                    to % 100,
                    23,
                    59,
                    59,
                    chronology).getMillis() / 1000L;
            }
            and(request);
            request.append("received_date:[");
            request.append(fromEpoch);
            request.append(" TO ");
            request.append(toEpoch);
            request.append(']');
        }
    }

    public static void applyScopes(
        final StringBuilder request,
        final String requestParamName,
        final LuceneQueryContext context,
        final CgiParams params)
        throws BadRequestException
    {
        List<ScopeRewrite> scopeRewrites = new ArrayList<>();
        scopeRewrites.add(
            new ScopeRewrite(requestParamName, context.defaultScope()));
        for (String name: REQUEST_TEXT_PARAMS) {
            scopeRewrites.add(
                new ScopeRewrite(name, scopeFor(name, params)));
        }
        for (ScopeRewrite rewrite: scopeRewrites) {
            List<SearchRequestText> texts = params.getAll(
                rewrite.paramName(),
                Collections.emptyList(),
                x -> new SearchRequestText(x, EMAIL_ALIASER),
                new ArrayList<>());
            texts.removeIf(x -> x.isEmpty());
            int size = texts.size();
            if (size != 0) {
                Collection<String> scope = rewrite.scope();
                and(request);
                request.append('(');
                for (int i = 0; i < size; ++i) {
                    or(request, i != 0);
                    request.append('(');
                    SearchRequestText text = texts.get(i);
                    if (text.hasWords()) {
                        text.fieldsQuery(request, scope);
                    } else {
                        request.append(context.selectAll());
                    }
                    text.negationsQuery(request, scope);
                    request.append(')');
                }
                request.append(')');
            }
        }
    }

    public static void applyScopesOrQueryLanguage(
        final StringBuilder request,
        final String requestText,
        final LuceneQueryContext context,
        final ProxySession session,
        final AsyncHttpServer server,
        final Labels labels,
        final Folders folders)
        throws BadRequestException
    {
        applyScopesOrQueryLanguage(request, requestText, context, session, session.params(), server, labels, folders);
    }

    public static void applyScopesOrQueryLanguage(
        final StringBuilder request,
        final String requestText,
        final LuceneQueryContext context,
        final ProxySession session,
        final CgiParams params,
        final AsyncHttpServer server,
        final Labels labels,
        final Folders folders)
        throws BadRequestException
    {
        String queryParserRequest = null;
        if (!new SearchRequestText(requestText).isEmpty()) {
            try {
                LuceneQueryConstructor query =
                    new LuceneQueryConstructor(context, labels, folders);
                QueryAtom queryAtom = new QueryParser(requestText).parse();
                queryAtom.accept(query);
                server.queryLanguageHit(true);
                server
                    .queryLanguageSimpleRequest(context.trivial());
                if (!context.trivial()) {
                    StringBuilder sb = new StringBuilder(
                        "QueryParser parsed non-trivial query:\n");
                    queryAtom.accept(new QueryPrintingVisitor(sb));
                    session.logger().info(new String(sb));
                }
                queryParserRequest = query.request();
            } catch (IOException | ParseException e) {
                session.logger().log(
                    Level.WARNING,
                    "QueryParser failed on string '" + requestText + '\'',
                    e);
                server.queryLanguageHit(false);
            }
        }
        session.logger()
            .info("QueryParser request: " + queryParserRequest);

        boolean extendedSearch = params.getBoolean("adv_search", false);
        boolean queryLanguage =
            queryParserRequest != null
                && !extendedSearch
                && params.getBoolean(
                "query-language",
                server.config().queryLanguage());
        if (queryLanguage) {
            and(request);
            request.append(queryParserRequest);
        } else {
            applyScopes(request, "request", context, params);
        }
        if (request.length() == 0) {
            request.append(context.selectAll());
        }
    }

    @Override
    public void execute(final SearchSession session)
        throws BadRequestException
    {
        CgiParams params = session.params();
        String mdb = params.getString(ProxyParams.MDB);
        boolean pg = mdb.equals("pg");
        String paramName;
        if (pg) {
            paramName = ProxyParams.UID;
        } else {
            paramName = ProxyParams.SUID;
        }
        PrefixType prefixType = server.searchMap().prefixType(mdb);
        Prefix prefix = params.get(paramName, prefixType);
        boolean corp = SearchRequest.corp(prefix);
        ImmutableURIConfig labelsConfig;
        ImmutableURIConfig foldersConfig;
        if (corp) {
            labelsConfig = server.config().corpLabelsConfig();
            foldersConfig = server.config().corpFoldersConfig();
        } else {
            labelsConfig = server.config().labelsConfig();
            foldersConfig = server.config().foldersConfig();
        }

        QueryConstructor query = new QueryConstructor(
            new StringBuilder(labelsConfig.uri().toASCIIString())
                .append(labelsConfig.firstCgiSeparator())
                .append(ProxyParams.WMI_SUFFIX));
        query.append(ProxyParams.MDB, mdb);
        query.append(paramName, prefix.toString());
        String labelsRequest = query.toString();

        query = new QueryConstructor(
            new StringBuilder(foldersConfig.uri().toASCIIString())
                .append(foldersConfig.firstCgiSeparator())
                .append(ProxyParams.WMI_SUFFIX));
        query.append(ProxyParams.MDB, mdb);
        query.append(paramName, prefix.toString());
        String foldersRequest = query.toString();

        RewriteRequestCallback rewriteRequestCallback =
            new RewriteRequestCallback(
                session,
                next,
                pg,
                new LuceneQueryContext(
                    StringUtils.concat(paramName, ':', prefix.toString()),
                    params,
                    server));


        String originalSuid = params.getString("original-suid", null);
        boolean noLabelsAndFolders = params.getBoolean("nolaf", false);

        if (!noLabelsAndFolders
            && (originalSuid == null
                || originalSuid.equals(
            params.getString(ProxyParams.SUID, null))))
        {
            DoubleFutureCallback<Labels, Folders> callback =
                new DoubleFutureCallback<>(rewriteRequestCallback);
            try {
                AsyncClient labelsClient = server.labelsClient(corp);
                labelsClient.execute(
                    new HeaderAsyncRequestProducerSupplier(
                        new AsyncGetURIRequestProducerSupplier(labelsRequest),
                        new BasicHeader(YandexHeaders.X_YA_SERVICE_TICKET,
                            server.filterSearchTvm2Ticket(corp))),
                    LabelsConsumerFactory.OK,
                    session.httpSession().requestsListener()
                        .createContextGeneratorFor(labelsClient),
                    callback.first());
                AsyncClient foldersClient = server.foldersClient(corp);
                foldersClient.execute(
                    new HeaderAsyncRequestProducerSupplier(
                        new AsyncGetURIRequestProducerSupplier(foldersRequest),
                        new BasicHeader(YandexHeaders.X_YA_SERVICE_TICKET,
                            server.filterSearchTvm2Ticket(corp))),
                    FoldersConsumerFactory.OK,
                    session.httpSession().requestsListener()
                        .createContextGeneratorFor(foldersClient),
                    callback.second());
            } catch (URISyntaxException e) {
                throw new BadRequestException(e);
            }
        } else  {
            rewriteRequestCallback.completed(
                new AbstractMap.SimpleImmutableEntry<>(
                    new Labels(
                        Collections.emptyMap(),
                        "__NO_LID__",
                        "__NO_LID__"),
                    new Folders(
                        Collections.emptyMap(),
                        Collections.emptyMap())));
        }
    }

    private static class RewriteRequestCallback
        extends AbstractSessionCallback<Map.Entry<Labels, Folders>>
    {
        private final SearchRule next;
        private final boolean pg;
        private final LuceneQueryContext context;

        public RewriteRequestCallback(
            final SearchSession session,
            final SearchRule next,
            final boolean pg,
            final LuceneQueryContext context)
        {
            super(session);
            this.next = next;
            this.pg = pg;
            this.context = context;
        }

        private void completed(final Labels labels, final Folders folders)
            throws HttpException
        {
            SearchSession sessionCopy = session.copy();
            CgiParams params = sessionCopy.params();

            StringBuilder request = new StringBuilder();
            checkFlag(request, params, "has_attachments", "has_attachments:1");
            checkFlag(request, params, "only_attachments", ATTACH);
            checkFlag(request, params, "has_links", "x_urls:1");

            filterAll(request, params, "thread_id");
            filterAll(request, params, "message_type");

            final String filter =
                params.getString(MailSearchHandler.SEARCH_FILTER, null);
            if (filter != null) {
                final SearchFilter searchFilter = session.httpSession().server()
                    .config().filtersConfig().filters().get(filter);
                if (searchFilter == null) {
                    throw new BadRequestException(
                        "Invalid &search-filter parameter: unknown filter name"
                        + "<" + filter + '>');
                } else {
                    addCondition(request, searchFilter.apply(labels));
                }
            }

            applyTimerangeFilter(request, context, params);

            if (pg) {
                filterAll(request, params, "fid");
                filterAll(request, params, "folder", "folder_type");
                filterAll(request, params, "lid", "lids");
                checkFlag(request, params, "unread", "unread:1");
            }

            String requestText = params.getString("request", "");
            applyScopesOrQueryLanguage(
                request,
                requestText,
                context,
                session.httpSession(),
                params,
                session.httpSession().server(),
                labels,
                folders);

            checkFlag(request, params, "exclude_attachments", "NOT " + ATTACH);
            Boolean shared = params.getBoolean("shared", null);
            if (shared != null) {
                if (shared == Boolean.TRUE) {
                    addCondition(request, "lids:FAKE_SYNCED_LBL");
                } else {
                    request.append(" AND NOT lids:FAKE_SYNCED_LBL");
                }
            }
            String requestStr = new String(request);
            session.httpSession().logger()
                .info("Rewrite rule request: " + requestStr);
            params.replace("request", requestStr);
            next.execute(sessionCopy);
        }

        @Override
        public void completed(final Map.Entry<Labels, Folders> result) {
            try {
                completed(result.getKey(), result.getValue());
            } catch (HttpException e) {
                failed(e);
            }
        }
    }

    private static class ScopeRewrite {
        private final String paramName;
        private final Collection<String> scope;

        public ScopeRewrite(
            final String paramName,
            final Collection<String> scope)
        {
            this.paramName = paramName;
            this.scope = scope;
        }

        public String paramName() {
            return paramName;
        }

        public Collection<String> scope() {
            return scope;
        }
    }

    private static final UnaryOperator<String> EMAIL_ALIASER =
        new UnaryOperator<String>() {
            @Override
            public String apply(String word) {
                word = word.toLowerCase(Locale.ROOT);
                int sep = MailAliases.emailSeparatorPos(word);
                if (sep != -1) {
                    Set<String> emails =
                        MailAliases.INSTANCE.equivalentEmails(word, sep);
                    if (emails.size() > 1) {
                        StringBuilder sb = new StringBuilder();
                        sb.append('(');
                        for (String email: emails) {
                            sb.append(email);
                            sb.append(" OR ");
                            //sb.append(' ');
                        }
                        sb.setLength(sb.length() - 4);
                        sb.append(')');
                        word = new String(sb);
                    }
                }
                return word;
            }
        };
}

