package ru.yandex.msearch.proxy.api.async.mail.rules;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.logging.Level;

import org.apache.http.HttpException;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.collection.IntInterval;

import ru.yandex.dbfields.MailIndexFields;

import ru.yandex.http.config.ImmutableHttpHostConfig;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;

import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;

import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;

import ru.yandex.json.parser.JsonException;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.proxy.api.async.ProxyParams;

import ru.yandex.msearch.proxy.api.async.mail.SearchSession;
import ru.yandex.msearch.proxy.api.async.mail.documents.Document;
import ru.yandex.msearch.proxy.api.async.mail.documents.Documents;
import ru.yandex.msearch.proxy.api.async.mail.documents.DocumentsGroup;
import ru.yandex.msearch.proxy.highlight
    .AdjustingWordsKeepingRequestMatchesConsumer;

import ru.yandex.msearch.proxy.highlight.DoubleRequestMatchesConsumer;
import ru.yandex.msearch.proxy.highlight.Highlighter;
import ru.yandex.msearch.proxy.highlight.HtmlHighlighter;
import ru.yandex.msearch.proxy.highlight.IntervalRequestMatchesConsumer;
import ru.yandex.msearch.proxy.highlight.RequestMatcher;

import ru.yandex.parser.uri.QueryConstructor;

public class SnippetFetchSearchRule implements SearchRule {
    private static final int DEFAULT_MAX_WORDS = 1;
    private static final int DEFAULT_MIN_SYMBOLS = 7;

    private static final String WAIT_SNIPPETS = "wait-snippets";
    private static final String FACTS_ROUTE = "/facts?mdb=pg";
    private static final String SNIPPET_FACT_NAMES = "snippet,snippet-text";
    private static final String LOG_PREFIX = "snippet-fetch-search-rule";

    private final AsyncClient client;
    private final SearchRule next;
    private final ImmutableHttpHostConfig iexProxyConfig;

    public SnippetFetchSearchRule(
        final SearchRule next,
        final AsyncClient client,
        final ImmutableHttpHostConfig iexProxyConfig)
    {
        this.client = client;
        this.next = next;
        this.iexProxyConfig = iexProxyConfig;
    }

    @Override
    public void execute(SearchSession session) throws HttpException {
        SnippetFetchSearchSession snippetFetchSearchSession =
            new SnippetFetchSearchSession(
                session,
                client,
                iexProxyConfig);
        next.execute(session.withCallback(snippetFetchSearchSession));
    }

    private static class SnippetFetchSearchSession
        extends AbstractSessionCallback<Documents>
    {
        private final AsyncClient client;
        private final PrefixedLogger logger;
        private final ImmutableHttpHostConfig iexProxyConfig;
        private final boolean waitSnippets;

        public SnippetFetchSearchSession(
            final SearchSession session,
            final AsyncClient client,
            final ImmutableHttpHostConfig iexProxyConfig)
            throws BadRequestException
        {
            super(session);
            this.logger = session.httpSession().logger()
                .addPrefix(LOG_PREFIX);
            this.client = client;
            this.iexProxyConfig = iexProxyConfig;
            this.waitSnippets =
                session.params().getBoolean(WAIT_SNIPPETS, false);
        }

        @Override
        public synchronized void completed(final Documents documentsGroups) {
            FutureCallback<JsonObject> callback;
            Map<String, Document> docMap;
            if (!waitSnippets) {
                session.callback().completed(documentsGroups);
                docMap = createMidParams(documentsGroups);
                callback = new LoggingSnippetFetchSearchCallback(logger);
            } else {
                logger.info("Waiting for snippets");
                docMap = createMidParams(documentsGroups);
                callback = new SnippetFetchSearchCallback(
                    session,
                    documentsGroups,
                    docMap);
            }

            try {
                Long uid = session.params().getLong(ProxyParams.UID, null);
                if (uid == null || docMap.isEmpty()) {
                    logger.log(Level.WARNING, "snippet fetch search skip");
                    if (waitSnippets) {
                        session.callback().completed(documentsGroups);
                    }
                    return;
                }
                String uri = createSnippetSearchQuery(uid, docMap.keySet());
                BasicAsyncRequestProducerGenerator generator =
                    new BasicAsyncRequestProducerGenerator(uri);
                client.adjust(session.httpSession().context()).execute(
                    iexProxyConfig.host(),
                    generator,
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    session.httpSession().requestsListener()
                        .createContextGeneratorFor(client),
                    callback);
            } catch (BadRequestException e) {
                logger.log(Level.WARNING, "failed snippet search request", e);
                if (waitSnippets) {
                    callback.failed(e);
                }
            }
        }

        private String createSnippetSearchQuery(
            final Long uid,
            final Collection<String> mids)
            throws BadRequestException
        {
            QueryConstructor uri = new QueryConstructor(FACTS_ROUTE);
            uri.append("uid", uid);
            for (String mid: mids) {
                uri.append("mid", mid);
            }
            uri.append("fact_names", SNIPPET_FACT_NAMES);

            return uri.toString();
        }

        private Map<String, Document> createMidParams(
            final Documents documentsGroups)
        {
            Map<String, Document> mids =
                new LinkedHashMap<>(documentsGroups.size());
            for (DocumentsGroup group : documentsGroups) {
                for (Document doc : group) {
                    mids.put(doc.doc().attrs().get(MailIndexFields.MID), doc);
                }
            }

            return mids;
        }
    }

    public static String prepareSnippet(
        final String request,
        final String snippet,
        final Highlighter highlighter)
    {
        return prepareSnippet(request, snippet, highlighter,
            DEFAULT_MAX_WORDS, DEFAULT_MIN_SYMBOLS);
    }

    public static String prepareSnippet(
        final String request,
        final String snippet,
        final Highlighter highlighter,
        final int maxWords,
        final int minSymbols)
    {
        AdjustingWordsKeepingRequestMatchesConsumer intervals =
            new AdjustingWordsKeepingRequestMatchesConsumer(snippet, maxWords, minSymbols);
        IntervalRequestMatchesConsumer highlights =
            new IntervalRequestMatchesConsumer();

        RequestMatcher.INSTANCE.match(
            snippet,
            request,
            new DoubleRequestMatchesConsumer(intervals, highlights));

        if (intervals.size() == 0) {
            return null;
        } else {
            StringBuilder text = new StringBuilder();
            if (intervals.size() > 0 && intervals.get(0).min() > 0) {
                text.append(".. ");
            }

            int hlIndex = 0;
            for (int i = 0; i < intervals.size(); i++) {
                IntInterval interval = intervals.get(i);
                int pos = interval.min();
                if (highlighter != null) {
                    while (hlIndex < highlights.size()) {
                        IntInterval hlInterval = highlights.get(hlIndex);

                        if (interval.min() <= hlInterval.min()
                            && interval.max() >= hlInterval.max())
                        {
                            if (pos < hlInterval.min()) {
                                text.append(
                                    snippet,
                                    pos,
                                    hlInterval.min());
                            }

                            HtmlHighlighter.INSTANCE.highlightString(
                                text,
                                snippet,
                                hlInterval.min(),
                                hlInterval.max());

                            pos = hlInterval.max();
                        } else if (hlInterval.min() > interval.max()) {
                            break;
                        }

                        hlIndex += 1;
                    }
                }

                if (pos < interval.max()) {
                    text.append(snippet, pos, interval.max());
                }

                boolean last = i == intervals.size() - 1;

                if (!last) {
                    text.append(" .. ");
                } else if (interval.max() < snippet.length() - 1) {
                    text.append(" ..");
                }
            }

            return text.toString();
        }
    }

    private static class LoggingSnippetFetchSearchCallback
        implements FutureCallback<JsonObject>
    {
        private final PrefixedLogger logger;

        private LoggingSnippetFetchSearchCallback(
            final PrefixedLogger logger)
        {
            this.logger = logger;
        }

        @Override
        public void completed(JsonObject o) {
        }

        @Override
        public void failed(final Exception e) {
            logger.log(Level.WARNING, "Snippet fetch search failed", e);
        }

        @Override
        public void cancelled() {
        }
    }

    private static class SnippetFetchSearchCallback
        implements FutureCallback<JsonObject>
    {
        private final SearchSession session;
        private final Documents documents;
        private final Map<String, Document> midsMap;

        public SnippetFetchSearchCallback(
            final SearchSession session,
            final Documents documents,
            final Map<String, Document> midsMap)
        {
            this.session = session;
            this.documents = documents;
            this.midsMap = midsMap;
        }

        @Override
        public void completed(JsonObject result) {
            try {
                JsonMap root = result.asMap();
                boolean fullSnippet =
                    session.params().getBoolean(
                        "full-snippet",
                        true);

                int maxWords =
                    session.params().getInt(
                        "snippet-surround-max-words",
                        DEFAULT_MAX_WORDS);

                int minSymbols =
                    session.params().getInt(
                        "snippet-surround-min-symbols",
                        DEFAULT_MIN_SYMBOLS);

                for (String mid: root.keySet()) {
                    for (JsonObject factObj: root.get(mid).asList()) {
                        JsonMap fact = factObj.asMap();
                        String snippet =
                            fact.getString("text", null);
                        if (snippet != null) {
                            Document doc = midsMap.get(mid);
                            if (doc != null) {
                                if (!fullSnippet) {
                                    snippet =
                                        prepareSnippet(
                                            session.requestInfo().options().request(),
                                            snippet,
                                            session.requestInfo().highlighter(),
                                            maxWords,
                                            minSymbols);
                                }

                                if (snippet != null) {
                                    doc.envelope().put(
                                        "snippet",
                                        new JsonString(snippet));
                                }
                            } else {
                                session.httpSession().logger().warning(
                                    "Missing mid in serp " + mid);
                            }
                        }
                    }
                }
            } catch (JsonException | BadRequestException e) {
                failed(e);
            }

            session.callback().completed(documents);
        }

        @Override
        public void failed(final Exception e) {
            session.httpSession().logger().log(
                Level.WARNING,
                "Snippet fetch search failed",
                e);
            session.callback().completed(documents);
        }

        @Override
        public void cancelled() {
            session.callback().completed(documents);
        }
    }
}