package ru.yandex.msearch.proxy.api.async.mail.rules;

import com.google.common.hash.Hashing;

import java.net.InetAddress;
import java.net.SocketTimeoutException;
import java.net.URISyntaxException;
import java.net.UnknownHostException;

import java.nio.charset.CharacterCodingException;
import java.nio.charset.StandardCharsets;

import java.util.Arrays;
import java.util.Locale;

import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;

import java.util.logging.Level;

import org.apache.http.HttpException;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.function.StringBuilderProcessorAdapter;
import ru.yandex.function.StringVoidProcessor;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.HttpStatusPredicates;

import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;

import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.ProxyParams;
import ru.yandex.msearch.proxy.api.async.mail.documents.Documents;
import ru.yandex.msearch.proxy.api.async.mail.SearchSession;
import ru.yandex.msearch.proxy.config.ImmutableSoCheckConfig;
import ru.yandex.msearch.proxy.socheck.HttpSoCheck;
import ru.yandex.msearch.proxy.socheck.SoCheck;
import ru.yandex.msearch.proxy.socheck.SoCheckConsumerFactory;

import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.PctEncoder;
import ru.yandex.parser.uri.PctEncodingRule;

public class SoCheckRule implements SearchRule {
    private static final int MAX_LENGTH = 4096;
    private static final StatusCheckAsyncResponseConsumerFactory<
    SoCheck.Result> CONSUMER_FACTORY =
        new StatusCheckAsyncResponseConsumerFactory<>(
            HttpStatusPredicates.OK,
            SoCheckConsumerFactory.INSTANCE);

    private final AtomicLong botNetRequestNumber = new AtomicLong();
    private final SearchRule next;
    private final AsyncHttpServer server;
    private final AsyncClient client;
    private final ImmutableSoCheckConfig config;
    private final String uri;
    private final String hostname;

    public SoCheckRule(
        final SearchRule next,
        final AsyncHttpServer server)
    {
        this.next = next;
        this.server = server;
        client = server.soCheckClient();
        config = server.config().soCheckConfig();
        uri = config.uri().toASCIIString()
            + config.firstCgiSeparator()
            + "so_form_name=msearchproxyform&so_service=MSEARCH-PROXY&so_uid=";
        try {
            hostname = InetAddress.getLocalHost().getHostName();
        } catch (UnknownHostException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void execute(final SearchSession session) throws HttpException {
        SoCheckSession soCheckSession = new SoCheckSession(session);
        if (soCheckSession.request() == null) {
            next.execute(session);
        } else {
            next.execute(session.withCallback(soCheckSession));
            try {
                String uri = soCheckSession.request() + '1';
                    session.httpSession().logger().info(
                        "socheck: Sending SO request: " + uri);
                client.execute(
                    new AsyncGetURIRequestProducerSupplier(uri),
                    CONSUMER_FACTORY,
                    session.httpSession().requestsListener()
                        .createContextGeneratorFor(client),
                    new SoCheckCallback(soCheckSession));
            } catch (URISyntaxException e) {
                throw new BadRequestException(e);
            }
        }
    }

    private class SoCheckSession extends AbstractSessionCallback<Documents> {
        private final long start = System.currentTimeMillis();
        private final PrefixedLogger logger;
        private final char imap;
        private final String user;
        private final String normalizedRequest;
        private final String request;
        private SoCheck.Result soCheckResult = null;
        private Documents documents = null;

        public SoCheckSession(final SearchSession session)
            throws HttpException
        {
            super(session);
            logger = session.httpSession().logger().addPrefix("socheck");
            CgiParams params = session.params();
            if (params.getBoolean("nosocheck", false)) {
                imap = '0';
                user = null;
                normalizedRequest = null;
                request = null;
                logger.fine("'nosocheck' requested, skipping");
            } else {
                String request = params.getString(ProxyParams.REQUEST, "");
                String mdb = params.getString(ProxyParams.MDB, null);
                String suid = params.getString(ProxyParams.SUID, null);
                String ip = params.getString("remote_ip", "127.0.0.1");
                if (suid == null || mdb == null || request.trim().isEmpty()) {
                    imap = '0';
                    user = null;
                    normalizedRequest = null;
                    this.request = null;
                    logger.fine("Request is incomplete, skipping SO check");
                } else {
                    try {
                        StringVoidProcessor<char[], CharacterCodingException>
                            encoder =
                                new StringVoidProcessor<>(
                                    new PctEncoder(PctEncodingRule.QUERY));
                        StringBuilder sb = new StringBuilder(uri);
                        StringBuilderProcessorAdapter sbAdapter =
                            new StringBuilderProcessorAdapter(sb);
                        sb.append(
                            Hashing.murmur3_128().newHasher()
                            .putString(
                                ip + request + hostname + suid,
                                StandardCharsets.UTF_8)
                            .putLong(start)
                            .hash()
                            .toString());
                        sb.append("&so_ip=");
                        sb.append(ip);
                        String side = params.getString("side", null);
                        if (side != null) {
                            sb.append("&side=");
                            sb.append(side);
                        }
                        sb.append("&text=");
                        if (request.length() > MAX_LENGTH) {
                            request = request.substring(0, MAX_LENGTH);
                        }
                        encoder.process(request);
                        encoder.processWith(sbAdapter);
                        if ("pg".equals(mdb)) {
                            sb.append("&uid=");
                            sb.append(params.getString(ProxyParams.UID));
                        }
                        sb.append("&suid=");
                        sb.append(suid);
                        sb.append("&so_mdb=");
                        sb.append(mdb);
                        sb.append("&so_offset=");
                        sb.append(params.getString("offset", "0"));
                        sb.append("&imap=");
                        if (params.getString("imap", "").equals("1")) {
                            String userRequest =
                                params.getString("user_request", null);
                            if (userRequest == null) {
                                imap = '1';
                            } else {
                                imap = '2';
                            }
                            normalizedRequest =
                                request.replaceAll(" +", " ").trim()
                                .replace('@', '.').toLowerCase(Locale.ENGLISH);
                        } else {
                            imap = '0';
                            normalizedRequest =
                                HttpSoCheck.normalizeRequest(request);
                        }
                        if (normalizedRequest.isEmpty()) {
                            user = null;
                            this.request = null;
                        } else {
                            sb.append(imap);
                            sb.append("&norm=");
                            encoder.process(normalizedRequest);
                            encoder.processWith(sbAdapter);
                            sb.append("&step=");
                            if ("pg".equals(mdb)) {
                                user =
                                    params.getString(ProxyParams.UID, null)
                                    + "@pg";
                            } else {
                                user = suid + '@' + mdb;
                            }
                            logger.fine(
                                "For user " + user
                                + " request normalized to: "
                                + normalizedRequest);
                            this.request = new String(sb);
                            logger.fine("Request prepared: " + request);
                        }
                    } catch (CharacterCodingException e) {
                        throw new BadRequestException(e);
                    }
                }
            }
        }

        public PrefixedLogger logger() {
            return logger;
        }

        public String request() {
            return request;
        }

        private void checkCompleted() {
            if (done || soCheckResult == null || documents == null) {
                return;
            }
            done = true;
            try {
                client.execute(
                    new AsyncGetURIRequestProducerSupplier(
                        request + "2&found=" + documents.size()),
                    EmptyFutureCallback.INSTANCE);
            } catch (URISyntaxException e) {
                logger.log(Level.WARNING, "Step 2 request failed", e);
            }
            server.soCheckResult(
                soCheckResult,
                documents.size() == 0,
                imap - '0');
            logger.info( "socheck: check result is: " + soCheckResult);
            if (soCheckResult == SoCheck.Result.SPAM) {
                if (config.banSpam()) {
                    logger.info("Looks like bot-net, dropping search results");
                    if (config.fakeRequest() != null
                        && botNetRequestNumber.incrementAndGet()
                            % config.fakeRequestInterval() == 0L)
                    {
                        logger.info("Performing fake request");
                        SearchSession session = this.session.copy();
                        CgiParams params = session.params();
                        params.replace("request", config.fakeRequest());
                        params.replace("force", "true");
                        params.replace(
                            "scope",
                            Arrays.asList("body_text", "pure_body"));
                        params.remove("imap");
                        try {
                            next.execute(session);
                        } catch (HttpException e) {
                            logger.log(
                                Level.WARNING,
                                "Failed to execute fake request",
                                e);
                            session.callback().failed(e);
                        }
                    } else {
                        logger.info("Returning empty SERP");
                        documents.clear();
                        session.callback().completed(documents);
                    }
                } else {
                    logger.info("ban_spam is not set, ignoring result");
                    session.callback().completed(documents);
                }
            } else {
                session.callback().completed(documents);
            }
            logger.info((System.currentTimeMillis() - start) + " ms taken");
        }

        public synchronized void soCheckResult(
            final SoCheck.Result soCheckResult)
        {
            this.soCheckResult = soCheckResult;
            logger.info("SO result: " + soCheckResult);
            logger.info(
                "socheck: " + soCheckResult + ", user: " + user
                + ", imap: " + imap + ", request: " + normalizedRequest);
            checkCompleted();
        }

        @Override
        public synchronized void completed(final Documents documents) {
            this.documents = documents;
            logger.info("Search result received");
            checkCompleted();
        }
    }

    private static class SoCheckCallback
        implements FutureCallback<SoCheck.Result>
    {
        private final SoCheckSession session;

        public SoCheckCallback(final SoCheckSession session) {
            this.session = session;
        }

        @Override
        public void completed(final SoCheck.Result result) {
            session.soCheckResult(result);
        }

        @Override
        public void failed(final Exception e) {
            String message;
            SoCheck.Result result;
            if (e instanceof BadResponseException) {
                message = "Error received from SO";
                result = SoCheck.Result.FAILED;
            } else if (e instanceof SocketTimeoutException) {
                message = "Request timed out";
                result = SoCheck.Result.TIMEOUT;
            } else if (e instanceof TimeoutException) {
                message = "No available connections to SO";
                result = SoCheck.Result.UNKNOWN;
            } else {
                message = "Request failed";
                result = SoCheck.Result.REQUEST_FAILED;
            }
            session.logger().log(Level.WARNING, message, e);
            session.soCheckResult(result);
        }

        @Override
        public void cancelled() {
            session.logger().warning("Request cancelled");
            session.soCheckResult(SoCheck.Result.INTERRUPTED);
        }
    }
}

