package ru.yandex.msearch.proxy.api.async.mail.rules;

import java.io.IOException;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;

import org.apache.http.HttpException;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.protocol.HttpContext;

import ru.yandex.concurrent.TimeFrameQueue;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.EmptyFutureCallback;

import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;

import ru.yandex.http.util.nio.client.AsyncClient;

import ru.yandex.http.util.request.RequestHandlerMapper;

import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.http.util.server.UpstreamStaterFutureCallback;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.proxy.AsyncHttpServer;

import ru.yandex.msearch.proxy.api.async.ProxyParams;

import ru.yandex.msearch.proxy.api.async.mail.SearchSession;
import ru.yandex.msearch.proxy.api.async.mail.documents.Documents;

import ru.yandex.msearch.proxy.api.async.suggest.history.AbstractStoredRequest;
import ru.yandex.msearch.proxy.api.async.suggest.history.DeleteStoredRequest;

import ru.yandex.msearch.proxy.api.async.suggest.history.UpdateStoredRequest;
import ru.yandex.msearch.proxy.api.mail.MailSearchOptions;

import ru.yandex.msearch.proxy.config.ImmutableMsearchProxyConfig;

import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.searchmap.User;

import ru.yandex.parser.uri.CgiParams;

import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.PrefixType;

/**
 * TokenizeRule for storing search requests from user in index. Only Successful
 * search attempts are stored.
 */
public class StoreSearchRequestRule implements SearchRule {
    // protected for adapter in api/mail
    protected final SearchMap searchMap;
    protected final AsyncClient producerClient;
    protected final ImmutableMsearchProxyConfig proxyConfig;
    protected final int minimumRequestStoreLen;

    private final SearchRule next;

    protected TimeFrameQueue<Long> storeRequests = null;
    protected UpstreamStater storeStater = null;

    public StoreSearchRequestRule(
        final SearchRule next,
        final AsyncClient client,
        final ImmutableMsearchProxyConfig proxyConfig,
        final SearchMap searchMap)
    {
        this.next = next;
        this.proxyConfig = proxyConfig;
        this.producerClient = client;
        this.searchMap = searchMap;
        minimumRequestStoreLen =
            proxyConfig.suggestConfig()
                .subjectConfig().minimumRequestStoreLen();
    }

    public StoreSearchRequestRule(
        final SearchRule next,
        final AsyncHttpServer server)
    {
        this(
            next,
            server.producerStoreClient(),
            server.config(),
            server.searchMap());

        this.storeRequests = server.producerStoreRequests();
        this.storeStater = server.producerStoreStater();
    }

    protected FutureCallback<Object> producerCallback() {
        FutureCallback<Object> callback =
            EmptyFutureCallback.INSTANCE;

        if (storeStater != null) {
            callback =
                new UpstreamStaterFutureCallback<>(
                    new ProducerStoreCallback(storeRequests),
                    storeStater);
        }

        return callback;
    }

    protected boolean isStorable(
        final String trimRequest,
        final CgiParams params)
        throws BadRequestException
    {
        return trimRequest.length() >= minimumRequestStoreLen
            && params.getBoolean("save-request", true)
            && !params.getBoolean("imap", false)
            && !params.getBoolean("nostore", false);
    }

    @Override
    public void execute(final SearchSession session) throws HttpException {
        String request = session.params().getString("request", "").trim();
        if (!isStorable(request, session.params())) {
            next.execute(session);
        } else {
            StoreSearchRequestSession storeSession =
                new StoreSearchRequestSession(session);
            next.execute(session.withCallback(storeSession));
        }
    }

    /**
     *
     * @param options
     * @return true if erratum was used as final user request
     */
    protected boolean erratumUsed(final MailSearchOptions options) {
        return options.rule() != null && options.suggest() == null;
    }

    protected List<AbstractStoredRequest> storeRequests(
        final Prefix prefix,
        final String text,
        final MailSearchOptions options,
        final int docsCount)
    {
        // we already sent request for storing,
        // if modified request was finally used, we should delete initial one
        List<AbstractStoredRequest> requests = new ArrayList<>(2);
        if (docsCount > 0) {
            if (erratumUsed(options) && !text.equalsIgnoreCase(options.request())) {
                requests.add(createDeleteRequest(prefix, text));
                requests.add(createUpdateRequest(prefix, text, options));
            } else {
                if (options.suggest() != null) {
                    requests.add(createUpdateRequest(prefix, text, options));
                }
            }
        } else {
            requests.add(createDeleteRequest(prefix, text));
        }

        return requests;
    }

    protected DeleteStoredRequest createDeleteRequest(
        final Prefix prefix,
        final String text)
    {
        return new DeleteStoredRequest(prefix, text);
    }

    protected UpdateStoredRequest createUpdateRequest(
        final Prefix prefix,
        final String text,
        final MailSearchOptions options)
    {
        UpdateStoredRequest request;
        if (options.rule() != null) {
            if (options.suggest() == null) {
                // erratumn result was used for request
                request = new UpdateStoredRequest(prefix, options.request());
                request.corrected(text);
                return request;
            }
        }

        request = new UpdateStoredRequest(prefix, text);

        if (options.suggest() != null) {
            request.suggest(options.suggest());
        }

        return request;
    }

    private class StoreSearchRequestSession
        extends AbstractSessionCallback<Documents>
    {
        private final PrefixedLogger logger;
        private final String userRequest;
        private final AsyncClient client;
        private final User user;

        StoreSearchRequestSession(
            final SearchSession session)
            throws HttpException
        {
            super(session);

            this.logger = session.httpSession().logger().addPrefix(
                "storeRequest");
            this.client =
                producerClient.adjust(session.httpSession().context());
            this.user = fetchUser(session.params());
            this.userRequest =
                session.params().getString(ProxyParams.REQUEST, "");

            AbstractStoredRequest request =
                createUpdateRequest(
                    user.prefix(),
                    userRequest,
                    session.requestInfo().options());

            // ok, for faster indexation we always store request in index
            // after request completed, we will delete it
            FutureCallback<Object> producerCallback = producerCallback();
            try {
                client.execute(
                    proxyConfig.producerStoreConfig().host(),
                    new BasicAsyncRequestProducerGenerator(
                        request.uri(
                            user,
                            "msproxy_async",
                            session.httpSession().context())
                            + "&prerequest",
                        request.toJsonString(),
                        ContentType.APPLICATION_JSON),
                    BasicAsyncResponseConsumerFactory.ANY_GOOD,
                    producerCallback);
            } catch (IOException e) {
                producerCallback.failed(e);
            }
        }

        private User fetchUser(final CgiParams params)
            throws BadRequestException
        {
            String mdb = params.getString(ProxyParams.MDB);

            Prefix prefix;
            if (mdb.equals("pg")) {
                prefix = params.get(ProxyParams.UID, PrefixType.LONG);
            } else {
                prefix = params.get(ProxyParams.SUID, PrefixType.LONG);
            }

            return new User(
                AsyncHttpServer.resolveService(mdb, prefix, proxyConfig),
                prefix);
        }

        @Override
        public void completed(final Documents documentsGroups) {
            done = true;

            try {

                HttpContext context = session.httpSession().context();

                MailSearchOptions options = session.requestInfo().options();
                int docs = documentsGroups.size();

                for (AbstractStoredRequest request
                    : storeRequests(user.prefix(), userRequest, options, docs))
                {
                    BasicAsyncRequestProducerGenerator generator;
                    if (request.method() == RequestHandlerMapper.GET) {
                        generator = new BasicAsyncRequestProducerGenerator(
                            request.uri(user, "msproxy_async", context));
                    } else {
                        generator =
                            new BasicAsyncRequestProducerGenerator(
                                request.uri(
                                    user,
                                    "msproxy_async",
                                    context),
                                request.toJsonString(),
                                ContentType.APPLICATION_JSON);
                    }

                    client.execute(
                        proxyConfig.producerStoreConfig().host(),
                        generator,
                        BasicAsyncResponseConsumerFactory.ANY_GOOD,
                        producerCallback());
                }

                logger.info("Request sent to be stored in index");
            } catch (IOException | HttpException e) {
                logger.log(Level.WARNING, "Failed to store request", e);
            }

            session.callback().completed(documentsGroups);
        }
    }

    private static final class ProducerStoreCallback
        implements FutureCallback<Object>
    {
        private final TimeFrameQueue<Long> storeRequests;

        public ProducerStoreCallback(final TimeFrameQueue<Long> storeRequests) {
            this.storeRequests = storeRequests;
        }

        @Override
        public void completed(final Object o) {
            storeRequests.accept(0L);
        }

        @Override
        public void failed(final Exception e) {
            storeRequests.accept(1L);
        }

        @Override
        public void cancelled() {
            storeRequests.accept(1L);
        }
    }
}
