package ru.yandex.msearch.proxy.api.async.mail.rules;

import java.util.logging.Level;

import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpRequest;
import org.apache.http.concurrent.FutureCallback;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.msearch.proxy.api.async.ProxyParams;
import ru.yandex.msearch.proxy.api.async.mail.SearchSession;
import ru.yandex.msearch.proxy.api.async.mail.documents.Documents;
import ru.yandex.msearch.proxy.experiment.MalformedUserSplitException;
import ru.yandex.msearch.proxy.experiment.UserSplit;
import ru.yandex.msearch.proxy.experiment.UserSplitAsyncResponseConsumerFactory;
import ru.yandex.parser.uri.QueryConstructor;

public class UserSplitRule implements SearchRule {
    private static final String SERVICE_NAME = "mail";
    private static final String ROUTE = "/" + SERVICE_NAME + "?";
    private static final String DEFAULT_USER_AGENT = "mail-search";
    private static final String USERSPLIT_PREFIX = "usersplit";

    private static final UserSplitAsyncResponseConsumerFactory CONSUMER_FACTORY
        = new UserSplitAsyncResponseConsumerFactory();

    private final AsyncClient client;
    private final SearchRule next;
    private final ImmutableHttpHostConfig userSplitConfig;

    public UserSplitRule(
        final SearchRule next,
        final AsyncClient client,
        final ImmutableHttpHostConfig userSplitConfig)
    {
        this.client = client;
        this.next = next;
        this.userSplitConfig = userSplitConfig;
    }

    @Override
    public void execute(final SearchSession session) throws HttpException {
        PrefixedLogger logger = session.httpSession().logger()
            .addPrefix(USERSPLIT_PREFIX);

        Long uid = session.params().getLong(ProxyParams.UID, null);
        HttpRequest request = session.requestInfo().request();

        Header enabledBoxed =
            request.getFirstHeader(YandexHeaders.X_ENABLED_BOXES);

        Header realIpHeader = request.getLastHeader(YandexHeaders.X_REAL_IP);

        if (enabledBoxed == null &&
            (userSplitConfig == null || uid == null || realIpHeader == null))
        {
            this.next.execute(session);
            session.httpSession().logger().warning("UserSplit skipped");
            return;
        }

        if (enabledBoxed != null) {
            UserSplit userSplit;
            try {
                session.requestInfo().options().userSplit(
                    new UserSplit("none", enabledBoxed.getValue(), ""));

                this.next.execute(session);
                return;
            } catch (MalformedUserSplitException mue) {
                logger.info(
                    "Bad header "
                        + YandexHeaders.X_ENABLED_BOXES
                        + " " + enabledBoxed.getValue());
            }
        }

        next.execute(session);

        Header userAgentHeader =
            session.requestInfo().request()
                .getLastHeader(HttpHeaders.USER_AGENT);

        QueryConstructor uri = new QueryConstructor(ROUTE);
        uri.append("uuid", uid.toString());
        uri.append("service", SERVICE_NAME);

        BasicAsyncRequestProducerGenerator generator =
            new BasicAsyncRequestProducerGenerator(uri.toString());
        generator.addHeader(
            YandexHeaders.X_FORWARDED_FOR_Y,
            realIpHeader.getValue());

        if (userAgentHeader != null) {
            generator.addHeader(userAgentHeader);
        } else {
            generator.addHeader(HttpHeaders.USER_AGENT, DEFAULT_USER_AGENT);
        }

        logger.info(generator.toString());

        session.httpSession().subscribeForCancellation(
            client.execute(
                userSplitConfig.host(),
                generator,
                CONSUMER_FACTORY,
                new UserSplitCallback(session)));
    }

    private static class UserSplitCallback
        implements FutureCallback<UserSplit>
    {
        private final SearchSession session;

        public UserSplitCallback(final SearchSession session) {
            this.session = session;
        }

        @Override
        public void completed(final UserSplit userSplit) {
            session.requestInfo().options().userSplit(userSplit);
            session.httpSession().logger().info(
                "User participating in experiments: "
                    + String.valueOf(userSplit));
        }

        @Override
        public void failed(final Exception e) {
            session.httpSession().logger()
                .log(Level.WARNING, "UserSplit failed", e);
        }

        @Override
        public void cancelled() {
        }
    }
}
