package ru.yandex.msearch.proxy.api.async.mail.searcher;

import java.util.List;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.stream.Collectors;

import org.apache.http.HttpHost;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;

import ru.yandex.client.producer.QueueHostInfo;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.Session;

import ru.yandex.parser.searchmap.User;

import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.proxy.SearchResultConsumerFactory;

import ru.yandex.search.result.SearchResult;

public class ProducerParallelSearcher
    extends AbstractPlainSearcher<SearchResult>
{
    protected final AsyncHttpServer server;
    protected final User user;

    public ProducerParallelSearcher(
        final Session session,
        final CgiParams params,
        final User user)
        throws BadRequestException
    {
        super(
            session,
            params,
            session.server().searchClient(),
            session.server().producerClient());

        this.server = session.server();
        this.user = user;
    }

    public ProducerParallelSearcher(
        final AsyncHttpServer server,
        final ProxySession session,
        final AsyncClient searchClient,
        final User user)
        throws BadRequestException
    {
        super(
            session,
            searchClient,
            server.producerClient());

        this.server = server;
        this.user = user;
    }

    public ProducerParallelSearcher(
        final AsyncHttpServer server,
        final ProxySession session,
        final User user)
        throws BadRequestException
    {
        this(server, session, server.searchClient(), user);
    }

    @Override
    public void search(
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
            request,
        final FutureCallback<SearchResult> callback)
    {
        List<HttpHost> hosts = hosts(server.searchMap(), user);

        ThresholdPositionSearcherCallback posCallback =
            new ThresholdPositionSearcherCallback(
                callback,
                server.freshBackendsResponsesCountStats(),
                logger(),
                hosts.size());

        if (producerClient != null) {
            long now = System.currentTimeMillis();

            producerClient.executeWithInfo(
                user,
                listener().createContextGeneratorFor(producerClient),
                new HostsCallback(posCallback, now));
        } else {
            logger().info("Producer not enabled, use searchmap");
        }

        for (int i = hosts.size(); i-- > 0;) {
            proxySession.subscribeForCancellation(
                searchClient.execute(
                    hosts.get(i),
                    request,
                    SearchResultConsumerFactory.OK,
                    listener().createContextGeneratorFor(searchClient),
                    posCallback));
        }
    }

    protected class HostsCallback
        implements FutureCallback<List<QueueHostInfo>>
    {
        protected final ThresholdPositionSearcherCallback callback;

        private final long startTime;
        protected boolean done = false;

        public HostsCallback(
            final ThresholdPositionSearcherCallback callback,
            final long startTime)
        {
            this.startTime = startTime;
            this.callback = callback;
        }

        @Override
        public synchronized void completed(final List<QueueHostInfo> hosts) {
            if (!done) {
                done = true;
                int size = hosts.size();
                server.resolvedSearchBackendsCount(size, startTime);

                if (size > 0) {
                    long position = hosts.get(0).queueId();
                    if (position > 0) {
                        callback.thresholdPosition(hosts.get(0).queueId());
                    }
                }
            }
        }

        @Override
        public synchronized void failed(final Exception e) {
            if (!done) {
                done = true;

                logger().log(
                    Level.WARNING,
                    "Producer call failed, waiting response from all backends",
                    e);
            }
        }

        @Override
        public synchronized void cancelled() {
            if (!done) {
                done = true;
            }
        }
    }
}
