package ru.yandex.msearch.proxy.api.async.mail.searcher;

import java.util.List;
import java.util.function.Function;

import org.apache.http.HttpHost;

import org.apache.http.concurrent.FutureCallback;

import org.apache.http.nio.protocol.HttpAsyncRequestProducer;

import ru.yandex.client.producer.ProducerClient;

import ru.yandex.http.proxy.ProxySession;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.NotFoundException;

import ru.yandex.http.util.nio.client.AsyncClient;

import ru.yandex.msearch.proxy.AsyncHttpServer;

import ru.yandex.msearch.proxy.api.async.mail.SearchSession;

import ru.yandex.parser.searchmap.User;

import ru.yandex.parser.uri.CgiParams;

import ru.yandex.search.proxy.SearchResultConsumerFactory;

import ru.yandex.search.result.BasicSearchResult;
import ru.yandex.search.result.SearchResult;

public class ProducerSequentialSearcher
    extends AbstractPlainSearcher<SearchResult>
{
    private final AsyncHttpServer server;
    private final User user;
    protected final long failoverDelay;

    public ProducerSequentialSearcher(
        final AsyncHttpServer server,
        final SearchSession session,
        final User user)
        throws BadRequestException
    {
        this(
            server,
            session,
            server.searchClient(),
            server.producerClient(),
            user);
    }

    public ProducerSequentialSearcher(
        final AsyncHttpServer server,
        final SearchSession session,
        final AsyncClient searchClient,
        final ProducerClient producerClient,
        final User user)
        throws BadRequestException
    {
        super(
            session.httpSession(),
            session.params(),
            searchClient,
            producerClient);

        this.server = server;
        this.user = user;

        this.failoverDelay = failoverSearchDelay(session.params());
    }

    public ProducerSequentialSearcher(
        final AsyncHttpServer server,
        final ProxySession session,
        final User user)
        throws BadRequestException
    {
        this(server, session, server.searchClient(), user);
    }

    public ProducerSequentialSearcher(
        final AsyncHttpServer server,
        final ProxySession session,
        final AsyncClient searchClient,
        final User user)
        throws BadRequestException
    {
        super(session, searchClient, server.producerClient());

        this.server = server;
        this.user = user;

        this.failoverDelay = failoverSearchDelay(session.params());
    }

    @Override
    public void search(
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
            request,
        final FutureCallback<SearchResult> callback)
    {
        long now = System.currentTimeMillis();

        if (producerClient == null) {
            List<HttpHost> hosts =
                server.searchMap().searchHosts(user);
            if (hosts.isEmpty()) {
                server.resolvedSearchBackendsCount(0, now);
                callback.failed(
                    new NotFoundException(
                        "No hosts found for user: " + user));
            } else {
                search(request, callback, hosts);
            }
        } else {
            producerClient.execute(
                user,
                listener().createContextGeneratorFor(producerClient),
                new HostsCallback(callback, now, request));
        }
    }

    private long failoverSearchDelay(final CgiParams params) {
        long failoverSearchDelay = server.config().failoverSearchDelay();
        try {
            failoverSearchDelay = params.getLongDuration(
                "failover-delay",
                failoverSearchDelay);
        } catch (BadRequestException bre) {
        }

        return failoverSearchDelay;
    }

    protected void search(
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer> r,
        final FutureCallback<SearchResult> callback,
        final List<HttpHost> hosts)
    {
        proxySession.subscribeForCancellation(
            searchClient.executeWithDelay(
                hosts,
                r,
                failoverDelay,
                SearchResultConsumerFactory.OK,
                listener().createContextGeneratorFor(searchClient),
                callback));
    }


    private class HostsCallback implements FutureCallback<List<HttpHost>> {
        private final long startTime;
        private final FutureCallback<SearchResult> callback;
        private final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
            request;
        private boolean done = false;


        public HostsCallback(
            final FutureCallback<SearchResult> callback,
            final long startTime,
            final Function<? super HttpHost, ? extends
                HttpAsyncRequestProducer> request)
        {
            this.startTime = startTime;
            this.request = request;
            this.callback = callback;
        }

        @Override
        public synchronized void completed(final List<HttpHost> hosts) {
            if (!done) {
                done = true;

                int size = hosts.size();
                server.resolvedSearchBackendsCount(size, startTime);

                if (size > 0) {
                    search(request, callback, hosts);
                } else {
                    callback.completed(new BasicSearchResult());
                }
            }
        }

        @Override
        public synchronized void failed(final Exception e) {
            if (!done) {
                done = true;
                logger.info("Producer request failed, use searchmap");
                List<HttpHost> hosts =
                    server.searchMap().searchHosts(user);
                if (hosts.isEmpty()) {
                    server.resolvedSearchBackendsCount(
                        0,
                        System.currentTimeMillis());

                    callback.failed(
                        new NotFoundException(
                            "No hosts found for user: " + user));
                } else {
                    search(request, callback, hosts);
                }
            }
        }

        @Override
        public synchronized void cancelled() {
            if (!done) {
                done = true;
                callback.cancelled();
            }
        }
    }
}
