package ru.yandex.msearch.proxy.api.async.mail.searcher;

import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

import java.util.function.Consumer;
import java.util.function.Function;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.concurrent.TimeFrameQueue;

import ru.yandex.http.util.RequestErrorType;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.search.result.SearchResult;

public class ThresholdPositionSearcherCallback
    implements FutureCallback<SearchResult>
{
    private final FutureCallback<SearchResult> callback;
    private final Function<Exception, RequestErrorType> errorClassifier;
    private final AtomicInteger requests;
    private final AtomicLong thresholdPos;
    private final Consumer<Integer> freshnessStater;
    private final PrefixedLogger logger;
    private SearchResult best;
    private Exception e = null;

    public ThresholdPositionSearcherCallback(
        final FutureCallback<SearchResult> callback,
        final Function<Exception, RequestErrorType> errorClassifier,
        final Consumer<Integer> stater,
        final PrefixedLogger logger,
        final int requests)
    {
        this.callback = callback;
        this.requests = new AtomicInteger(requests);
        this.thresholdPos = new AtomicLong(Long.MAX_VALUE);
        this.errorClassifier = errorClassifier;
        this.freshnessStater = stater;
        this.logger = logger;
    }

    public ThresholdPositionSearcherCallback(
        final FutureCallback<SearchResult> callback,
        final PrefixedLogger logger,
        final int requests)
    {
        this(
            callback,
            RequestErrorType.ERROR_CLASSIFIER,
            new NullIntegerConsumer(),
            logger,
            requests);
    }

    public ThresholdPositionSearcherCallback(
        final FutureCallback<SearchResult> callback,
        final TimeFrameQueue<Integer> freshnessStater,
        final PrefixedLogger logger,
        final int requests)
    {
        this(
            callback,
            RequestErrorType.ERROR_CLASSIFIER,
            freshnessStater,
            logger,
            requests);
    }

    @Override
    public void cancelled() {
        if (requests.decrementAndGet() == 0) {
            callback.cancelled();
        }
    }

    public void thresholdPosition(final long thresholdPos) {
        this.thresholdPos.set(thresholdPos);
        logger.info("Top backend position set to " + thresholdPos);

        SearchResult result = null;
        synchronized (this) {
            if (best != null && best.zooQueueId() >= thresholdPos) {
                result = best;
            }
        }

        if (result != null) {
            if (requests.getAndSet(0) > 0) {
                logger.info(
                    resultInfo(result).append(" already over threshold")
                        .toString());

                freshnessStater.accept(1);
                callback.completed(result);
            }
        }
    }

    private StringBuilder resultInfo(final SearchResult result) {
        StringBuilder sb = new StringBuilder("Backend ");
        if (result.host() != null) {
            sb.append(result.host().toHostString());
        }

        sb.append(" position ");
        sb.append(result.zooQueueId());

        return sb;
    }

    @Override
    public void completed(final SearchResult result) {
        if (requests.get() <= 0) {
            return;
        }

        if (result.zooQueueId() >= thresholdPos.get()) {
            if (requests.getAndSet(0) > 0) {
                logger.info(
                    resultInfo(result).append(" over threshold").toString());

                freshnessStater.accept(1);
                callback.completed(result);
            }

            return;
        }

        SearchResult retResult = null;

        synchronized (this) {
            if (best == null) {
                best = result;
            } else {
                if (best.zooQueueId() < result.zooQueueId()) {
                    best = result;
                }
            }

            if (requests.decrementAndGet() == 0) {
                retResult = best;
            }
        }

        if (retResult != null) {
            logger.info(resultInfo(result).append(" choose best").toString());
            freshnessStater.accept(0);
            callback.completed(retResult);
        } else {
            logger.info(resultInfo(result).append(" completed").toString());
        }
    }

    @Override
    public void failed(final Exception e) {
        RequestErrorType errorType = this.errorClassifier.apply(e);
        if (errorType == RequestErrorType.NON_RETRIABLE) {
            if (requests.getAndSet(0) > 0) {
                callback.failed(e);
            }

            return;
        }

        Exception ex = null;
        SearchResult result = null;
        synchronized (this) {
            if (this.e == null) {
                this.e = e;
            } else {
                this.e.addSuppressed(e);
            }
            if (requests.decrementAndGet() == 0) {
                if (best != null) {
                    result = best;
                } else {
                    ex = this.e;
                    this.e = null;
                }
            }
        }

        if (result != null) {
            logger.info(
                resultInfo(result).append(" choose best, after last failed")
                    .toString());

            freshnessStater.accept(0);
            callback.completed(result);
        } else if (ex != null) {
            callback.failed(ex);
        }
    }

    private static class NullIntegerConsumer implements Consumer<Integer> {
        @Override
        public void accept(final Integer integer) {
        }
    }
}