package ru.yandex.msearch.proxy.api.async.mail.spaniel;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Logger;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.client.wmi.Folders;
import ru.yandex.client.wmi.Labels;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.logger.SearchProxyAccessLoggerConfigDefaults;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.mail.DefaultSearchAttributes;
import ru.yandex.msearch.proxy.api.async.mail.rules.LuceneQueryContext;
import ru.yandex.msearch.proxy.api.async.mail.rules.RewriteRequestRule;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.string.PositiveLongValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.document.mail.FolderType;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.search.request.util.SearchRequestText;

public class SpanielSearchHandler implements ProxyRequestHandler {
    private static final long FAILOVER_DELAY = 1000L;
    private static List<String> SEARCH_FIELDS
        = Arrays.asList("pure_body", "hdr_from", "hdr_to", "hdr_cc", "hdr_subject");

    private final AsyncHttpServer server;


    public SpanielSearchHandler(final AsyncHttpServer server) {
        this.server = server;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        SpanielSearchContext context;
        if (session.request() instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity = ((HttpEntityEnclosingRequest) session.request()).getEntity();
            context = new SpanielSearchContext(session, entity);
        } else {
            context = new SpanielSearchContext(session, null);
        }

        SpanielPrinter printer = new SpanielPrinter(context);

        RequestExecutor executor = new RequestExecutor(printer, context, 20);
        executor.start();
//        Map<SearchMapShard, List<User>> shardMap
//            = new LinkedHashMap<>(context.uids().size() << 1);
//
//        for (Long uid: context.uids()) {
//            User user = new User(server.config().pgQueue(), new LongPrefix(uid));
//            SearchMapShard shard = server.searchMap().apply(user);
//            shardMap.computeIfAbsent(shard, (k) -> new ArrayList<>()).add(user);
//        }
    }

    private class RequestExecutor {
        private final SpanielSearchContext context;
        private final int batchSize;
        private final MultiFutureCallback<UserResult> mfcb;
        private final AtomicInteger cnt = new AtomicInteger(0);

        public RequestExecutor(
            final FutureCallback<List<UserResult>> callback,
            final SpanielSearchContext context,
            final int batchSize)
        {
            this.context = context;
            this.batchSize = batchSize;

            mfcb = new MultiFutureCallback<>(callback);
        }

        private QueryConstructor buildRequest(final User user)
            throws BadRequestException
        {
            QueryConstructor qc = new QueryConstructor("/search?");
            qc.append("get", "mid,received_date");
            qc.append("length", context.length());
            qc.append("prefix", user.prefix().toStringFast());
            qc.append("service", user.service());
            qc.append("sort", "received_date");
            qc.append("text", context.queryText());

            qc.sb().append("+AND+hid:0");
            return qc;
        }

        private void launchNext() throws BadRequestException {
            int i = cnt.getAndIncrement();
            if (i >= context.uids().size()) {
                return;
            }
            long uid = context.uids().get(i);
            SpanielSingleUserContext userContext =
                new SpanielSingleUserContext(context, uid);
            server.sequentialRequest(
                context.session(),
                userContext,
                new BasicAsyncRequestProducerGenerator(
                    buildRequest(userContext.user()).toString()),
                FAILOVER_DELAY,
                false,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                context.session().listener().createContextGeneratorFor(
                    context.client()),
                new SingleUserCallback(this, mfcb.newCallback(), uid));

            if (i == context.uids().size() - 1) {
                mfcb.done();
            }
        }

        public void start() throws BadRequestException {
            for (int i = 0; i < Math.min(batchSize, context.uids().size()); i++) {
                launchNext();
            }
        }

        public void failed(final Exception e) {
            mfcb.newCallback().failed(e);
            mfcb.done();
        }
    }

    private static class SpanielPrinter
        extends AbstractProxySessionCallback<List<UserResult>>
        implements Comparator<UserResult>
    {
        private final JsonType jsonType;
        private final SpanielSearchContext context;

        public SpanielPrinter(final SpanielSearchContext context) throws BadRequestException {
            super(context.session());

            this.context = context;
            this.jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        }

        @Override
        public void completed(final List<UserResult> userResults) {
            List<UserResult> list = new ArrayList<>(userResults);
            Collections.sort(list, this);

            StringBuilderWriter sbw = new StringBuilderWriter();

            long totalMids = 0;
            long nonEmptyUsers = 0;
            try (JsonWriter writer = jsonType.create(sbw)) {
                writer.startObject();
                writer.key("result");
                writer.startArray();
                for (UserResult ur: list) {
                    writer.value(ur);
                    totalMids += ur.items().size();
                    if (ur.items().size() > 0) {
                        nonEmptyUsers += 1;
                    }
                }
                writer.endArray();
                writer.endObject();

                context.session().connection().setSessionInfo(
                    SearchProxyAccessLoggerConfigDefaults.HITS_COUNT,
                    Long.toString(totalMids));
            } catch (IOException e) {
                failed(e);
                return;
            }

            session.response(
                HttpStatus.SC_OK,
                new NStringEntity(
                    sbw.toString(),
                    ContentType.APPLICATION_JSON
                        .withCharset(context.session().acceptedCharset())));
            context.session().logger().info(
                "Total mids found " + totalMids
                    + " non empty users cnt " + nonEmptyUsers);
            context.session().logger().info(sbw.toString());
        }

        @Override
        public int compare(final UserResult o1, final UserResult o2) {
            return Long.compare(o2.score(), o1.score());
        }
    }

    private static class SingleUserCallback
        extends AbstractFilterFutureCallback<JsonObject, UserResult>
    {
        private final Long uid;
        private final RequestExecutor executor;

        public SingleUserCallback(
            final RequestExecutor executor,
            final FutureCallback<? super UserResult> callback,
            final Long uid)
        {
            super(callback);
            this.uid = uid;
            this.executor = executor;
        }

        @Override
        public void completed(final JsonObject resultObj) {
            try {
                JsonMap map = resultObj.asMap();
                JsonList items = map.getList("hitsArray");
                List<ResultItem> mids = new ArrayList<>(items.size());
                for (JsonObject item: items) {
                    mids.add(new ResultItem(
                        item.asMap().getString("mid"),
                        item.asMap().getLong("received_date")));
                }

                callback.completed(new UserResult(uid, mids));
            } catch (JsonException je) {
                failed(je);
            } finally {
                try {
                    executor.launchNext();
                } catch (BadRequestException bre) {
                    executor.failed(bre);
                }
            }
        }
    }

    private static final class UserResult implements JsonValue {
        private final Long uid;
        private final List<ResultItem> items;
        private final long score;

        public UserResult(final Long uid, final List<ResultItem> items) {
            this.uid = uid;
            this.items = items;
            if (items.size() > 0) {
                score = items.get(0).receivedDate();
            } else {
                score = Long.MIN_VALUE;
            }
        }

        public Long uid() {
            return uid;
        }

        public List<ResultItem> items() {
            return items;
        }

        public long score() {
            return score;
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.startObject();
            writer.key("uid");
            writer.value(uid);
            writer.key("documents");
            writer.value(items);
            writer.endObject();
        }
    }

    private static final class ResultItem implements JsonValue {
        private final String mid;
        private final long receivedDate;

        public ResultItem(final String mid, final long receivedDate) {
            this.mid = mid;
            this.receivedDate = receivedDate;
        }

        public String mid() {
            return mid;
        }

        public long receivedDate() {
            return receivedDate;
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.startObject();
            writer.key("mid");
            writer.value(mid);
            writer.key("received_date");
            writer.value(receivedDate);
            writer.endObject();
        }
    }

    private class SpanielSingleUserContext implements UniversalSearchProxyRequestContext {
        private final SpanielSearchContext context;
        private final User user;

        public SpanielSingleUserContext(final SpanielSearchContext context, final Long uid) {
            this.context = context;
            this.user = new User(server.config().pgQueue(), new LongPrefix(uid));
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return null;
        }

        @Override
        public AbstractAsyncClient<?> client() {
            return context.client();
        }

        @Override
        public Logger logger() {
            return context.session().logger();
        }

        @Override
        public long lagTolerance() {
            return Long.MAX_VALUE;
        }
    }

    private class SpanielSearchContext {
        private final ProxySession session;
        private final String queryText;
        private final List<Long> uids;
        private final int length;
        private final AsyncClient client;

        public SpanielSearchContext(
            final ProxySession session,
            final HttpEntity entity)
            throws HttpException, IOException
        {
            this.session = session;
            this.client = server.searchClient().adjust(session.context());
            if (entity != null) {
                try {
                    JsonMap root = TypesafeValueContentHandler.parse(CharsetUtils.content(entity)).asMap();
                    JsonList userJsonList = root.getList("users");
                    Set<Long> uids = new LinkedHashSet<>(userJsonList.size() << 1);
                    for (JsonObject uidObj: userJsonList) {
                        uids.add(uidObj.asLong());
                    }

                    this.uids = new ArrayList<>(uids);
                } catch (JsonException je) {
                    throw new BadRequestException("Failed to parse body", je);
                }
            } else {
                uids = new ArrayList<>(
                    session.params().get(
                        "uid",
                        new CollectionParser<>(
                            PositiveLongValidator.INSTANCE,
                            LinkedHashSet::new)));
            }

            if (uids.size() <= 0) {
                throw new BadRequestException("No users supplied for request");
            }

            CgiParams params = session.params();
            StringBuilder request = new StringBuilder();
            LuceneQueryContext queryContext = new LuceneQueryContext("hid:0", params, server);
            RewriteRequestRule.checkFlag(request, params, "has_attachments", "has_attachments:1");
            RewriteRequestRule.checkFlag(request, params, "only_attachments", RewriteRequestRule.ATTACH);
            RewriteRequestRule.checkFlag(request, params, "has_links", "x_urls:1");
            try {
                if (!RewriteRequestRule.applyUnixtimeTimerangeFilter(request, queryContext, params, "from", "to")) {
                    RewriteRequestRule.applyUnixtimeTimerangeFilter(request, queryContext, params, "date_from", "date_to");
                }
            } catch (Exception e) {
                throw new BadRequestException("Failed to apply timerange filter", e);
            }

            String requestText = params.getString("request", "");
            RewriteRequestRule.applyScopesOrQueryLanguage(
                request,
                requestText,
                queryContext,
                session,
                server,
                Labels.EMPTY,
                Folders.EMPTY);

            StringBuilder folders = new StringBuilder();
            Set<FolderType> folderTypes = params.getAll(
                "folder",
                Collections.emptySet(),
                DefaultSearchAttributes.FOLDER_TYPE_PARSER,
                new HashSet<>());
            if (!folderTypes.isEmpty()) {
                folders.append(" AND folder_type:(");
                for (FolderType folderType: folderTypes) {
                    folders.append(folderType.fieldValue());
                    folders.append(" OR ");
                }
                folders.setLength(folders.length() - 4);
                folders.append(')');
            } else {
                boolean excludeTrash = params.getBoolean("exclude-trash", true);
                boolean excludeHiddenTrash = params.getBoolean("exclude-hidden-trash", false);
                boolean excludeDraft = params.getBoolean("exclude-draft", true);
                boolean excludeSpam = params.getBoolean("exclude-spam", true);

                if (excludeDraft) {
                    folders.append(" AND NOT folder_type:");
                    folders.append(FolderType.DRAFT.fieldValue());
                }

                if (excludeTrash) {
                    folders.append(" AND NOT folder_type:");
                    folders.append(FolderType.TRASH.fieldValue());
                }

                if (excludeSpam) {
                    folders.append(" AND NOT folder_type:");
                    folders.append(FolderType.SPAM.fieldValue());
                }

                if (excludeHiddenTrash) {
                    folders.append(" AND NOT folder_type:");
                    folders.append(FolderType.HIDDEN_TRASH.fieldValue());
                }
            }



            if (request.length() == 0) {
                request.append(queryContext.selectAll());
            }

            if (folders.length() > 0) {
                request.append(folders);
            }


            //request = session.params().getString("request");
//            SearchRequestText.SearchCollector collector =
//                new SearchRequestText.SearchCollector(
//                    SearchRequestText.DEFAULT_WORDS_MODIFIER,
//                    SearchRequestText.DEFAULT_PHRASES_MODIFIER);
//
//            SearchRequestText.parse(SearchRequestText.normalize(request), collector);
//
//            queryText = new SearchRequestText(
//                request,
//                collector.words(),
//                collector.negations(),
//                collector.phrases(),
//                Collections.emptySet());

            queryText = request.toString();
            length = session.params().get("length", 1, PositiveIntegerValidator.INSTANCE);
        }

        public AsyncClient client() {
            return client;
        }

        public String queryText() {
            return queryText;
        }

        public ProxySession session() {
            return session;
        }

        public List<Long> uids() {
            return uids;
        }

        public int length() {
            return length;
        }
    }

}
