package ru.yandex.msearch.proxy.api.async.mail.subscriptions;

import java.io.IOException;
import java.nio.file.Files;
import java.util.Collections;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.http.HttpException;

import org.joda.time.Chronology;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.chrono.ISOChronology;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.msearch.proxy.AsyncHttpServer;

import ru.yandex.msearch.proxy.api.async.ProxyParams;

import ru.yandex.msearch.proxy.config.ImmutableSubscriptionsConfig;

import ru.yandex.parser.searchmap.User;

import ru.yandex.parser.string.NonNegativeIntegerValidator;

import ru.yandex.parser.uri.CgiParams;

import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.PrefixType;

public class AbstractSubscriptionsPlainRule {
    protected static final DateTimeZone MSK_TIMEZONE =
        DateTimeZone.forID("Europe/Moscow");

    protected final AsyncHttpServer server;
    protected final ImmutableSubscriptionsConfig config;
    protected final Map<String, String> pins;

    public AbstractSubscriptionsPlainRule(
        final AsyncHttpServer server)
        throws IOException
    {
        this.config = server.config().subscriptionsConfig();
        this.server = server;

        if (config.pinnedDisplayNames() != null) {
            this.pins = loadPinnedDisplayNames();
        } else {
            this.pins = Collections.emptyMap();
        }
    }

    protected Map<String, String> loadPinnedDisplayNames() throws IOException {
        return Files.lines(config.pinnedDisplayNames().toPath())
            .map((line) -> line.trim().split("\\s+"))
            .filter((s) -> s.length == 2)
            .collect(Collectors.toMap((s) -> s[0], (s) -> s[1]));
    }

    protected User buildUser(final CgiParams params) throws HttpException {
        String mdb = params.getString(ProxyParams.MDB);
        PrefixType prefixType = server.searchMap().prefixType(mdb);
        Prefix suid;
        Prefix uid;
        Prefix prefix;
        if (mdb.equals(ProxyParams.PG)) {
            suid = params.get(ProxyParams.SUID, null, prefixType);
            uid = params.get(ProxyParams.UID, prefixType);
            prefix = uid;
        } else {
            suid = params.get(ProxyParams.SUID, prefixType);
            uid = params.get(ProxyParams.UID, null, prefixType);
            prefix = suid;
        }

        return new User(server.resolveService(mdb, prefix), prefix);
    }

    protected long  toTs(final CgiParams params)
        throws BadRequestException
    {
        return params.getLong("toTs", Long.MAX_VALUE);
    }

    protected long fromTs(final CgiParams params)
        throws BadRequestException
    {
        DateTimeZone timezone = params.get(
            "tzoffset",
            MSK_TIMEZONE,
            x -> DateTimeZone.forOffsetMillis(Integer.parseInt(x) * 60000));
        long fromTs = params.getLong("fromTs", -1L);
        if (fromTs >= 0) {
            return fromTs;
        }

        int lastDays =
            params.get(
                "last",
                config.minimalMailAge(),
                NonNegativeIntegerValidator.INSTANCE);

        Chronology chronology =
            ISOChronology.getInstance(timezone);
        return new DateTime(chronology).minusDays(lastDays).getMillis() / 1000L;
    }
}
