package ru.yandex.msearch.proxy.api.async.mail.subscriptions;

import java.io.IOException;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import java.util.logging.Level;

import org.apache.http.HttpException;

import ru.yandex.dbfields.SubscriptionsIndexField;

import ru.yandex.http.proxy.AbstractProxySessionCallback;

import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.msearch.proxy.AsyncHttpServer;

import ru.yandex.msearch.proxy.api.async.mail.searcher.ProducerParallelSearcher;
import ru.yandex.parser.email.types.MessageType;
import ru.yandex.parser.email.types.MessageTypeToString;
import ru.yandex.parser.searchmap.User;

import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;

import ru.yandex.search.json.fieldfunction.FieldFunctionException;
import ru.yandex.search.json.fieldfunction.SumMapFunction;

import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;

import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

import ru.yandex.search.rules.RequestParams;
import ru.yandex.search.rules.SearchInfo;
import ru.yandex.search.rules.SearchRequest;
import ru.yandex.search.rules.SearchRule;

public class FastSubscriptionsRule
    extends AbstractSubscriptionsPlainRule
    implements SearchRule<SubscriptionsStat, RequestParams, SearchInfo>
{
    private static final String DATE_FALLBACK = "fallback("
        + SubscriptionsIndexField.SUBS_LAST_RECEIVED_DATE.fieldName() + ','
        + SubscriptionsIndexField.SUBS_RECEIVED_MONTH.fieldName() + " date)";

    private static final CollectionParser<
        MessageType,
        List<MessageType>,
        RuntimeException> TYPES_PARSER = new CollectionParser<>(
        MessageTypeToString.INSTANCE,
        ArrayList::new);

    public FastSubscriptionsRule(
        final AsyncHttpServer server)
        throws IOException
    {
        super(server);
    }

    @Override
    public void execute(
        final SearchRequest<SubscriptionsStat, RequestParams, SearchInfo> request)
        throws HttpException
    {
        CgiParams params = request.cgiParams();
        User user = buildUser(params);

        List<MessageType> typesList =
            params.getAll("types", config.messageTypes(), TYPES_PARSER);
        Set<String> types = new LinkedHashSet<>(typesList.size());
        typesList.forEach((t) -> types.add(String.valueOf(t.typeNumber())));

        QueryConstructor qc = new QueryConstructor("/search?");
        qc.append("prefix", user.prefix().toString());

        long from = fromTs(params);
        long to = toTs(params);

        StringBuilder text = new StringBuilder(
            SubscriptionsIndexField.SUBS_MESSAGE_TYPES.fieldName());
        text.append(":(");
        int i = 0;
        for (String type: types) {
            if (i != 0) {
                text.append(" OR ");
            }

            text.append(type);
            i++;
        }
        text.append(") AND ");
        text.append(
            SubscriptionsIndexField.SUBS_RECEIVED_MONTH_PREFIXED.fieldName());
        text.append(":[");
        text.append(from);
        text.append(" TO ");
        text.append(to);
        text.append(']');
        qc.append("text", text.toString());
        qc.append("service", user.service());
        qc.append("get", "*");
        qc.append("dp", DATE_FALLBACK);
        qc.append("sort", "date");
        qc.append("length", "1000000");
        qc.append("IO_PRIO", 100);

        ProducerParallelSearcher searcher = new ProducerParallelSearcher(
            server, request.session(), user);
        PlainUniversalSearchProxyRequestContext requestContext =
            new PlainUniversalSearchProxyRequestContext(
                user,
                null,
                true,
                server.searchClient(),
                request.session().logger());

        searcher.search(
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            new FastSubscriptionsCallback(request, types));
    }

    private final class FastSubscriptionsCallback
        extends AbstractProxySessionCallback<SearchResult>
    {
        private final Set<String> types;
        private final SearchRequest<SubscriptionsStat, RequestParams, SearchInfo> request;

        public FastSubscriptionsCallback(
            final SearchRequest<SubscriptionsStat, RequestParams, SearchInfo> request,
            final Set<String> types)
        {
            super(request.session());

            this.request = request;
            this.types = types;
        }

        @Override
        public void completed(final SearchResult result) {
            Map<String, Map<String, SubscriptionEntry>> senders =
                new LinkedHashMap<>();
            for (SearchDocument sd: result.hitsArray()) {
                Map<String, String> doc = sd.attrs();
                String monthStr =
                    doc.get(SubscriptionsIndexField.SUBS_RECEIVED_MONTH.fieldName());
                String name =
                    doc.get(SubscriptionsIndexField.SUBS_NAME.fieldName());
                String email =
                    doc.get(SubscriptionsIndexField.SUBS_EMAIL.fieldName());
                String readTypesStr =
                    doc.get(SubscriptionsIndexField.SUBS_READ_TYPES_MAP.fieldName());
                String receivedTypesStr =
                    doc.get(SubscriptionsIndexField.SUBS_RECEIVED_TYPES_MAP.fieldName());

                Map<String, Long> receivedTypes = new LinkedHashMap<>();
                Map<String, Long> readTypes = new LinkedHashMap<>();

                Map<String, SubscriptionEntry> sender =
                    senders.computeIfAbsent(email, (k) -> new LinkedHashMap<>());

                if (email == null || monthStr == null) {
                    session.logger().warning(
                        "Bad subscriptions record " + doc);
                    continue;
                }

                long date;
                try {
                    date = Long.parseLong(monthStr);
                    if (readTypesStr != null) {
                        SumMapFunction.addToMap(readTypes, readTypesStr);
                    }

                    if (receivedTypesStr != null) {
                        SumMapFunction.addToMap(
                            receivedTypes,
                            receivedTypesStr);
                    }
                } catch (FieldFunctionException | NumberFormatException e) {
                    session.logger().log(
                        Level.WARNING,
                        "Bad data stored read: " + readTypesStr
                            + " received: " + receivedTypesStr,
                        e);
                    continue;
                }

                for (String type: types) {
                    Long total = receivedTypes.get(type);
                    if (total == null || total <= 0) {
                        continue;
                    }

                    long read = readTypes.getOrDefault(type, 0L);
                    if (read < 0) {
                        read = 0L;
                    }

                    SubscriptionEntry entry = sender.get(type);
                    if (entry == null) {
                        entry =
                            new SubscriptionEntry(
                                "",
                                email,
                                nameWithPins(name, email),
                                "",
                                date,
                                total,
                                0,
                                read);
                        sender.put(type, entry);
                    } else {
                        entry.incTotal(total);
                        entry.incShows(read);
                        if (entry.displayName() == null
                            || entry.displayName().isEmpty())
                        {
                            entry.displayName(nameWithPins(name, email));
                        }
                    }
                }
            }

            SubscriptionsStat stat =
                new SubscriptionsStat(result.hitsArray().size());
            senders.forEach(
                (k, v) -> stat.add(
                    new SubscriptionsSender(v, 0)));

            request.callback().completed(stat);
        }

        protected String nameWithPins(final String nameP, final String email) {
            String name = nameP;
            int atIndex = email.indexOf('@');
            String domain = email;
            if (atIndex >= 0) {
                domain =
                    email.substring(atIndex + 1).toLowerCase(Locale.ROOT);
                if (name == null) {
                    name = email.substring(0, atIndex);
                }
            } else if (name == null) {
                name = email;
            }

            return pins.getOrDefault(domain, name);
        }
    }
}
