package ru.yandex.msearch.proxy.api.async.mail.subscriptions;

import java.io.IOException;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import java.util.stream.Collectors;

import org.apache.http.HttpException;

import org.apache.http.client.config.RequestConfig;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.AbstractFilterFutureCallback;

import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;

import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.HttpClientContextGenerator;
import ru.yandex.msearch.proxy.AsyncHttpServer;

import ru.yandex.msearch.proxy.api.async.suggest.contact.ContactParser;
import ru.yandex.msearch.proxy.api.async.suggest.contact.ContactParser.Email;
import ru.yandex.msearch.proxy.api.async.suggest.contact.ContactSuggestBuilder;

import ru.yandex.parser.email.types.MessageType;
import ru.yandex.parser.email.types.MessageTypeToString;

import ru.yandex.parser.searchmap.User;

import ru.yandex.parser.string.CollectionParser;

import ru.yandex.parser.string.EnumParser;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;

import ru.yandex.search.document.mail.FolderType;
import ru.yandex.search.proxy.SearchProxy;
import ru.yandex.search.proxy.SearchResultConsumerFactory;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

import ru.yandex.search.rules.RequestParams;
import ru.yandex.search.rules.SearchInfo;
import ru.yandex.search.rules.SearchRequest;
import ru.yandex.search.rules.SearchRule;
import ru.yandex.util.string.StringUtils;

public class FullScanSubscriptionsRule
    extends AbstractSubscriptionsPlainRule
    implements SearchRule<SubscriptionsStat, RequestParams, SearchInfo>
{
    private static final CollectionParser<
        MessageType,
        List<MessageType>,
        RuntimeException> TYPES_PARSER = new CollectionParser<>(
            MessageTypeToString.INSTANCE,
            ArrayList::new);

    private final HttpClientContextGenerator httpClientContextGenerator;

    public FullScanSubscriptionsRule(
        final AsyncHttpServer server)
        throws IOException
    {
        super(server);

        RequestConfig requestConfig = AsyncClient.createRequestConfig(config);

        httpClientContextGenerator =
            new HttpClientContextGenerator(requestConfig);
    }

    @Override
    public void execute(
        final SearchRequest<SubscriptionsStat, RequestParams, SearchInfo> request)
        throws HttpException
    {
        CgiParams params = request.cgiParams();
        User user = buildUser(params);

        List<MessageType> types =
            params.get("types", config.messageTypes(), TYPES_PARSER);

        QueryConstructor qc = new QueryConstructor("/search?");
        qc.append("prefix", user.prefix().toString());

        long from = fromTs(params);
        long to = toTs(params);

        StringBuilder text = new StringBuilder("hid:0");
        text.append(" AND received_date:[");
        text.append(from);
        text.append(" TO ");
        text.append(to);
        text.append(']');

        Collection<String> folderTypes =
            params.getOrDefault("folder-type", Collections.emptyList());

        if (!folderTypes.isEmpty()) {
            text.append(" AND folder_type:(");
            text.append(StringUtils.join(folderTypes, " OR "));
            text.append(')');
        } else {
            text.append(" AND folder_type:inbox");
        }

        StringBuilder typesJoined = null;
        if (types.size() > 0) {
            typesJoined = new StringBuilder("multi(hdr_from_normalized,");
            text.append(" AND (");

            for (int i =0; i < types.size(); i++) {
                if (i != 0) {
                    text.append(" OR ");
                    typesJoined.append(',');
                }

                text.append("message_type:");
                String type = types.get(i).toString();
                text.append(type);
                typesJoined.append(type);
            }

            text.append(')');
            typesJoined.append(')');

            //if (params.getBoolean("excludeSpam", true)) {
            //    text.append(" AND NOT folder_type:spam");
            //}

        }

        qc.append("text", text.toString());

        if (typesJoined == null) {
            qc.append("group", "hdr_from_normalized");
        } else {
            qc.append("group", new String(typesJoined));
        }

        qc.append(
            "aggregate",
            "sum(seen) seen_cnt,sum(total) total_cnt,sum(shows_bool) shows_sum");

        qc.append("service", user.service());
        //qc.append("get", "**");
        StringBuilder getSb =
            new StringBuilder(
                "mid,hdr_from,hdr_from_normalized,message_type,received_date,"
                    + "total_cnt,seen_cnt,shows_sum");

        qc.append("length", "1000000");
        qc.append("merge_func", "count");
        qc.append("dp","contains(lids,FAKE_SEEN_LBL seen)");
        qc.append("dp","const(1 total)");
        qc.append("dp", "const(0 zero_field)");
        qc.append("dp", "const(1 one)");
        qc.append("dp", "fallback(clicks_total_count,zero_field shows)");
        qc.append("dp", "regex_contains(shows,^0$ shows_bool_inv)");
        qc.append("dp", "sub(one,shows_bool_inv shows_bool)");

        for (MessageType type: types) {
            getSb.append(',');
            getSb.append(type);
            qc.append(
                "dp",
                "contains(message_type,"
                    + type.typeNumber() + ' ' + type + ' ' + type + ')');
        }

        qc.append("get", getSb.toString());

        qc.append("sort", "received_date");

        long minTotal =
            params.getInt("minTotal", config.minMailsInSubcscriptions());
        SubscriptionsProxyContext proxyContext =
            new SubscriptionsProxyContext(request.session(), server, user);
        server.searchClient().execute(
            SearchProxy.shuffle(server.searchMap().searchHosts(user), user),
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            System.currentTimeMillis() + 2 * config.timeout(),
            config.timeout() / 3,
            SearchResultConsumerFactory.OK,
            request.session().listener().adjustContextGenerator(
                httpClientContextGenerator),
            new SubscriptionsCallback(request.callback(), minTotal, types));
    }

    private SubscriptionEntry createEntry(
        final String mid,
        final String from,
        final String displayName,
        final long date)
    {
        String trimmedEmail = from.trim();
        String normalized = this.normalizeEmail(trimmedEmail);
        String searchRequest;
        String fromNormalized;
        String displayNameNormalized;
        if (normalized != trimmedEmail) {
            int atIndex = normalized.indexOf('@');
            searchRequest =
                StringUtils.concat(
                    normalized.substring(0, atIndex),
                    "*",
                    normalized.substring(atIndex));
            fromNormalized = normalized;
            displayNameNormalized = normalized.substring(0, atIndex);
        } else {
            searchRequest = trimmedEmail;
            displayNameNormalized = displayName;
            fromNormalized = trimmedEmail;
        }

        int atIndex = fromNormalized.indexOf('@');
        String domain = fromNormalized;
        if (atIndex >= 0) {
            domain =
                fromNormalized.substring(atIndex + 1).toLowerCase(Locale.ROOT);
        }

        displayNameNormalized =
            pins.getOrDefault(domain, displayNameNormalized);
        return new SubscriptionEntry(
            mid,
            fromNormalized,
            displayNameNormalized,
            searchRequest,
            date,
            0,
            0,
            0);
    }

    private String normalizeEmail(final String email) {
        if (email == null) {
            return email;
        }

        int atIndex = email.indexOf('@');
        if (atIndex <= 0) {
            return email;
        }

        String normalized = email;
        int plusIndex = email.indexOf('+');
        if (plusIndex > 0 && plusIndex < atIndex) {
            normalized =
                email.substring(0, plusIndex) + email.substring(atIndex);
        }

        return normalized;
    }

    private class SubscriptionsCallback
        extends AbstractFilterFutureCallback<SearchResult, SubscriptionsStat>
    {
        private final List<MessageType> types;
        private final long minimumTotal;

        public SubscriptionsCallback(
            final FutureCallback<? super SubscriptionsStat> callback,
            final long minTotal,
            final List<MessageType> types)
        {
            super(callback);

            this.types = types;
            this.minimumTotal = minTotal;
        }

        @Override
        public void completed(final SearchResult result) {
            Map<String, Map<String, SubscriptionEntry>> senders =
                new LinkedHashMap<>();

            for (SearchDocument sd: result.hitsArray()) {
                Map<String, String> attrs = sd.attrs();
                String from = attrs.get("hdr_from_normalized");

                Email email =
                    ContactParser.parse(sd, "hdr_from", Collections.emptySet());
                String displayName;
                if (email != null) {
                    displayName =
                        ContactSuggestBuilder.formName(
                            email.name(),
                            email.address());
                } else {
                    displayName = from;
                }

                long receiveDate = Long.parseLong(attrs.get("received_date"));
                long total = Long.parseLong(attrs.get("total_cnt"));
                long seen = Long.parseLong(attrs.get("seen_cnt"));
                long shows = Long.parseLong(attrs.get("shows_sum"));
                String mid = attrs.get("mid");

                Set<MessageType> itemTypes =
                    types.stream()
                        .filter(
                            t -> Integer.parseInt(attrs.get(t.toString())) == 1)
                        .collect(Collectors.toSet());

                SubscriptionEntry current =
                    createEntry(mid, from, displayName, receiveDate);

                Map<String, SubscriptionEntry> sender =
                    senders.get(current.from());

                if (sender == null) {
                    sender = new HashMap<>();
                    senders.put(current.from(), sender);
                }

                for (MessageType type: itemTypes) {
                    SubscriptionEntry entry =
                        sender.get(String.valueOf(type.typeNumber()));
                    if (entry == null) {
                        entry =
                            createEntry(mid, from, displayName, receiveDate);

                        sender.put(String.valueOf(type.typeNumber()), entry);
                    } else {
                        if (entry.searchEmail().length()
                            < current.searchEmail().length())
                        {
                            SubscriptionEntry hashedEntry =
                                createEntry(mid, from, displayName, receiveDate);

                            // if we have notification@facebook.com
                            // and notification*@facebook.com
                            // we should choose most wide
                            hashedEntry.incSeenFlag(entry.seenFlag());
                            hashedEntry.incShows(entry.shows());
                            hashedEntry.incTotal(entry.total());
                            sender.put(
                                String.valueOf(type.typeNumber()),
                                hashedEntry);
                            entry = hashedEntry;
                        }
                    }

                    entry.incSeenFlag(seen);
                    entry.incTotal(total);
                    entry.incShows(shows);
                }
            }

            SubscriptionsStat stat =
                new SubscriptionsStat(result.hitsArray().size());
            senders.forEach(
                (k, v) -> stat.add(
                    new SubscriptionsSender(v, minimumTotal)));

            callback.completed(stat);
        }
    }
}
