package ru.yandex.msearch.proxy.api.async.mail.subscriptions;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.TimeUnit;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.mail.search.mail.MailSearchDatabases;
import ru.yandex.mail.search.subscriptions.SubscriptionsConstants;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.mail.search.SubscriptionsFields;

public class SubscriptionsCountHandler implements ProxyRequestHandler {
    private static final int OPTIN_FRESH_MONTHES = 1;
    private static final long ONE_MONTH = TimeUnit.DAYS.toSeconds(OPTIN_FRESH_MONTHES);
    private final AsyncHttpServer server;

    public SubscriptionsCountHandler(final AsyncHttpServer server) {
        this.server = server;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        SubscriptionsBaseContext context =
            new SubscriptionsBaseContext(server, session);

        QueryConstructor qc = new QueryConstructor("/search?");
        qc.append("prefix", context.user().prefix().toString());
        qc.append("service", context.user().service());
        qc.append("get", "url");
        qc.append("length", "1");
        qc.append("collector", "passthrough(1)");
        qc.append("IO_PRIO", 100);
        qc.append("db", MailSearchDatabases.SUBSCRIPTIONS.dbName());

        MultiFutureCallback<Counter> mfcb =
            new MultiFutureCallback<>(
                new SubscriptionsCountPrinter(context));

        StringBuilder hiddenText = new StringBuilder();
        hiddenText.append(SubscriptionsFields.SUBS_HIDDEN_TYPES.prefixed());
        hiddenText.append(":*");

        int length = qc.sb().length();
        qc.append("text", hiddenText.toString());

        server.sequentialRequest(
            session,
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            100L,
            true,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().adjustContextGenerator(
                context.client().httpClientContextGenerator()),
            new CountCallback(SubscriptionStatus.HIDDEN, mfcb.newCallback()));

        if (context.optIn()) {
            StringBuilder activeText = new StringBuilder();
            activeText.append(SubscriptionsFields.SUBS_OPTIN_ACTIVE_TYPES.prefixed());
            activeText.append(":*");

            qc.sb().setLength(length);
            qc.append("text", activeText.toString());

            server.sequentialRequest(
                session,
                context,
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                100L,
                true,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.listener().adjustContextGenerator(
                    context.client().httpClientContextGenerator()),
                new CountCallback(SubscriptionStatus.ACTIVE, mfcb.newCallback()));

            StringBuilder pendingText = new StringBuilder();
            pendingText.append("(");
            pendingText.append(SubscriptionsFields.SUBS_MESSAGE_TYPES.prefixed());
            pendingText.append(":(");
            int i = 0;
            for (String type: SubscriptionsConstants.OPT_IN_TYPES) {
                if (i != 0) {
                    pendingText.append(" OR ");
                }

                pendingText.append(type);
                i++;
            }
            pendingText.append(")");

            long currentMonth = System.currentTimeMillis() / 1000;
            long startMonth = session.params().getLong("opt_in_subs_at", currentMonth) - 2 * ONE_MONTH;

            pendingText.append(" AND subs_received_month_p:(");
            // +1 month due to search optimization
            for (long month = currentMonth; month >= startMonth; month -= ONE_MONTH) {
                pendingText.append(month);
                pendingText.append(' ');
            }

            pendingText.setLength(pendingText.length() - 1);
            pendingText.append("))");
            pendingText.append(" AND NOT ");
            pendingText.append(SubscriptionsFields.SUBS_HIDDEN_TYPES.prefixed());
            pendingText.append(":*");
            pendingText.append(" AND NOT ");
            pendingText.append(SubscriptionsFields.SUBS_OPTIN_ACTIVE_TYPES.prefixed());
            pendingText.append(":*");

            qc.sb().setLength(length);
            qc.append("text", pendingText.toString());
            long delta = context.optInTurnTime() - SubscriptionsListContext.OPTIN_FRESH_TIME;
            qc.append("postfilter", "subs_received_date >= " + delta);
            qc.append("dp", "contains(subs_email,@ has_at)");
            qc.append("dp", "contains(subs_email,  has_space)");
            qc.append("postfilter", "has_at == 1");
            qc.append("postfilter", "has_space != 1");

            server.sequentialRequest(
                session,
                context,
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                100L,
                true,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.listener().adjustContextGenerator(
                    context.client().httpClientContextGenerator()),
                new CountCallback(SubscriptionStatus.PENDING, mfcb.newCallback()));
        } else {
            StringBuilder activeText = new StringBuilder();
            activeText.append(SubscriptionsFields.SUBS_MESSAGE_TYPES.prefixed());
            activeText.append(":(");
            int i = 0;
            for (String type: SubscriptionsConstants.TYPES) {
                if (i != 0) {
                    activeText.append(" OR ");
                }

                activeText.append(type);
                i++;
            }
            activeText.append(")");

            activeText.append(" AND NOT ");
            activeText.append(SubscriptionsFields.SUBS_HIDDEN_TYPES.prefixed());
            activeText.append(":*");

            qc.sb().setLength(length);
            qc.append("text", activeText.toString());

            server.sequentialRequest(
                session,
                context,
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                100L,
                true,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.listener().adjustContextGenerator(
                    context.client().httpClientContextGenerator()),
                new CountCallback(SubscriptionStatus.ACTIVE, mfcb.newCallback()));
        }

        mfcb.done();
    }

    private static class CountCallback extends AbstractFilterFutureCallback<JsonObject, Counter> {
        private final SubscriptionStatus status;
        public CountCallback(
            final SubscriptionStatus status,
            final FutureCallback<? super Counter> callback)
        {
            super(callback);
            this.status = status;
        }

        @Override
        public void completed(final JsonObject resultObject) {
            try {
                callback.completed(
                    new Counter(
                        status,
                        resultObject.asMap().getInt("hitsCount")));
            } catch (JsonException e) {
                failed(e);
            }
        }
    }

    private static class SubscriptionsCountPrinter extends AbstractProxySessionCallback<List<Counter>> {
        private final SubscriptionsBaseContext context;

        public SubscriptionsCountPrinter(final SubscriptionsBaseContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final List<Counter> counters) {
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = context.jsonType().create(sbw)) {
                writer.startObject();
                for (Counter counter: counters) {
                    writer.value(counter);
                }
                writer.endObject();
                session.response(
                    HttpStatus.SC_OK,
                    new NStringEntity(
                        sbw.toString(),
                        ContentType.APPLICATION_JSON
                            .withCharset(context.session().acceptedCharset())));
            } catch (IOException ioe) {

            }
        }
    }

    private static class Counter implements JsonValue {
        private final SubscriptionStatus status;
        private final int counter;

        public Counter(final SubscriptionStatus status, final int counter) {
            this.status = status;
            this.counter = counter;
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.key(status.value());
            writer.value(counter);
        }
    }

}
