package ru.yandex.msearch.proxy.api.async.mail.subscriptions;

import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import org.apache.commons.validator.routines.EmailValidator;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.mail.search.mail.MailSearchDatabases;
import ru.yandex.mail.search.subscriptions.SubscriptionsConstants;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.mail.search.SubscriptionsFields;
import ru.yandex.search.json.fieldfunction.SumMapFunction;
import ru.yandex.util.string.StringUtils;

public class SubscriptionsListHandler implements ProxyRequestHandler {
    public static final long ONE_MONTH = TimeUnit.DAYS.toSeconds(1);
    private final AsyncHttpServer proxy;

    public SubscriptionsListHandler(final AsyncHttpServer proxy) {
        this.proxy = proxy;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        SubscriptionsListContext context =
            new SubscriptionsListContext(proxy, session);

        QueryConstructor qc = new QueryConstructor("/search?");
        qc.append("prefix", context.user().prefix().toString());

        StringBuilder text = new StringBuilder();
        if (context.statuses().contains(SubscriptionStatus.HIDDEN)) {
            text.append(SubscriptionsFields.SUBS_HIDDEN_TYPES.prefixed());
            text.append(":*");
        }

        Set<String> types = SubscriptionsConstants.TYPES;
        if (context.optIn()) {
            types = SubscriptionsConstants.OPT_IN_TYPES;
        }

        if (context.optIn() && context.statuses().contains(SubscriptionStatus.PENDING)) {
            if (text.length() > 0) {
                text.append(" OR ");
            }

            text.append(SubscriptionsFields.SUBS_OPTIN_ACTIVE_TYPES.prefixed());
            text.append(":(");
            int i = 0;
            for (String type: types) {
                if (i != 0) {
                    text.append(" OR ");
                }

                text.append(type);
                i++;
            }
            text.append(")");
        }

        if (context.statuses().contains(SubscriptionStatus.ACTIVE)
            || (context.optIn() && context.statuses().contains(SubscriptionStatus.PENDING)))
        {
            if (text.length() > 0) {
                text.append(" OR ");
            }
            text.append('(');
            text.append(SubscriptionsFields.SUBS_MESSAGE_TYPES.prefixed());
            text.append(":(");
            int i = 0;
            for (String type: types) {
                if (i != 0) {
                    text.append(" OR ");
                }

                text.append(type);
                i++;
            }
            text.append(")");

            if (context.optIn()) {

                text.append(" AND subs_received_month_p:(");
                // +1 month due to search optimization
                long start = context.optInTurnTime() - SubscriptionsListContext.OPTIN_FRESH_TIME - ONE_MONTH;
                for (long month = context.currentTime(); month >= start; month -= ONE_MONTH) {
                    text.append(month);
                    text.append(' ');
                }

                text.setLength(text.length() - 1);
                text.append(')');
            }

            text.append(')');

            if (context.statuses().size() == 1) {
                text.append(" AND NOT ");
                text.append(SubscriptionsFields.SUBS_HIDDEN_TYPES.prefixed());
                text.append(":*");
            }
        }

        qc.append("text", text.toString());
        qc.append("service", context.user().service());
        qc.append("get", "*");
        qc.append("sort", SubscriptionsFields.SUBS_RECEIVED_DATE.stored());
        if (context.length() > 0) {
            qc.append("length", context.length());
        }

        if (context.offset() > 0) {
            qc.append("offset", context.offset());
        }

        qc.append("IO_PRIO", 100);
        qc.append("db", MailSearchDatabases.SUBSCRIPTIONS.dbName());

        proxy.sequentialRequest(
            session,
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            100L,
            true,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().adjustContextGenerator(
                context.client().httpClientContextGenerator()),
            new SubscriptionsCallback(context));
    }

    private final class SubscriptionsCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final SubscriptionsListContext context;

        public SubscriptionsCallback(
            final SubscriptionsListContext context)
        {
            super(context.session());

            this.context = context;
        }

        @Override
        public void completed(final JsonObject resultObj) {
            try {
                JsonMap result = resultObj.asMap();
                JsonList hits = result.getList("hitsArray");
                StringBuilderWriter sbw = new StringBuilderWriter();
                JsonWriter writer = context.jsonType().create(sbw);
                Map<SubscriptionStatus, List<JsonMap>> statuses = new LinkedHashMap<>();

                for (SubscriptionStatus status: context.statuses()) {
                    statuses.put(status, new ArrayList<>(hits.size()));
                }
                for (JsonObject docObj: hits) {
                    JsonMap doc = docObj.asMap();
                    String hidden = doc.getString(SubscriptionsFields.SUBS_HIDDEN_TYPES.stored(), "");
                    boolean isHidden = hidden != null && !hidden.isEmpty();

                    if (context.optIn()) {
                        String optinActive = doc.getString(SubscriptionsFields.SUBS_OPTIN_ACTIVE_TYPES.stored(), "");
                        boolean isOptinActive = optinActive != null && !optinActive.isEmpty();
//                        boolean partialOptinActive = false;
//                        if (isOptinActive) {
//                            Set<String> activeTypes = doc.get(
//                                SubscriptionsFields.SUBS_OPTIN_ACTIVE_TYPES.stored(),
//                                Collections.emptySet(),
//                                SubscriptionsConstants.BACKEND_RESP_TYPES_PARSER);
//                            partialOptinActive = !activeTypes.containsAll(SubscriptionsConstants.OPT_IN_TYPES);
//                            if (partialOptinActive) {
//                                statuses.computeIfAbsent(
//                                    SubscriptionStatus.PENDING,
//                                    (x) -> new ArrayList<>(hits.size()))
//                                    .add(doc);
//                            }
//                        }

                        if (!isHidden && !isOptinActive) {
                            long delta = context.optInTurnTime() - doc.getLong(SubscriptionsFields.SUBS_RECEIVED_DATE.stored(), 0L);
                            if (delta > SubscriptionsListContext.OPTIN_FRESH_TIME) {
                                continue;
                            }
                            statuses.computeIfAbsent(
                                SubscriptionStatus.PENDING,
                                (x) -> new ArrayList<>(hits.size()))
                                .add(doc);
                        } else if (isHidden) {
                            statuses.computeIfAbsent(
                                SubscriptionStatus.HIDDEN,
                                (x) -> new ArrayList<>(hits.size()))
                                .add(doc);
                        } else {
                            statuses.computeIfAbsent(
                                SubscriptionStatus.ACTIVE,
                                (x) -> new ArrayList<>(hits.size()))
                                .add(doc);
                        }
                    } else {
                        if (!isHidden) {
                            statuses.computeIfAbsent(
                                SubscriptionStatus.ACTIVE,
                                (x) -> new ArrayList<>(hits.size()))
                                .add(doc);
                        } else {
                            statuses.computeIfAbsent(
                                SubscriptionStatus.HIDDEN,
                                (x) -> new ArrayList<>(hits.size()))
                                .add(doc);
                        }
                    }
                }

                writer.startObject();
                EmailValidator validator = EmailValidator.getInstance();
                for (Map.Entry<SubscriptionStatus, List<JsonMap>> entry: statuses.entrySet()) {
                    writer.key(entry.getKey().value());
                    writer.startArray();
                    for (JsonMap doc: entry.getValue()) {
                        String email = doc.getString(SubscriptionsFields.SUBS_EMAIL.stored());

                        if (!validator.isValid(email)) {
                            session.logger().warning("Invalid email " + email);
                            continue;
                        }
//                        try {
//                            InternetAddress emailAddr = new InternetAddress(email);
//                            emailAddr.validate();
//                        } catch (AddressException ex) {
//                            session.logger().warning("Invalid email " + email);
//                            continue;
//                        }

                        writer.startObject();
                        writer.key("email");
                        writer.value(email);
                        String receivedTypes = doc.getString(SubscriptionsFields.SUBS_RECEIVED_TYPES.stored(), "");
                        String readTypes = doc.getString(SubscriptionsFields.SUBS_READ_TYPES.stored(), "");
                        Map<String, Long> receivedTypesMap = new LinkedHashMap<>();
                        SumMapFunction.parseMap(receivedTypesMap, receivedTypes);
                        Map<String, Long> readTypesMap = new LinkedHashMap<>();
                        SumMapFunction.parseMap(readTypesMap, readTypes);
                        writer.key("types");
                        writer.value(StringUtils.join(receivedTypesMap.keySet(), ','));
                        long received = 0;
                        for (Map.Entry<String, Long> rcEntry: receivedTypesMap.entrySet()) {
                            received += rcEntry.getValue();
                        }

                        long read = 0;
                        for (Map.Entry<String, Long> rcEntry: readTypesMap.entrySet()) {
                            read += rcEntry.getValue();
                        }

                        double readFrequency = 0.0;
                        if (received > 0) {
                            readFrequency = (double) read / received;
                        }

                        writer.key("readFrequency");
                        writer.value(readFrequency);
                        writer.key("displayName");
                        String names = doc.getString(SubscriptionsFields.SUBS_NAMES.stored(), "");
                        int sep = names.indexOf('\n');
                        if (sep > 0) {
                            names= names.substring(0, sep);
                        }

                        if (names.isEmpty()) {
                            names = email;
                        }
                        writer.value(names);
                        writer.endObject();
                    }
                    writer.endArray();
                }
                writer.endObject();

                session.response(
                    HttpStatus.SC_OK,
                    new NStringEntity(
                        sbw.toString(),
                        ContentType.APPLICATION_JSON
                            .withCharset(context.session().acceptedCharset())));
            } catch (JsonException | IOException e) {
                failed(e);
            }
        }
    }

}
