package ru.yandex.msearch.proxy.api.async.mail.subscriptions;

import java.io.IOException;

import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;

import org.apache.http.entity.ContentType;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import ru.yandex.client.producer.ProducerClient;
import ru.yandex.dbfields.SubscriptionsIndexField;
import ru.yandex.http.proxy.AbstractProxySessionCallback;

import ru.yandex.http.util.AbstractFilterFutureCallback;

import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;

import ru.yandex.io.StringBuilderWriter;

import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.ProxyParams;

import ru.yandex.parser.searchmap.User;

import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;

import ru.yandex.search.json.fieldfunction.SumMapFunction;

import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.PrefixType;

import ru.yandex.search.rules.RequestParams;
import ru.yandex.search.rules.SearchInfo;
import ru.yandex.search.rules.SearchRequest;
import ru.yandex.search.rules.SearchRule;

public class SubscriptionsReindexer
    implements SearchRule<SubscriptionsStat, RequestParams, SearchInfo>
{
    protected static final DateTimeZone MSK_TIMEZONE =
        DateTimeZone.forID("Europe/Moscow");

    private final AsyncHttpServer server;
    private final FullScanSubscriptionsRule next;

    public SubscriptionsReindexer(
        final AsyncHttpServer server,
        final FullScanSubscriptionsRule next)
    {
        this.server = server;
        this.next = next;
    }

    @Override
    public void execute(
        final SearchRequest<SubscriptionsStat, RequestParams, SearchInfo> request)
        throws HttpException
    {
        PrefixedLogger logger = request.session().logger();
        long startMs =
            request.cgiParams().getLong("startTs", System.currentTimeMillis());
        int months = request.cgiParams().getInt("months", 12);
        DateTime current = new DateTime(startMs, MSK_TIMEZONE);
        DateTime startTs =
            new DateTime(
                current.year().get(),
                current.monthOfYear().get(),
                1,
                0,
                0,
                MSK_TIMEZONE);
        DateTime endTs =
            new DateTime(
                current.year().get(),
                current.monthOfYear().get(),
                1,
                0,
                0,
                MSK_TIMEZONE);
        endTs = endTs.plusMonths(1);
        endTs = endTs.minusMillis(1);

        User user = buildUser(request.cgiParams());
        reindex(
            new SubsReindexContext(request, user, current.minusMonths(months)),
            startTs,
            endTs);
    }

    private User buildUser(final CgiParams params) throws HttpException {
        String mdb = params.getString(ProxyParams.MDB);
        PrefixType prefixType = server.searchMap().prefixType(mdb);
        Prefix suid;
        Prefix uid;
        Prefix prefix;
        if (mdb.equals(ProxyParams.PG)) {
            suid = params.get(ProxyParams.SUID, null, prefixType);
            uid = params.get(ProxyParams.UID, prefixType);
            prefix = uid;
        } else {
            suid = params.get(ProxyParams.SUID, prefixType);

            prefix = suid;
        }

        return new User(server.resolveService(mdb, prefix), prefix);
    }

    private void reindex(
        final SubsReindexContext context,
        final DateTime start,
        final DateTime end)
        throws HttpException
    {
        if (start.getMillis() <= context.limit().getMillis()) {
            context.request().callback().completed(null);
            return;
        }

        CgiParams params = new CgiParams(context.request().cgiParams());
        params.add("toTs", String.valueOf(end.getMillis() / 1000));
        params.add("fromTs", String.valueOf(start.getMillis() / 1000));

        context.request().session().logger().info(
            "Gathering data for period [" + start.getMillis()
                + ',' + end.getMillis() + ']');
        next.execute(
            context.request().withCgiParams(params).withCallback(
                new MonthReindexCallback(context, start, end)));
    }

    private final class MonthReindexCallback
        extends AbstractProxySessionCallback<SubscriptionsStat>
    {
        private final DateTime startTs;
        private final DateTime endTs;
        private final SubsReindexContext context;
        private final long startMonth;

        public MonthReindexCallback(
            final SubsReindexContext context,
            final DateTime startTs,
            final DateTime endTs)
        {
            super(context.request().session());

            this.context = context;
            this.startTs = startTs;
            this.endTs = endTs;
            this.startMonth = startTs.getMillis() / 1000;
        }

        @Override
        public void completed(final SubscriptionsStat stat) {
            session.logger().info(
                "Gathering data completed for [" + startTs.getMillis()
                    + ',' + endTs.getMillis() + ']');

            final ProducerClient producerClient =
                server.producerClient().adjust(
                    context.request().session().context());
            String cleanupUri =
                context.deleteUri()
                    + "&text=url:subs_"
                    + context.user().prefix()
                    + "_*+AND+subs_received_month:" + startMonth;

            List<SubscriptionsSender> senders = stat.sort();
            Map<String, SubsIndexItem> items = new LinkedHashMap<>();

            for (SubscriptionsSender sender: senders) {
                for (Map.Entry<String, SubscriptionEntry> type
                    : sender.types().entrySet())
                {
                    items.computeIfAbsent(
                        type.getValue().from(), SubsIndexItem::new)
                        .add(
                            type.getKey(),
                            Collections.singleton(type.getValue().displayName()),
                            type.getValue().shows(),
                            type.getValue().total());
                }
            }

            ProducerCallback callback =
                new ProducerCallback(context, startTs, endTs);
            if (items.size() > 0) {
                StringBuilderWriter sbWriter = new StringBuilderWriter();
                try (JsonWriterBase writer =
                        JsonType.NORMAL.create(sbWriter))
                {
                    writer.startObject();
                    writer.key("prefix");
                    writer.value(context.prefix());
                    writer.key("AddIfNotExists");
                    writer.value(true);
                    writer.key("docs");
                    writer.startArray();
                    for (SubsIndexItem item: items.values()) {
                        append(writer, item);
                    }
                    writer.endArray();
                    writer.endObject();
                } catch (IOException ioe) {
                    failed(ioe);
                    return;
                }

                String updateUri =
                    context.producerUri() + "&subsMonth="
                        + startTs.monthOfYear().getAsString()
                        + '_' + startTs.yearOfEra().getAsString();

                session.logger().info(
                    "Sending cleaning up request " + cleanupUri
                        + " for "
                        + startTs.getMillis()
                        + ',' + endTs.getMillis() + ']');
                producerClient.execute(
                    server.config().producerClientConfig().host(),
                    new BasicAsyncRequestProducerGenerator(cleanupUri),
                    EmptyAsyncConsumerFactory.ANY_GOOD,
                    new AbstractFilterFutureCallback<Object, Object>(callback) {
                        @Override
                        public void completed(final Object o) {
                            session.logger().info(
                                "Reindex update batch sent for ["
                                    + startTs.getMillis()
                                    + ',' + endTs.getMillis() + ']');
                            producerClient.execute(
                                server.config().producerClientConfig().host(),
                                new BasicAsyncRequestProducerGenerator(
                                    updateUri,
                                    sbWriter.toString(),
                                    ContentType.APPLICATION_JSON),
                                AsyncStringConsumerFactory.ANY_GOOD,
                                callback);
                        }
                    }
                );
            } else {
                context.request().session().logger().warning(
                    "For month " + startTs.monthOfYear().getAsShortText()
                        + " no data, cleaning up");
                producerClient.execute(
                    server.config().producerClientConfig().host(),
                    new BasicAsyncRequestProducerGenerator(cleanupUri),
                    EmptyAsyncConsumerFactory.ANY_GOOD,
                    callback);
            }
        }

        private void append(
            final JsonWriterBase writer,
            final SubsIndexItem item)
            throws IOException
        {
            writer.startObject();
            writer.key("url");
            writer.value(
                "subs_" + context.prefix()
                    + '_' + item.from + '_' + startMonth);
            writer.key("subs_received_month");
            writer.value(startMonth);
            writer.value(item);
            writer.endObject();
        }
    }

    private final class ProducerCallback
        extends AbstractFilterFutureCallback<Object, SubscriptionsStat>
    {
        private final SubsReindexContext context;
        private final DateTime startTs;
        private final DateTime endTs;

        public ProducerCallback(
            final SubsReindexContext context,
            final DateTime startTs,
            final DateTime endTs)
        {
            super(context.request().callback());

            this.context = context;
            this.startTs = startTs;
            this.endTs = endTs;
        }

        @Override
        public void completed(final Object o) {
            context.request().session().logger().info(
                "Batch for " + startTs.getMillis()
                    + " completed " + String.valueOf(o));
            try {
                reindex(
                    context,
                    startTs.minusMonths(1),
                    endTs.minusMonths(1));
            } catch (HttpException e) {
                failed(e);
            }
        }
    }

    private static final class SubsIndexItem implements JsonValue {
        private final String from;
        private final LinkedHashSet<String> names;
        private final Map<String, Long> total;
        private final Map<String, Long> read;

        public SubsIndexItem(final String from) {
            this.from = from;
            this.total = new LinkedHashMap<>();
            this.read = new LinkedHashMap<>();
            this.names = new LinkedHashSet<>();
        }

        public void add(
            final String type,
            final Set<String> names,
            final long read,
            final long total)
        {
            this.total.put(type, total + this.total.getOrDefault(type, 0L));
            this.names.addAll(names);
            if (read > 0) {
                this.read.put(type, read + this.read.getOrDefault(type, 0L));
            }
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.key(SubscriptionsIndexField.SUBS_EMAIL.fieldName());
            writer.value(from);
            writer.key(
                SubscriptionsIndexField.SUBS_READ_TYPES_MAP.fieldName());
            writer.value(SumMapFunction.mapToString(read));
            writer.key(
                SubscriptionsIndexField.SUBS_RECEIVED_TYPES_MAP.fieldName());
            writer.value(SumMapFunction.mapToString(total));
            writer.key(SubscriptionsIndexField.SUBS_NAME.fieldName());

            String name = "";
            Iterator<String> iterator = names.iterator();
            while (iterator.hasNext() && name.isEmpty()) {
                name = iterator.next();
            }

            writer.value(name);
        }
    }

    private static final class SubsReindexContext {
        private final SearchRequest<SubscriptionsStat, RequestParams, SearchInfo> request;
        private final User user;
        private final String updateUri;
        private final String deleteUri;
        private final DateTime limit;

        public SubsReindexContext(
            final SearchRequest<SubscriptionsStat, RequestParams, SearchInfo> request,
            final User user,
            final DateTime limit)
            throws HttpException
        {
            this.request = request;
            this.user = user;
            this.limit = limit;

            QueryConstructor updateUri = new QueryConstructor("/update?");
            updateUri.append("prefix", String.valueOf(user.prefix()));
            updateUri.append("service", user.service());
            updateUri.append("caller", "subscriptions_reindex");
            this.updateUri = updateUri.toString();

            QueryConstructor deleteUri = new QueryConstructor("/delete?");
            deleteUri.append("prefix", String.valueOf(user.prefix()));
            deleteUri.append("service", user.service());
            deleteUri.append("caller", "subscriptions_reindex");
            this.deleteUri = deleteUri.toString();
        }

        public SearchRequest<SubscriptionsStat, RequestParams, SearchInfo> request() {
            return request;
        }

        public User user() {
            return user;
        }

        public String producerUri() {
            return updateUri;
        }

        public String deleteUri() {
            return deleteUri;
        }

        public DateTime limit() {
            return limit;
        }

        public String prefix() {
            return user.prefix().toStringFast();
        }
    }
}
