package ru.yandex.msearch.proxy.api.async.mail.subscriptions;

import java.io.IOException;
import java.util.AbstractMap;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumer;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.mail.search.mail.MailSearchDatabases;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.parser.string.LongParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.mail.search.SubscriptionsFields;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;

public class SubscriptionsStatusHandler implements HttpAsyncRequestHandler<JsonObject> {
    public static final String SUBSCRIPTIONS_SERVICE = "change_log";
    private static final Set<String> TYPES =
        Collections.unmodifiableSet(new LinkedHashSet<>(Arrays.asList("7", "13")));

    private static final CollectionParser<SubscriptionStatus, Set<SubscriptionStatus>, RuntimeException> STATUS_SET_PARSER =
        new CollectionParser<>(
            new EnumParser<>(SubscriptionStatus.class),
            LinkedHashSet::new);
    private static final CollectionParser<String, Set<String>, Exception> SET_PARSER =
        new CollectionParser<>(
            NonEmptyValidator.TRIMMED,
            LinkedHashSet::new);
    private static final CollectionParser<Long, Set<Long>, Exception> UIDS_PARSER =
        new CollectionParser<>(
            LongParser.INSTANCE,
            LinkedHashSet::new);

    private final AsyncHttpServer proxy;

    public SubscriptionsStatusHandler(final AsyncHttpServer proxy) {
        this.proxy = proxy;
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException, IOException
    {
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity = ((HttpEntityEnclosingRequest) request).getEntity();
            return new JsonAsyncTypesafeDomConsumer(
                entity,
                StringCollectorsFactory.INSTANCE,
                BasicContainerFactory.INSTANCE);
        } else {
            return new EmptyAsyncConsumer<>(JsonNull.INSTANCE);
        }
    }

    @Override
    public void handle(
        final JsonObject data,
        final HttpAsyncExchange httpAsyncExchange,
        final HttpContext httpContext)
        throws HttpException, IOException
    {
        Set<String> emails = Collections.emptySet();
        if (data != JsonNull.INSTANCE) {
            try {
                JsonList emailsObj = data.asMap().getList("emails");
                emails = new LinkedHashSet<>(emailsObj.size() << 1);
                for (JsonObject jobj: emailsObj) {
                    emails.add(jobj.asString());
                }
            } catch (JsonException je) {
                throw new BadRequestException(je);
            }
        }

        ProxySession session = new BasicProxySession(proxy, httpAsyncExchange, httpContext);
        SubscriptionsStatusContext context = new SubscriptionsStatusContext(session, emails);

        MultiFutureCallback<Map.Entry<Long, Set<String>>> mfcb
            = new MultiFutureCallback<>(
                new SubscriptionsCallback(context));
        for (Long uid: context.uids()) {
            launchSingleUid(
                new SubscriptionsSingleUserStatusContext(context, uid),
                new SingleUidCallback(mfcb.newCallback(), uid));
        }
        mfcb.done();

    }

    private void launchSingleUid(
        final SubscriptionsSingleUserStatusContext context,
        final FutureCallback<JsonObject> callback)
        throws BadRequestException
    {
        QueryConstructor qc = new QueryConstructor("/search?");
        qc.append("prefix", context.user().prefix().toString());

        StringBuilder text = new StringBuilder();
        text.append("url:");
        text.append('(');
        for (String email: context.emails()) {
            text.append(subsId(context.uid, email));
            text.append(" OR ");
        }
        text.setLength(text.length() - 4);
        text.append(')');
        text.append(" AND ");
        text.append(SubscriptionsFields.SUBS_HIDDEN_TYPES.prefixed());
        text.append(":(");
        int i = 0;
        for (String type: TYPES) {
            if (i != 0) {
                text.append(" OR ");
            }

            text.append(type);
            i++;
        }
        text.append(")");

        qc.append("text", text.toString());
        qc.append("service", context.user().service());
        qc.append("get", "*");
        qc.append("sort", SubscriptionsFields.SUBS_RECEIVED_DATE.stored());
        qc.append("IO_PRIO", 100);
        qc.append("db", MailSearchDatabases.SUBSCRIPTIONS.dbName());

        proxy.sequentialRequest(
            context.session(),
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            100L,
            true,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.session().listener().adjustContextGenerator(
                context.client().httpClientContextGenerator()),
            callback);
    }

    private static final class SingleUidCallback
        extends AbstractFilterFutureCallback<JsonObject, Map.Entry<Long, Set<String>>> {
        private final Long uid;

        public SingleUidCallback(
            final FutureCallback<? super Map.Entry<Long, Set<String>>> callback,
            final Long uid)
        {
            super(callback);
            this.uid = uid;
        }

        @Override
        public void completed(final JsonObject backendResultObj) {
            try {
                JsonMap result = backendResultObj.asMap();
                JsonList hits = result.getList("hitsArray");

                Set<String> hidden = new LinkedHashSet<>(hits.size());
                for (JsonObject docObj: hits) {
                    JsonMap doc = docObj.asMap();
                    String email = doc.getString(SubscriptionsFields.SUBS_EMAIL.stored());
                    hidden.add(email);
                }

                callback.completed(new AbstractMap.SimpleEntry<>(uid, hidden));
            } catch (JsonException je) {
                failed(je);
            }
        }
    }

    private final class SubscriptionsCallback
        extends AbstractProxySessionCallback<List<Map.Entry<Long, Set<String>>>>
    {
        private final SubscriptionsStatusContext context;

        public SubscriptionsCallback(
            final SubscriptionsStatusContext context)
        {
            super(context.session());

            this.context = context;
        }

        @Override
        public void completed(final List<Map.Entry<Long, Set<String>>> hiddenByUid) {
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = context.jsonType().create(sbw)) {
                writer.startObject();
                writer.key("subscriptions");
                writer.startArray();
                for (Map.Entry<Long, Set<String>> uidEntry: hiddenByUid) {
                    for (String email: context.emails()) {
                        writer.startObject();
                        writer.key("uid");
                        writer.value(uidEntry.getKey());
                        writer.key("email");
                        writer.value(email);
                        writer.key("status");
                        if (uidEntry.getValue().contains(email)) {
                            writer.value(SubscriptionStatus.HIDDEN.value());
                        } else {
                            writer.value(SubscriptionStatus.ACTIVE.value());
                        }
                        writer.endObject();
                    }
                }
                writer.endArray();
                writer.endObject();

                session.response(
                    HttpStatus.SC_OK,
                    new NStringEntity(
                        sbw.toString(),
                        ContentType.APPLICATION_JSON
                            .withCharset(context.session().acceptedCharset())));
            } catch (IOException e) {
                failed(e);
            }
        }
    }

    private final class SubscriptionsStatusContext {
        private final AsyncClient client;
        private final ProxySession session;
        private final JsonType jsonType;
        private final Set<String> emails;
        private final Set<Long> uids;

        public SubscriptionsStatusContext(
            final ProxySession session,
            final Set<String> emails)
            throws BadRequestException
        {
            this.uids = session.params().get("uid", UIDS_PARSER);
            this.session = session;
            CgiParams params = session.params();
            jsonType = JsonTypeExtractor.NORMAL.extract(params);

            if (emails == null || emails.isEmpty()) {
                this.emails = params.get("email", SET_PARSER);
            } else {
                this.emails = emails;
            }

            if (this.emails == null || this.emails.isEmpty()) {
                throw new BadRequestException("No emails were supplied");
            }

            client = proxy.searchClient().adjust(session.context());
        }

        public AsyncClient client() {
            return client;
        }

        public ProxySession session() {
            return session;
        }

        public JsonType jsonType() {
            return jsonType;
        }

        public Set<String> emails() {
            return emails;
        }

        public Set<Long> uids() {
            return uids;
        }
    }

    private static final class SubscriptionsSingleUserStatusContext
        implements UniversalSearchProxyRequestContext
    {
        private final long uid;
        private final User user;
        private final SubscriptionsStatusContext context;

        public SubscriptionsSingleUserStatusContext(
            final SubscriptionsStatusContext context,
            final Long uid)
        {
            this.context = context;
            this.uid = uid;
            user = new User(SUBSCRIPTIONS_SERVICE, new LongPrefix(uid));
        }

        public Set<String> emails() {
            return context.emails();
        }

        public ProxySession session() {
            return context.session();
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return null;
        }

        @Override
        public AbstractAsyncClient<?> client() {
            return context.client();
        }

        @Override
        public Logger logger() {
            return context.session().logger();
        }

        @Override
        public long lagTolerance() {
            return 0L;
        }
    }

    public static final String subsId(final long uid, final String email) {
        StringBuilder sb = new StringBuilder(50);
        sb.append("subs_");
        sb.append(uid);
        sb.append('_');
        sb.append(email);
        return sb.toString();
    }
}
