package ru.yandex.msearch.proxy.api.async.mail.subscriptions.update;

import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicReference;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.StringEntity;
import org.apache.http.message.BasicHeader;

import ru.yandex.http.config.URIConfig;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.EmptyCancellationSubscriber;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.msearch.proxy.AsyncHttpServerBase;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.util.string.StringUtils;

public class SubscriptionsFuritaAppliableRules {
    private final SubscriptionsUpdateContext context;
    private final AsyncHttpServerBase<?> proxy;
    private final AsyncClient tupitaClient;
    private final HttpHost tupitaHost;
    private final AsyncClient furitaClient;
    private final HttpHost furitaHost;
    private final AsyncClient filterSearchClient;
    private final URIConfig filterSearchConfig;

    private final AtomicReference<UserFuritaRules> furitaRules =
        new AtomicReference<>();

    public SubscriptionsFuritaAppliableRules(
        final AsyncHttpServerBase<?> server,
        final SubscriptionsUpdateContext context)
    {
        this.furitaClient = server.furitaClient().adjust(context.httpContext());
        this.tupitaClient = server.tupitaClient().adjust(context.httpContext());
        this.furitaHost = server.config().furitaConfig().host();
        this.tupitaHost = server.config().tupitaConfig().host();
        this.filterSearchClient = server.filterSearchClient();
        this.filterSearchConfig = server.config().filterSearchConfig();
        this.proxy = server;
        this.context = context;
    }

    public void next(
        final FutureCallback<List<Map.Entry<String, List<FuritaRule>>>> callback,
        final Collection<String> emails,
        final int offset)
        throws BadRequestException
    {
        UserFuritaRules rules = furitaRules.get();
        if (rules == null) {
            furitaClient.execute(
                furitaHost,
                new BasicAsyncRequestProducerGenerator("/api/list.json?db=pg&detailed=1&uid=" + context.uid()),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                context.listener().createContextGeneratorFor(
                    furitaClient),
                new FuritaCallback(callback, context, emails, offset)
            );
        } else {
            next(
                callback,
                emails,
                rules,
                offset);
        }
    }

    private void next(
        final FutureCallback<? super List<Map.Entry<String, List<FuritaRule>>>> callback,
        final Collection<String> emails,
        final UserFuritaRules rules,
        final int offset)
        throws BadRequestException
    {
        context.logger().info("Mops prceeding to batch with offset " + offset);
        QueryConstructor qc =
            new QueryConstructor("/search-subscriptions-move-existing?");
        qc.append("prefix", context.user().prefix().toStringFast());
        qc.append("service", context.user().service());
        qc.append("offset", offset);
        qc.append("length", context.moveExistingBatchSize());
        qc.append("sort", "received_date");
        qc.append("get", "mid");

        StringBuilder text = new StringBuilder();
        text.append("hid:0 AND ");
        text.append("hdr_from_normalized:(");
        for (String email: emails) {
            text.append(SearchRequestText.fullEscape(email, false));
            text.append(' ');
        }
        text.setLength(text.length() - 1);
        text.append(')');
        text.append(" AND message_type:(");
        text.append(StringUtils.join(context.types(), " OR "));
        text.append(")");
        text.append(" AND fid:");
        text.append(context.pendingFid());

        qc.append("text", text.toString());
        proxy.sequentialRequest(
                EmptyCancellationSubscriber.INSTANCE,
                context.listener(),
                context.httpContext(),
                context,
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                5000L,
                false,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                context.listener().createContextGeneratorFor(
                        context.searchClient()),
                new SingleBatchFetchCallback(callback, rules, offset));
    }

    private class FuritaCallback
        extends AbstractFilterFutureCallback<JsonObject, List<Map.Entry<String, List<FuritaRule>>>>
    {
        final SubscriptionsUpdateContext context;
        final Collection<String> emails;
        final int offset;

        public FuritaCallback(
            final FutureCallback<? super List<Map.Entry<String, List<FuritaRule>>>> callback,
            final SubscriptionsUpdateContext context,
            final Collection<String> emails,
            final int offset)
        {
            super(callback);
            this.context = context;
            this.emails = emails;
            this.offset = offset;
        }

        @Override
        public void completed(final JsonObject furitaJsonObj) {
            UserFuritaRules result;
            try {
                JsonList rulesObj = furitaJsonObj.asMap().getList("rules");
                Map<String, FuritaRule> rules = new LinkedHashMap<>(rulesObj.size() << 1);
                for (JsonObject ruleObj: rulesObj) {
                    FuritaRule rule = FuritaRule.parseFromFurita(ruleObj.asMap());
                    rules.put(rule.id(), rule);
                }

                result = new UserFuritaRules(rules);
            } catch (JsonException e) {
                failed(e);
                return;
            }

            furitaRules.set(result);
            try {
                next(callback, emails, result, offset);
            } catch (BadRequestException bre) {
                failed(bre);
            }
        }
    }

    private class SingleBatchFetchCallback
        extends AbstractFilterFutureCallback<JsonObject, List<Map.Entry<String, List<FuritaRule>>>>
    {
        private final UserFuritaRules rules;
        private final int offset;

        public SingleBatchFetchCallback(
            final FutureCallback<? super List<Map.Entry<String, List<FuritaRule>>>> callback,
            final UserFuritaRules rules,
            final int offset)
        {
            super(callback);
            this.rules = rules;
            this.offset = offset;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                int totalLeft = result.asMap().getInt("hitsCount");
                JsonList list = result.asMap().getList("hitsArray");
                context.logger().info("Got from backend " + list.size() + " totalLeft " + totalLeft);
                if (list.size() > 0) {
                    // estimating mid size <20 (actual on code write moment = 18 + comma)

                    QueryConstructor fsQuery = new QueryConstructor(
                        filterSearchConfig.uri().toString() + '?');

                    fsQuery.append("uid", context.uid());
                    fsQuery.append("mdb", "pg");
                    fsQuery.append("full_folders_and_labels", "1");
                    fsQuery.append("incl_folders", "pending");
                    fsQuery.append("order", "default");

                    for (JsonObject item: list) {
                        String mid = item.asMap().getString("mid");
                        fsQuery.append("mids", mid);
                    }

                    filterSearchClient.execute(
                        new HeaderAsyncRequestProducerSupplier(
                            new AsyncGetURIRequestProducerSupplier(fsQuery.toString()),
                            new BasicHeader(YandexHeaders.X_YA_SERVICE_TICKET,
                                proxy.filterSearchTvm2Ticket())),
                        JsonAsyncTypesafeDomConsumerFactory.INTERNING_OK,
                        context.listener()
                            .createContextGeneratorFor(filterSearchClient),
                        new FilterSearchCallback(callback, rules));
                } else {
                    context.logger().info(
                        "Last request from offset " + offset + " returned 0 results, completing");
                    super.callback.completed(Collections.emptyList());
                    return;
                }
            } catch (JsonException | BadRequestException | URISyntaxException e) {
                callback.failed(e);
            }
        }
    }

    private class FilterSearchCallback
        extends AbstractFilterFutureCallback<JsonObject, List<Map.Entry<String, List<FuritaRule>>>>
    {
        private final UserFuritaRules furitaRules;

        public FilterSearchCallback(
            final FutureCallback<? super List<Map.Entry<String, List<FuritaRule>>>> callback,
            final UserFuritaRules furitaRules)
        {
            super(callback);
            this.furitaRules = furitaRules;
        }

        @Override
        public void completed(final JsonObject root) {
            try {
                JsonList envelopes = root.get("envelopes").asList();
                StringBuilderWriter sbw = new StringBuilderWriter();
                JsonWriter writer = JsonType.NORMAL.create(sbw);
                MultiFutureCallback<Map.Entry<String, List<FuritaRule>>> mfcb = new MultiFutureCallback<>(callback);
                for (int i = 0; i < envelopes.size(); ++i) {
                    JsonMap envelope = envelopes.get(i).asMap();
                    String mid = envelope.getString("mid");
                    writer.startObject();
                    writer.key("message");
                    writer.value((JsonValue) envelope);
                    writer.key("users");
                    writer.startArray();
                    writer.startObject();
                    writer.key("uid");
                    writer.value(context.uid());
                    writer.key("queries");
                    writer.value(furitaRules);
                    writer.endObject();
                    writer.endArray();
                    writer.endObject();

                    StringEntity body =
                        new StringEntity(
                            sbw.toString(),
                            StandardCharsets.UTF_8);

                    tupitaClient.execute(
                        tupitaHost,
                        new BasicAsyncRequestProducerGenerator(
                            "/check?subscriptions&uid=" + context.uid(),
                            body),
                        JsonAsyncTypesafeDomConsumerFactory.POSITION_SAVING_OK,
                        context.listener().createContextGeneratorFor(tupitaClient),
                        new SingleTupitaCallback(mfcb.newCallback(), furitaRules, mid)
                    );

                    writer.reset();
                    sbw.clear();
                }

                mfcb.done();
            } catch (JsonException | IOException e) {
                failed(new JsonException(
                    "Failed to parse: " + JsonType.NORMAL.toString(root), e));
                return;
            }
        }
    }

    private static class SingleTupitaCallback
        extends AbstractFilterFutureCallback<JsonObject, Map.Entry<String, List<FuritaRule>>>
    {
        private final UserFuritaRules rules;
        private final String mid;

        public SingleTupitaCallback(
            final FutureCallback<? super Map.Entry<String, List<FuritaRule>>> callback,
            final UserFuritaRules rules,
            final String mid)
        {
            super(callback);
            this.rules = rules;
            this.mid = mid;
        }

        @Override
        public void completed(final JsonObject tupitaResp) {
            List<FuritaRule> matchedRules;
            try {
                JsonList list = tupitaResp.asMap().getList("result");
                if (list.size() == 0) {
                    failed(new Exception("Tupita empty result"));
                    return;
                }

                JsonList matched = list.get(0).asMap().getList("matched_queries");
                matchedRules = new ArrayList<>(matched.size());
                for (JsonObject jo: matched) {
                    String id = jo.asString();
                    matchedRules.add(rules.get(id));
                }
            } catch (JsonException je) {
                failed(je);
                return;
            }

            callback.completed(new AbstractMap.SimpleEntry<>(mid, matchedRules));
        }
    }
}
