package ru.yandex.msearch.proxy.api.async.mail.subscriptions.update;

import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.message.BasicHeader;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.EmptyCancellationSubscriber;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.EmptyAsyncConsumer;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.mail.search.mail.MailSearchDatabases;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.AsyncHttpServerBase;
import ru.yandex.msearch.proxy.MsearchProxyExperiment;
import ru.yandex.msearch.proxy.api.async.ProxyParams;
import ru.yandex.msearch.proxy.api.async.mail.subscriptions.update.dao.MigrationsTasksPostgresDao;
import ru.yandex.msearch.proxy.api.async.mail.subscriptions.update.dao.pojo.MigrationTask;
import ru.yandex.msearch.proxy.api.async.suggest.BasicSuggestRequestParams;
import ru.yandex.msearch.proxy.config.ImmutableAsyncHttpServerBaseConfig;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.util.string.StringUtils;

public class SubscriptionsUpdateStatusHandler implements HttpAsyncRequestHandler<JsonObject> {
    public static final String OPT_IN_PENDING_FOLDER = "pending";

    private static final StringBody EMPTY_BODY =
        new StringBody("", ContentType.TEXT_PLAIN);
    public static final String SUBSCRIPTIONS_SERVICE
        = "subscriptions_prod_1";
        //= MailSearchServices.SUBSCRIPTIONS_1.service();
    public static final String FALLBACK_SUBSCRIPTIONS_SERVICE
            = "subscriptions_prod_2";
        //= MailSearchServices.SUBSCRIPTIONS_2.service();

    public static final CollectionParser<String, Set<String>, Exception> SET_PARSER =
        new CollectionParser<>(
            NonEmptyValidator.TRIMMED,
            LinkedHashSet::new);

    private final AsyncHttpServer proxy;
    private final AsyncClient storeClient;

    private final MigrationsTasksPostgresDao migrationsTasksPostgresDao;

    public SubscriptionsUpdateStatusHandler(AsyncHttpServer proxy) throws ConfigException  {
        this.proxy = proxy;
        this.storeClient =
            proxy.client(
                "SubscriptionsStoreClient",
                new HttpTargetConfigBuilder()
                    .connections(100)
                    .timeout((int) TimeUnit.MINUTES.toMillis(10))
                    .build());
        this.migrationsTasksPostgresDao = proxy.migrationsTasksDao();
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException, IOException {
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity = ((HttpEntityEnclosingRequest) request).getEntity();
            return new JsonAsyncTypesafeDomConsumer(
                entity,
                StringCollectorsFactory.INSTANCE,
                BasicContainerFactory.INSTANCE);
        } else {
            return new EmptyAsyncConsumer<>(JsonNull.INSTANCE);
        }
    }

    public static String updateBodyRequest(
        final SubscriptionsUpdateContext context,
        final List<UpdateItem> updates)
        throws IOException
    {
        StringBuilderWriter sbw = new StringBuilderWriter();
        JsonWriter writer = JsonType.NORMAL.create(sbw);
        writer.startObject();
        writer.key("prefix");
        writer.value(context.uid());
        writer.key("AddIfNotExists");
        writer.value(true);
        writer.key("docs");
        writer.startArray();
        for (UpdateItem item: updates) {
            writer.value(item);
        }
        writer.endArray();
        writer.endObject();
        return sbw.toString();
    }

    public static QueryConstructor updateUri(
        final SubscriptionsUpdateContext context,
        final String service,
        final List<UpdateItem> updates)
        throws BadRequestException
    {
        QueryConstructor qc = new QueryConstructor("/update?subscriptions");
        qc.append("prefix", context.user().prefix().toStringFast());
        qc.append("service", service);
        //qc.append("wait", "true");
        qc.append("size", updates.size());
        qc.append("db", MailSearchDatabases.SUBSCRIPTIONS.dbName());
        return qc;
    }

    public static BasicAsyncRequestProducerGenerator updateRequest(
        final SubscriptionsUpdateContext context,
        final String service,
        final List<UpdateItem> updates)
        throws IOException, BadRequestException
    {
        return new BasicAsyncRequestProducerGenerator(
            updateUri(context, service, updates).toString(),
            updateBodyRequest(context, updates),
            StandardCharsets.UTF_8);
    }

    @Override
    public void handle(
        final JsonObject data,
        final HttpAsyncExchange httpAsyncExchange,
        final HttpContext httpContext)
        throws HttpException, IOException
    {
        ProxySession session = new BasicProxySession(proxy, httpAsyncExchange, httpContext);
        SubscriptionsUpdateContext context = SubscriptionsUpdateContext.fromSession(this.proxy, storeClient, session);

        List<UpdateItem> updatesSimple;
        List<UpdateItem> updatesWithMoveExisting;
        if (data != null && data != JsonNull.INSTANCE) {
            try {
                JsonList list = data.asList();
                updatesSimple = new ArrayList<>(list.size());
                updatesWithMoveExisting = Collections.emptyList();
                for (JsonObject item: list) {
                    UpdateItem updateItem = UpdateItem.fromMap(item.asMap(), context);
                    if (updateItem.moveExisting()) {
                        if (updatesWithMoveExisting.size() == 0) {
                            updatesWithMoveExisting = new ArrayList<>(list.size());
                        }
                        updatesWithMoveExisting.add(updateItem);
                    } else {
                        updatesSimple.add(updateItem);
                    }
                }
            } catch (JsonException je) {
                throw new BadRequestException(je);
            }
        } else {
            UpdateItem updateItem = UpdateItem.fromMap(session.params(), context);
            if (updateItem.moveExisting()) {
                updatesWithMoveExisting = Collections.singletonList(updateItem);
                updatesSimple = Collections.emptyList();
            } else {
                updatesSimple = Collections.singletonList(updateItem);
                updatesWithMoveExisting = Collections.emptyList();
            }
        }

        FutureCallback<Object> requestCallback = new SubscriptionsUpdateRequestCallback(session);
        if (context.moveFromFurita()) {
            if (updatesWithMoveExisting.size() > 0) {
                throw new BadRequestException(
                    "Transfer from furita and moving mails forbidden in same time");
            }
            MultipartEntityBuilder builder = MultipartEntityBuilder.create();
            builder.setMimeSubtype("mixed");
            long shard = context.uid() % SearchMap.SHARDS_COUNT;
            builder.addPart(
                FormBodyPartBuilder
                    .create()
                    .addField(
                        YandexHeaders.ZOO_SHARD_ID,
                        String.valueOf(shard))
                    .addField(
                        YandexHeaders.URI,
                        "/subscriptions/move_from_furita?prefix=" + context.uid())
                    .addField(YandexHeaders.ZOO_HTTP_METHOD, "GET")
                    .setBody(EMPTY_BODY)
                    .setName("subscriptions_move.json")
                    .build());

            builder.addPart(
                FormBodyPartBuilder
                    .create()
                    .addField(
                        YandexHeaders.ZOO_SHARD_ID,
                        String.valueOf(shard))
                    .addField(
                        YandexHeaders.URI,
                        updateUri(context, SUBSCRIPTIONS_SERVICE, updatesSimple).toString())
                    .setBody(new StringBody(updateBodyRequest(context, updatesSimple),  ContentType.APPLICATION_JSON))
                    .setName("subscriptions_move.json")
                    .build());
            QueryConstructor qc = new QueryConstructor("/notify?subscriptions");
            qc.append("uid", context.uid());
            qc.append("size", updatesSimple.size());

            BasicAsyncRequestProducerGenerator generator = new BasicAsyncRequestProducerGenerator(qc.toString(), builder.build());
            generator.addHeader(YandexHeaders.SERVICE, SUBSCRIPTIONS_SERVICE);

            BasicAsyncRequestProducerGenerator fallbackGenerator =
                new BasicAsyncRequestProducerGenerator(qc.toString(), builder.build());
            fallbackGenerator.addHeader(YandexHeaders.SERVICE, FALLBACK_SUBSCRIPTIONS_SERVICE);

            store(context, requestCallback, proxy, generator, fallbackGenerator);
        } else {
            BasicAsyncRequestProducerGenerator generator = updateRequest(context, SUBSCRIPTIONS_SERVICE, updatesSimple);
            generator.addHeader(YandexHeaders.SERVICE, SUBSCRIPTIONS_SERVICE);

            BasicAsyncRequestProducerGenerator fallbackGenerator = updateRequest(context, FALLBACK_SUBSCRIPTIONS_SERVICE, updatesSimple);
            fallbackGenerator.addHeader(YandexHeaders.SERVICE, FALLBACK_SUBSCRIPTIONS_SERVICE);

            if (updatesWithMoveExisting.size() <= 0) {
                context.logger().info("No move existing flags, just update status");
                store(context, requestCallback, proxy, generator, fallbackGenerator);
            } else {
                context.logger().info("Moving emails subs size " + updatesWithMoveExisting.size());
                context.logger().info("Writing to postgres");

                migrationsTasksPostgresDao.insert(
                        updatesWithMoveExisting
                                .stream()
                                .map(item -> MigrationTask.fromUpdateItem(item, context))
                                .collect(Collectors.toList())
                );

                List<UpdateItem> allUpdates = Stream.concat(
                        updatesSimple.stream(),
                        updatesWithMoveExisting.stream()
                ).collect(Collectors.toList());

                BasicAsyncRequestProducerGenerator storeAllGenerator =
                        updateRequest(context, SUBSCRIPTIONS_SERVICE, allUpdates);
                generator.addHeader(YandexHeaders.SERVICE, SUBSCRIPTIONS_SERVICE);

                BasicAsyncRequestProducerGenerator storeAllFallbackGenerator =
                        updateRequest(context, FALLBACK_SUBSCRIPTIONS_SERVICE, allUpdates);
                fallbackGenerator.addHeader(YandexHeaders.SERVICE, FALLBACK_SUBSCRIPTIONS_SERVICE);

                store(
                        context,
                        requestCallback,
                        proxy,
                        storeAllGenerator,
                        storeAllFallbackGenerator);
            }
        }
    }

    public static void store(
        final SubscriptionsUpdateContext context,
        final FutureCallback<Object> callback,
        final AsyncHttpServerBase<? extends ImmutableAsyncHttpServerBaseConfig>
            proxy,
        final BasicAsyncRequestProducerGenerator generator,
        final BasicAsyncRequestProducerGenerator fallbackGenerator)
    {
        FutureCallback<Object> storeCallback;
        if (fallbackGenerator != null) {
            storeCallback = new FallbackStoreCallback(
                context,
                fallbackGenerator,
                proxy.config().producerStoreConfig().host(),
                callback);
        } else {
            storeCallback = null;
        }

        context.storeClient().execute(
            proxy.config().producerStoreConfig().host(),
            generator,
            BasicAsyncResponseConsumerFactory.ANY_GOOD,
            context.listener().adjustContextGenerator(
                context.storeClient().httpClientContextGenerator()),
            storeCallback);
    }

    public static void moveExisting(
        final FutureCallback<Object> callback,
        final SubscriptionsUpdateContext context,
        final AsyncHttpServerBase<? extends ImmutableAsyncHttpServerBaseConfig>
            proxy,
        final UpdateItem item,
        final int offset)
        throws BadRequestException
    {
        if (context.optIn() && item.action() == UpdateAction.ACTIVATE && context.pendingFid() == null) {
            context.logger().info("Pending fid null, skipping move " + item.email());
            callback.completed(null);
            return;
        }
        context.logger().info("Mops prceeding to batch with offset " + offset);
        QueryConstructor qc =
            new QueryConstructor("/search-subscriptions-move-existing?");
        qc.append("prefix", context.user().prefix().toStringFast());
        qc.append("service", context.user().service());
        qc.append("offset", offset);
        qc.append("length", context.moveExistingBatchSize());
        qc.append("sort", "received_date");
        qc.append("get", "mid");

        StringBuilder text = new StringBuilder();
        text.append("hid:0 AND ");
        text.append("hdr_from_normalized:");
        text.append(SearchRequestText.fullEscape(item.email(), false));
        text.append(" AND message_type:(");
        text.append(StringUtils.join(context.types(), " OR "));
        text.append(")");

        if (context.optIn() && item.action() == UpdateAction.ACTIVATE) {
            text.append(" AND fid:");
            text.append(context.pendingFid());
        }

        qc.append("text", text.toString());
        proxy.sequentialRequest(
                EmptyCancellationSubscriber.INSTANCE,
                context.listener(),
                context.httpContext(),
                context,
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                5000L,
                false,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                context.listener().createContextGeneratorFor(
                context.searchClient()),
                new SingleBatchFetchCallback(callback, context, proxy, item, offset)
        );
    }

    private static class SingleBatchFetchCallback
        extends AbstractFilterFutureCallback<JsonObject, Object>
    {
        private final FutureCallback<Object> allMovedCallback;
        private final SubscriptionsUpdateContext context;
        private final AsyncHttpServerBase
            <? extends ImmutableAsyncHttpServerBaseConfig> proxy;
        private final UpdateItem item;
        private final int offset;

        public SingleBatchFetchCallback(
            final FutureCallback<Object> allMovedCallback,
            final SubscriptionsUpdateContext context,
            final AsyncHttpServerBase
                <? extends ImmutableAsyncHttpServerBaseConfig> proxy,
            final UpdateItem item,
            final int offset)
        {
            super(allMovedCallback);
            this.allMovedCallback = allMovedCallback;
            this.context = context;
            this.item = item;
            this.offset = offset;
            this.proxy = proxy;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                int totalLeft = result.asMap().getInt("hitsCount");
                JsonList list = result.asMap().getList("hitsArray");
                context.logger().info("Got from backend " + list.size() + " totalLeft " + totalLeft);
                if (list.size() > 0) {
                    // estimating mid size <20 (actual on code write moment = 18 + comma)
                    StringBuilder mids = new StringBuilder(list.size() * 20);
                    for (JsonObject item: list) {
                        String mid = item.asMap().getString("mid");
                        mids.append(mid);
                        mids.append(',');
                    }

                    if (mids.length() > 0) {
                        mids.setLength(mids.length() - 1);
                    }
                    QueryConstructor mopsRequest;
                    if (context.optIn() && item.action() == UpdateAction.ACTIVATE) {
                        mopsRequest = new QueryConstructor(
                                "/complex_move?&subscription_activate_subs_optin");
                        mopsRequest.append("dest_fid", context.inboxFid());
                        mopsRequest.append("with_sent", "0");
                    } else {
                        mopsRequest = new QueryConstructor(
                            "/remove?&subscription_hide_with_remove");
                        mopsRequest.append("nopurge", "1");
                    }

                    mopsRequest.append("uid", context.uid());
                    mopsRequest.append("request_mids_count", list.size());
                    mopsRequest.append("source", "mail_search");
                    mopsRequest.append("mids", mids.toString());

                    BasicAsyncRequestProducerGenerator generator =
                        new BasicAsyncRequestProducerGenerator(
                            mopsRequest.toString(),
                            "",
                            ContentType.TEXT_PLAIN);

                    generator.addHeader(YandexHeaders.X_REQUEST_ID, context.requestId());

                    boolean doNext = list.size() >= context.moveExistingBatchSize();

                    context.logger().info(
                        "Mops request " + mopsRequest.toString()
                            + "doNext " + doNext);
                    context.mopsClient().execute(
                        proxy.config().mopsClientConfig().host(),
                        generator,
                        JsonAsyncTypesafeDomConsumerFactory.OK,
                        context.listener().createContextGeneratorFor(
                            proxy.searchClient()),
                        new MopsCallback(allMovedCallback, context, proxy, item, offset, doNext));
                } else {
                    context.logger().info(
                        "Last request from offset " + offset + " returned 0 results, completing");
                    super.callback.completed(null);
                    return;
                }
            } catch (JsonException  | BadRequestException e) {
                callback.failed(e);
            }
        }
    }

    private static class MopsCallback extends AbstractFilterFutureCallback<JsonObject, Object> {
        private final FutureCallback<Object> allMovedCallback;
        private final SubscriptionsUpdateContext context;
        private final AsyncHttpServerBase proxy;
        private final UpdateItem item;
        private final int offset;
        private final boolean doNext;

        public MopsCallback(
            final FutureCallback<Object> allMovedCallback,
            final SubscriptionsUpdateContext context,
            final AsyncHttpServerBase proxy,
            final UpdateItem item,
            final int offset,
            final boolean doNext)
        {
            super(allMovedCallback);
            this.allMovedCallback = allMovedCallback;
            this.context = context;
            this.item = item;
            this.offset = offset;
            this.doNext = doNext;
            this.proxy = proxy;
        }

        @Override
        public void completed(final JsonObject mopsResult) {
            try {
                JsonMap map = mopsResult.asMap();
                String result = map.getString("result");
                String error = map.getString("error", "");
                //String taskType = map.getString("taskType", "");
                if (!"ok".equalsIgnoreCase(result)) {
                    context.logger().info("Mops returned error " + error + " and status " + result);
                    failed(new ServerException(HttpStatus.SC_INTERNAL_SERVER_ERROR, "Mops failed on offset " + offset + " with error " + error));
                    return;
                }

                if (doNext) {
                    moveExisting(
                        callback,
                        context,
                        proxy,
                        item,
                        offset + context.moveExistingBatchSize());
                } else {
                    allMovedCallback.completed(null);
                }
            } catch (JsonException | BadRequestException e) {
                failed(e);
                return;
            }
        }
    }


    private static class FallbackStoreCallback extends FilterFutureCallback<Object> {
        private final BasicAsyncRequestProducerGenerator generator;
        private final SubscriptionsUpdateContext context;
        private final HttpHost producerHost;

        public FallbackStoreCallback(
            final SubscriptionsUpdateContext context,
            final BasicAsyncRequestProducerGenerator generator,
            final HttpHost producerHost,
            final FutureCallback<? super Object> callback)
        {
            super(callback);

            this.context = context;
            this.generator = generator;
            this.producerHost = producerHost;
        }

        @Override
        public void failed(final Exception e) {
            context.storeClient().execute(
                producerHost,
                generator,
                BasicAsyncResponseConsumerFactory.ANY_GOOD,
                context.listener().adjustContextGenerator(
                    context.storeClient().httpClientContextGenerator()),
                callback);
        }
    }
    private static class SubscriptionsUpdateRequestCallback extends AbstractProxySessionCallback<Object> {
        public SubscriptionsUpdateRequestCallback(ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final Object o) {
            session.response(HttpStatus.SC_OK);
        }
    }
}
