package ru.yandex.msearch.proxy.api.async.mail.tabs.category;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.nio.protocol.BasicAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.ProxyParams;
import ru.yandex.msearch.proxy.mail.SearchFilter;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.PrefixType;
import ru.yandex.search.result.SearchResult;

public class CategoryHandler implements HttpAsyncRequestHandler<HttpRequest>
{
    private static final List<String> DEFAULT_CATEGORIES =
        Arrays.asList("people", "eshops", "social", "trips");

    private static final String QUERY_BASE =
        "/search-category?group=mid&merge_func=none&sort=received_date"
            + "&get=mid&length=1";

    private final List<String> categories;
    private final AsyncHttpServer server;

    public CategoryHandler(final AsyncHttpServer server) {
        this.server = server;

        Set<String> filters =
            server.config().filtersConfig().filters().keySet();
        this.categories =
            DEFAULT_CATEGORIES.stream()
                .filter(filters::contains)
                .collect(Collectors.toList());
    }

    @Override
    public HttpAsyncRequestConsumer<HttpRequest> processRequest(
        final HttpRequest httpRequest,
        final HttpContext httpContext)
        throws HttpException, IOException
    {
        return new BasicAsyncRequestConsumer();
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException, IOException
    {
        ProxySession session = new BasicProxySession(server, exchange, context);
        FutureCallback<Map<String, List<String>>> callback
            = new CategoriesPrinter(session);

        CgiParams params = session.params();
        Long suid = params.getLong(ProxyParams.SUID, null);
        Long uid = params.getLong(ProxyParams.UID, null);

        String mdb = params.getString(ProxyParams.MDB, null);

        if (suid == null && uid == null) {
            callback.failed(new BadRequestException("No user id supplied"));
            return;
        }

        if (mdb == null) {
            callback.failed(new BadRequestException("No mdb supplied"));
            return;
        }

        PrefixType prefixType = server.searchMap().prefixType(mdb);
        Prefix prefix;
        if (mdb.equals("pg")) {
            prefix = params.get(ProxyParams.UID, prefixType);
        } else {
            prefix = params.get(ProxyParams.SUID, prefixType);
        }

        User user = new User(server.resolveService(mdb, prefix), prefix);
        long position = params.getLong("position");

        PositionPlainSearcher searcher
            = new PositionPlainSearcher(server, session, user, position);

        List<String> mids = params.getAll("mid");

        MultiFutureCallback<List<String>> aggregateCallback =
            new MultiFutureCallback<>(new AggregateCallback(mids, callback));

        Map<String, SearchFilter> filters =
            server.config().filtersConfig().filters();

        StringBuilder query = new StringBuilder(QUERY_BASE);
        query.append("&prefix=");
        query.append(user.prefix().toString());
        query.append("&service=");
        query.append(user.service());
        query.append("&user-id-term=");
        query.append(user.prefix().toString());

        for (String mid: mids) {
            MultiFutureCallback<SearchResult> mfcb =
                new MultiFutureCallback<>(
                    new MailSearchCallback(
                        session,
                        categories,
                        aggregateCallback.newCallback()));

            for (String category: categories) {
                QueryConstructor qc = new QueryConstructor(query.toString());
                StringBuilder text = new StringBuilder("url:");
                text.append(user.prefix().toString());
                text.append('_');
                text.append(mid);
                text.append("/0 AND ");
                filters.get(category).apply(text, null);
                qc.append("text", text.toString());

                session.logger().info("Request " + qc.toString());
                searcher.search(
                    new BasicAsyncRequestProducerGenerator(qc.toString()),
                    mfcb.newCallback());
            }

            mfcb.done();
        }

        aggregateCallback.done();
    }

    private static final class AggregateCallback
        extends AbstractFilterFutureCallback<List<List<String>>, Map<String, List<String>>>
    {
        private final List<String> mids;

        public AggregateCallback(
            final List<String> mids,
            final FutureCallback<? super Map<String, List<String>>> callback)
        {
            super(callback);

            this.mids = mids;
        }

        @Override
        public void completed(final List<List<String>> lists) {
            Map<String, List<String>> result =
                new LinkedHashMap<>(lists.size());

            for (int i = 0; i < lists.size(); i++) {
                result.put(mids.get(i), lists.get(i));
            }

            callback.completed(result);
        }
    }

    private static final class MailSearchCallback
        extends AbstractFilterFutureCallback<List<SearchResult>, List<String>>
    {
        private final List<String> filters;
        private final ProxySession session;

        public MailSearchCallback(
            final ProxySession session,
            final List<String> filters,
            final FutureCallback<? super List<String>> callback)
        {
            super(callback);

            this.session = session;
            this.filters = filters;
        }

        @Override
        public void completed(final List<SearchResult> results) {
            List<String> result = new ArrayList<>(filters.size());

            for (int i = 0; i < filters.size(); i++) {
                if (!results.get(i).hitsArray().isEmpty()) {
                    result.add(filters.get(i));
                }
            }

            callback.completed(result);
        }
    }
}
