package ru.yandex.msearch.proxy.api.async.mail.tabs.content;

import java.util.Comparator;
import java.util.function.Function;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;

import org.apache.http.nio.protocol.BasicAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;

import org.apache.http.protocol.HttpContext;

import ru.yandex.function.GenericFunction;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.msearch.proxy.AsyncHttpServer;

import ru.yandex.msearch.proxy.api.async.BasicSession;
import ru.yandex.msearch.proxy.api.async.Session;

import ru.yandex.msearch.proxy.api.async.mail.BasicSearchSession;
import ru.yandex.msearch.proxy.api.async.mail.DefaultSearchAttributes;
import ru.yandex.msearch.proxy.api.async.mail.MailSearchHandler;
import ru.yandex.msearch.proxy.api.async.mail.RequestInfo;
import ru.yandex.msearch.proxy.api.async.mail.result.ResultPrinter;
import ru.yandex.msearch.proxy.api.async.mail.SearchAttributes;
import ru.yandex.msearch.proxy.api.async.mail.result.ThreadedResultPrinter;

import ru.yandex.msearch.proxy.api.async.mail.documents.Document;
import ru.yandex.msearch.proxy.api.async.mail.documents.Documents;

import ru.yandex.msearch.proxy.api.async.mail.rules.PlainSearchRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.RewriteRequestRule;
import ru.yandex.msearch.proxy.api.async.mail.tabs.content.TabsSearchCallback.ThreadedDocuments;
import ru.yandex.msearch.proxy.api.async.mail.rules.ResolveUidRule;
import ru.yandex.msearch.proxy.api.async.mail.rules.RuleContext;
import ru.yandex.msearch.proxy.api.async.mail.rules.SearchRule;

import ru.yandex.parser.uri.CgiParams;

public class TabsContentHandler
    implements HttpAsyncRequestHandler<HttpRequest>
{
    private final AsyncHttpServer server;
    private final SearchRule threadedRule;
    private final SearchRule nonThreadedRule;

    public TabsContentHandler(final AsyncHttpServer server) {
        this.server = server;

        RuleContext context =
            new RuleContext(
                server,
                ThreadedDocumentsFactory.INSTANCE,
                TabsContentSearchAttributesFactory.INSTANCE);

        SearchRule rule = new TabsPlainSearchRule(context);
        rule = new TabsRewriteRequestRule(rule);

        this.threadedRule = new ResolveUidRule(rule, server);

        SearchRule plainSearchRule =
            new PlainSearchRule(
                new RuleContext(
                    server,
                    MailSearchHandler.BasicDocumentsFactory.INSTANCE,
                    NonThreadAttributesFactory.INSTANCE));
        SearchRule rewriteRequestRule =
            new RewriteRequestRule(server, plainSearchRule);

        this.nonThreadedRule = new ResolveUidRule(rewriteRequestRule, server);
    }

    @Override
    public BasicAsyncRequestConsumer processRequest(
        final HttpRequest request,
        final HttpContext context)
    {
        return new BasicAsyncRequestConsumer();
    }

    @Override
    public String toString() {
        return "Returns list of threads for tab";
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        Session httpSession = new BasicSession(server, exchange, context);
        CgiParams params = new CgiParams(request);
        RequestInfo requestInfo =
            new RequestInfo(httpSession, request, params);

        boolean threaded = params.getBoolean("threaded", false);
        SearchRule rule;
        ResultPrinter printer;

        if (threaded) {
            rule = threadedRule;
            printer = new ThreadedResultPrinter(requestInfo);
        } else {
            params = new CgiParams(params);
            params.replace("request", "");
            params.replace("search-filter", params.getString("tab"));
            rule = nonThreadedRule;
            printer = new ResultPrinter(requestInfo);
        }

        rule.execute(
            new BasicSearchSession(requestInfo, params, printer));
    }

    private enum TabsContentSearchAttributesFactory
        implements GenericFunction<
        CgiParams,
        SearchAttributes,
        BadRequestException>
    {
        INSTANCE;

        @Override
        public SearchAttributes apply(final CgiParams params)
            throws BadRequestException
        {
            return new TabsContentSearchAttributes(params);
        }
    }

    public enum ThreadedDocumentsFactory
        implements Function<Comparator<Document>, Documents>
    {
        INSTANCE;

        @Override
        public Documents apply(final Comparator<Document> comparator) {
            return new ThreadedDocuments();
        }
    }

    private enum NonThreadAttributesFactory
        implements GenericFunction<
        CgiParams,
        SearchAttributes,
        BadRequestException>
    {
        INSTANCE;

        @Override
        public SearchAttributes apply(final CgiParams params)
            throws BadRequestException
        {
            return new DefaultSearchAttributes(params);
        }
    }
}

