package ru.yandex.msearch.proxy.api.async.mail.tabs.content;

import java.net.URISyntaxException;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map.Entry;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.message.BasicHeader;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.msearch.proxy.api.async.mail.SearchRequest;
import ru.yandex.msearch.proxy.api.async.mail.SearchSession;
import ru.yandex.msearch.proxy.api.async.mail.documents.Document;
import ru.yandex.msearch.proxy.api.async.mail.documents.Documents;
import ru.yandex.msearch.proxy.api.async.mail.documents.DocumentsGroup;
import ru.yandex.msearch.proxy.api.async.mail.rules.AbstractSessionCallback;
import ru.yandex.msearch.proxy.api.async.mail.rules.FilterPlainSearchCallback;
import ru.yandex.msearch.proxy.api.async.mail.rules.RuleContext;
import ru.yandex.search.result.BasicSearchResult;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;
import ru.yandex.util.string.StringUtils;

public class TabsSearchCallback extends FilterPlainSearchCallback {
    private final RuleContext context;
    protected AsyncClient threadsClient;

    public TabsSearchCallback(
        final SearchSession session,
        final RuleContext context,
        final SearchRequest request)
        throws BadRequestException
    {
        super(session, context, request);

        threadsClient =
            context.server().threadsClient().adjust(
                session.httpSession().context());
        this.context = context;
    }

    private void addUniqueMid(
        final List<SearchDocument> docs,
        final Set<String> mids,
        final SearchDocument doc)
    {
        if (mids.add(doc.attrs().get("mid"))) {
            docs.add(doc);
        }
    }

    @Override
    protected int sendSubrequests(final SearchResult result) {
        // Extract all mids merged by group=thread_id, so we can filter them
        // This is the deduplicating collection
        Set<String> mids = new HashSet<>();
        // Do not add documents with empty mid
        mids.add(null);
        List<SearchDocument> docs = new ArrayList<>();
        for (SearchDocument hit : result.hitsArray()) {
            addUniqueMid(docs, mids, hit);
            for (SearchDocument mergedDoc : hit.mergedDocs()) {
                addUniqueMid(docs, mids, mergedDoc);
            }
            hit.mergedDocs().clear();
        }
        return super.sendSubrequests(new BasicSearchResult(docs, 0));
    }

    private boolean filterSearch(
        final Map<String, SearchDocument> docs,
        final FutureCallback<JsonObject> callback)
    {
        StringBuilder sb =
            new StringBuilder(request.filterSearchRequest());
        for (String mid : docs.keySet()) {
            sb.append("&mids=");
            sb.append(mid);
        }
        try {
            filterSearchClient.execute(
                new HeaderAsyncRequestProducerSupplier(
                    new AsyncGetURIRequestProducerSupplier(new String(sb)),
                    new BasicHeader(YandexHeaders.X_YA_SERVICE_TICKET,
                        context.server().filterSearchTvm2Ticket(request.corp()))),
                JsonAsyncTypesafeDomConsumerFactory.INTERNING_OK,
                session.httpSession().requestsListener()
                    .createContextGeneratorFor(filterSearchClient),
                callback);
            return true;
        } catch (URISyntaxException e) {
            failed(e);
            return false;
        }
    }

    private boolean threadSearch(
        final Map<String, SearchDocument> docs,
        final FutureCallback<JsonObject> callback)
    {
        StringBuilder sb =
            new StringBuilder(request.threadsSearchRequest());
        Set<String> tids = new LinkedHashSet<>(docs.size());
        for (SearchDocument doc : docs.values()) {
            tids.add(doc.attrs().get("thread_id"));
        }

        for (String tid : tids) {
            sb.append("&tid=");
            sb.append(tid);
        }

        try {
            threadsClient.execute(
                new HeaderAsyncRequestProducerSupplier(
                    new AsyncGetURIRequestProducerSupplier(new String(sb)),
                    new BasicHeader(YandexHeaders.X_YA_SERVICE_TICKET,
                        context
                            .server()
                            .filterSearchTvm2Ticket(request.corp()))),
                JsonAsyncTypesafeDomConsumerFactory.INTERNING_OK,
                session.httpSession().requestsListener()
                    .createContextGeneratorFor(threadsClient),
                callback);
            return true;
        } catch (URISyntaxException e) {
            failed(e);
            return false;
        }
    }

    @Override
    protected boolean filter(final Map<String, SearchDocument> docs) {
        DoubleFutureCallback<JsonObject, JsonObject> dc =
            new DoubleFutureCallback<>(
                new ThreadsFilterCallback(
                    session, docs));

        return filterSearch(docs, dc.first())
            && threadSearch(docs, dc.second());
    }

    @Override
    protected Documents buildResultDocuments() {
        return collector.documents();
    }

    public static final class ThreadDocument
        implements DocumentsGroup, Document
    {
        private final SearchDocument document;
        private final JsonMap envelope;
        private final JsonObject labels;
        private final String id;
        private final long ts;

        public ThreadDocument(
            final String id,
            final JsonMap envelope,
            final JsonObject labels,
            final SearchDocument document)
            throws JsonException
        {
            this.document = document;
            this.envelope = envelope;
            this.labels = labels;
            this.id = id;
            this.ts = envelope.getLong("receiveDate");
        }

        @Override
        public Iterator<Document> iterator() {
            return Collections.<Document>singletonList(this).iterator();
        }

        @Override
        public String id() {
            return this.id;
        }

        @Override
        public SearchDocument doc() {
            return document;
        }

        @Override
        public JsonMap envelope() {
            return envelope;
        }

        public JsonObject labels() {
            return labels;
        }

        @Override
        public void add(final Document doc) {
            throw new UnsupportedOperationException("Unable add to ThreadDoc");
        }

        @Override
        public Document best() {
            return this;
        }

        @Override
        public int size() {
            return 1;
        }

        public long ts() {
            return ts;
        }
    }

    private static final class ThreadComparator
        implements Comparator<ThreadDocument>
    {
        private static final ThreadComparator INSTANCE
            = new ThreadComparator();

        @Override
        public int compare(
            final ThreadDocument o1,
            final ThreadDocument o2)
        {
            return -Long.compare(o1.ts, o2.ts);
        }
    }

    public static final class ThreadedDocuments implements Documents {
        private final Map<String, ThreadDocument> documents;

        public ThreadedDocuments() {
            this.documents = new LinkedHashMap<>();
        }

        public ThreadedDocuments(final int size) {
            this.documents = new LinkedHashMap<>(size);
        }

        @Override
        public void add(
            final String groupId,
            final Document doc)
        {
            throw new UnsupportedOperationException("Unable add to ThreadDoc");
        }

        public void add(final ThreadDocument doc)
            throws JsonException
        {
            this.documents.put(doc.id(), doc);
        }

        public List<ThreadDocument> documents() {
            List<ThreadDocument> result = new ArrayList<>(documents.values());
            Collections.sort(result, ThreadComparator.INSTANCE);
            return result;
        }

        @Override
        public int size() {
            return this.documents.size();
        }

        @Override
        public void clear() {
            this.documents.clear();
        }

        @Override
        public List<? extends DocumentsGroup> sort() {
            return documents();
        }

        @Override
        public long relevant() {
            return 0;
        }

        @Override
        public void total(final long total) {
        }

        @Override
        public long total() {
            return this.documents.size();
        }

        @Override
        public Comparator<Document> comparator() {
            return null;
        }

        @Override
        public Iterator<DocumentsGroup> iterator() {
            return new ArrayList<DocumentsGroup>(documents.values()).iterator();
        }
    }

    private final class ThreadsFilterCallback
        extends AbstractSessionCallback<Entry<JsonObject, JsonObject>>
    {
        private final Map<String, SearchDocument> docs;

        public ThreadsFilterCallback(
            final SearchSession session,
            final Map<String, SearchDocument> docs)
        {
            super(session);

            this.docs = docs;
        }

        private Map<String, Map.Entry<JsonMap, JsonObject>> buildThreads(
            final JsonObject trRoot)
            throws JsonException
        {
            JsonObject threadsInfo = trRoot.asMap().get("threads_info");
            JsonList trEnvelopes = threadsInfo.get("envelopes").asList();
            JsonList trLabels = threadsInfo.get("threadLabels").asList();
            if (trEnvelopes.size() != trLabels.size()) {
                throw new JsonException(
                    "Thread labels list and envelopes have different sizes "
                        + trLabels.size() + " against " + trEnvelopes.size());
            }

            Map<String, Map.Entry<JsonMap, JsonObject>> result =
                new HashMap<>(trEnvelopes.size());

            Map<String, JsonMap> threadsMap = new HashMap<>();
            for (JsonObject threadObj : trLabels) {
                JsonMap threadLabel = threadObj.asMap();
                threadsMap.put(threadLabel.get("tid").asString(), threadLabel);
            }
            for (int i = 0; i < trEnvelopes.size(); i++) {
                JsonMap envelope = trEnvelopes.get(i).asMap();
                String threadId = envelope.get("threadId").asStringOrNull();
                JsonMap threadLabel = threadsMap.get(threadId);

                if (threadId == null || threadLabel == null) {
                    throw new JsonException(
                        "ThreadId null for "
                            + new HashMap<>(envelope).toString());
                }

                result.put(
                    threadId,
                    new AbstractMap.SimpleEntry<>(envelope, threadLabel));
            }

            return result;
        }

        @Override
        public synchronized void completed(
            final Entry<JsonObject, JsonObject> entry)
        {
            if (done) {
                return;
            }

            JsonObject fsRoot = entry.getKey();
            JsonObject trRoot = entry.getValue();

            StringBuilder mids = StringUtils.join(
                docs.keySet(),
                ',',
                "After filtration of mids ",
                0);
            mids.append(" the following mids left: ");
            Map<String, List<SearchDocument>> filteredThreads =
                new LinkedHashMap<>(docs.size());

            Function<String, List<SearchDocument>> listFactory =
                x -> new ArrayList<>();

            try {
                Map<String, Map.Entry<JsonMap, JsonObject>> threads
                    = buildThreads(trRoot);

                JsonList envelopes = fsRoot.get("envelopes").asList();

                for (int i = 0; i < envelopes.size(); ++i) {
                    JsonMap envelope = envelopes.get(i).asMap();
                    String mid = envelope.getString("mid");
                    String threadId = envelope.getString("threadId");

                    mids.append(mid);
                    mids.append(',');
                    SearchDocument doc = docs.get(mid);
                    if (doc == null || threadId == null) {
                        throw new JsonException(
                            "At envelope #" + i
                                + " unexpected mid found: " + mid);
                    }

                    filteredThreads
                        .computeIfAbsent(threadId, listFactory).add(doc);
                }

                ThreadedDocuments docs =
                    (ThreadedDocuments) collector.documents();

                for (Map.Entry<String, List<SearchDocument>> tentry :
                    filteredThreads.entrySet()) {

                    Map.Entry<JsonMap, JsonObject> thread =
                        threads.get(tentry.getKey());
                    if (thread == null) {
                        throw new JsonException(
                            "Thread not found "
                                + entry.getKey() + " threads "
                                + threads.toString());
                    }
                    JsonMap envelope = thread.getKey();
                    envelope.put(
                        "threadSize",
                        new JsonLong(envelope.getLong("threadCount")));
                    synchronized (collector) {
                        docs.add(
                            new ThreadDocument(
                                tentry.getKey(),
                                thread.getKey(),
                                thread.getValue(),
                                tentry.getValue().get(0)));
                    }
                }
            } catch (JsonException e) {
                failed(
                    new JsonException(
                        "Failed to parse: " + fsRoot + "\n" + trRoot, e));
                return;
            }
            mids.setLength(mids.length() - 1);
            session.httpSession().logger().fine(new String(mids));
            done = true;
            subrequestDone();
        }
    }
}


