package ru.yandex.msearch.proxy.api.async.mail.tabs.count;

import java.net.URISyntaxException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.collections.CollectionUtils;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.message.BasicHeader;

import ru.yandex.client.wmi.Labels;
import ru.yandex.client.wmi.LabelsConsumerFactory;
import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.ProxyParams;
import ru.yandex.msearch.proxy.api.async.mail.searcher.StrictPositionSearcherCallback;
import ru.yandex.msearch.proxy.api.async.mail.searcher.PlainSearcher;
import ru.yandex.msearch.proxy.api.async.mail.searcher.ProducerParallelSearcher;
import ru.yandex.msearch.proxy.api.async.mail.tabs.category.PositionPlainSearcher;
import ru.yandex.msearch.proxy.mail.SearchFilter;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.PrefixType;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;
import ru.yandex.search.rules.RequestParams;
import ru.yandex.search.rules.SearchInfo;
import ru.yandex.search.rules.SearchRequest;
import ru.yandex.search.rules.SearchRule;

public class FilterCountSearchRule
    implements SearchRule<DocumentsCount, RequestParams, SearchInfo>
{
    private static final int DEFAULT_FETCH_COUNT = 100;
    private static final int MALFORMED_THRESHOLD = 500;

    private final AsyncHttpServer server;
    private final ImmutableURIConfig labelsConfig;
    private final Map<String, SearchFilter> filters;
    private final Set<String> reindexUids;

    public FilterCountSearchRule(
        final AsyncHttpServer server)
    {
        this.server = server;
        this.filters = server.config().filtersConfig().filters();
        this.labelsConfig = server.config().labelsConfig();
        this.reindexUids = new HashSet<>();
//
//        try (BufferedReader reader =
//                 new BufferedReader(
//                     new InputStreamReader(
//                         new FileInputStream(
//                             new File(REINDEXED_USERS_FILE)))))
//        {
//            String line;
//            while ((line = reader.readLine()) != null) {
//                line = line.trim();
//                if (line.isEmpty()) {
//                    continue;
//                }
//
//                Long.parseLong(line);
//                reindexUids.add(line);
//            }
//        } catch (Exception e) {
//            server.logger().log(
//                Level.WARNING,
//                "Failed to load reindexed uids " + e.getMessage());
//        }
    }

    @Override
    public void execute(
        final SearchRequest<DocumentsCount, RequestParams, SearchInfo> request)
        throws HttpException
    {
        CgiParams params = request.cgiParams();

        String mdb = params.getString("mdb");

        PrefixType prefixType = server.searchMap().prefixType(mdb);
        Prefix suid;
        Prefix uid;
        Prefix prefix;
        String userIdField;
        if (mdb.equals("pg")) {
            suid = params.get(ProxyParams.SUID, null, prefixType);
            uid = params.get(ProxyParams.UID, prefixType);
            prefix = uid;
            userIdField = ProxyParams.UID;
        } else {
            suid = params.get(ProxyParams.SUID, prefixType);
            uid = params.get(ProxyParams.UID, null, prefixType);
            prefix = suid;
            userIdField = ProxyParams.SUID;
        }

        User user =
            new User(server.resolveService(mdb, prefix), prefix);

        LabelsCallback labelsCallback =
            new LabelsCallback(request, user, userIdField);

        boolean noLabelsAndFolders = params.getBoolean("nolaf", false);
        if (noLabelsAndFolders) {
            labelsCallback.completed(
                new Labels(
                    Collections.emptyMap(),
                    "_NO_LID_",
                    "_NO_LID_"));
        } else {
            boolean corp =
                ru.yandex.msearch.proxy.api.async.mail.SearchRequest.corp(prefix);

            QueryConstructor query = new QueryConstructor(
                new StringBuilder(labelsConfig.uri().toASCIIString())
                    .append(labelsConfig.firstCgiSeparator())
                    .append(ProxyParams.WMI_SUFFIX));
            query.append(ProxyParams.MDB, mdb);
            query.append(userIdField, prefix.toString());
            String labelsRequest = query.toString();

            AsyncClient labelsClient = server.labelsClient(corp);
            try {
                labelsClient.execute(
                    new HeaderAsyncRequestProducerSupplier(
                        new AsyncGetURIRequestProducerSupplier(labelsRequest),
                        new BasicHeader(YandexHeaders.X_YA_SERVICE_TICKET,
                            server.filterSearchTvm2Ticket(corp))),
                    LabelsConsumerFactory.OK,
                    request.session().listener()
                        .createContextGeneratorFor(labelsClient),
                    labelsCallback);
            } catch (URISyntaxException e) {
                throw new BadRequestException(e);
            }
        }
    }

    private class LabelsCallback
        extends AbstractFilterFutureCallback<Labels, DocumentsCount>
    {
        private final SearchRequest<DocumentsCount, RequestParams, SearchInfo> request;
        private final User user;
        private final String userIdField;

        public LabelsCallback(
            final SearchRequest<DocumentsCount, RequestParams, SearchInfo> request,
            final User user,
            final String userIdField)
        {
            super(request.callback());

            this.request = request;
            this.user = user;
            this.userIdField = userIdField;
        }

        private void process(final Labels labels)
            throws BadRequestException
        {
            CgiParams params = request.cgiParams();

            long position = params.getLong("position", -1L);
            PlainSearcher searcher;

            if (position > 0) {
                searcher =
                    new ConsistencyReportingPositionSearcher(
                        server,
                        request.session(),
                        user,
                        position);
            } else {
                searcher =
                    new ProducerParallelSearcher(
                        server,
                        request.session(),
                        user);
            }

            String filterStr = params.getString("filters").trim();
            if (filterStr.isEmpty()) {
                request.callback().completed(new DocumentsCount(new long[0]));
            } else {
                String[] requestFilters = filterStr.split(",");

                MultiFutureCallback<SearchResult> mfCallback =
                    new MultiFutureCallback<>(
                        new FilterCountCallback(request.callback()));

                int length = params.getInt("precise", DEFAULT_FETCH_COUNT);
                String queryBase =
                    "/search?group=mid&merge_func=none&sort=received_date"
                        + "&collector=pruning%28received_day_p%29&get=mid,received_date&length="
                        + length + "&user-id-field_disable=" + userIdField;
                boolean unread = params.getBoolean("unread", false);
                List<String> fids = params.getAll("fid");
                boolean excludeTrash = params.getBoolean("exclude-trash", true);
                for (String filter: requestFilters) {
                    StringBuilder text = new StringBuilder("(");
                    text.append(filters.get(filter).apply(labels));
                    text.append(')');

                    if (unread) {
                        text.append(" AND unread:1");
                    }

                    text.append(" AND hid:0");

                    if (!fids.isEmpty()) {
                        text.append(" AND (");
                        for (int i = 0; i < fids.size(); i++) {
                            if (i != 0) {
                                text.append(" OR ");
                            }

                            text.append("fid:");
                            text.append(fids.get(i));
                        }
                        text.append(')');
                    } else if (excludeTrash) {
                        text.append(" AND NOT folder_type:trash");
                    }

                    QueryConstructor qc = new QueryConstructor(queryBase);
                    qc.append("service", user.service());
                    qc.append("text", new String(text));
                    qc.append("prefix", user.prefix().toString());
                    qc.append("user-id-term_disable", user.prefix().toString());
                    searcher.search(
                        new BasicAsyncRequestProducerGenerator(qc.toString()),
                        mfCallback.newCallback());
                }

                mfCallback.done();
            }
        }

        @Override
        public void completed(final Labels labels) {
            try {
                process(labels);
            } catch (BadRequestException bre) {
                failed(bre);
            }
        }
    }

    private class FilterCountCallback
        implements FutureCallback<List<SearchResult>>
    {
        private final FutureCallback<? super DocumentsCount> callback;

        public FilterCountCallback(
            final FutureCallback<? super DocumentsCount> callback)
        {
            this.callback = callback;
        }

        @Override
        public void completed(final List<SearchResult> results) {
            long[] counts = new long[results.size()];

            for (int i = 0; i < results.size(); i++) {
                counts[i] = results.get(i).hitsCount();
            }

            callback.completed(new DocumentsCount(counts));
        }

        @Override
        public void failed(final Exception e) {
            callback.failed(e);
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }
    }

    private class ConsistencyReportingPositionSearcher
        extends PositionPlainSearcher
    {
        public ConsistencyReportingPositionSearcher(
            final AsyncHttpServer server,
            final ProxySession session,
            final User user,
            final long position)
            throws BadRequestException
        {
            super(server, session, user, position);
        }

        @Override
        protected FutureCallback<SearchResult> positionCallback(
            final FutureCallback<SearchResult> callback,
            final List<HttpHost> hosts)
        {
            return new ConsistencyReportingPositionCallback(
                callback,
                logger(),
                position,
                hosts.size());
        }
    }

    private class ConsistencyReportingPositionCallback
        extends StrictPositionSearcherCallback
    {
        private final Map<Long, SearchResult> counters;

        public ConsistencyReportingPositionCallback(
            final FutureCallback<SearchResult> callback,
            final PrefixedLogger logger,
            final long position,
            final int requests)
        {
            super(callback, logger, position, requests);

            this.counters = new ConcurrentHashMap<>(requests);
        }

        private Map<String, Long> parseResult(final SearchResult result) {
            Map<String, Long> resMap
                = new HashMap<>((int)result.hitsCount());
            for (SearchDocument doc: result.hitsArray()) {
                long ts =
                    Long.parseLong(
                        doc.attrs().getOrDefault("received_date", "0"));
                String mid = doc.attrs().get("mid");
                resMap.put(mid, ts);
            }

            return resMap;
        }

        @Override
        public void completed(final SearchResult result) {
            if (result != null && result.zooQueueId() > 0) {
                SearchResult current =
                    counters.putIfAbsent(result.zooQueueId(), result);

                if (current != null
                    && (Math.abs(result.hitsCount() - current.hitsCount())
                    >= MALFORMED_THRESHOLD))
                {
                    server.malformedIndex();
                    StringBuilder sb = new StringBuilder("MalformedIndex ");
                    sb.append("Host ");
                    sb.append(result.host().toHostString());
                    sb.append(" counter ");
                    sb.append(result.hitsCount());

                    sb.append(" Host ");
                    sb.append(current.host().toHostString());
                    sb.append(" counter ");
                    sb.append(current.hitsCount());
                    logger.info(sb.toString());
                    Map<String, Long> currentMap = parseResult(current);
                    Map<String, Long> resMap = parseResult(result);
                    Collection<String> disjunction =
                        CollectionUtils.disjunction(
                            currentMap.keySet(),
                            resMap.keySet());
                    long days5 = System.currentTimeMillis() - 432000000L;
                    StringBuilder freshSb = new StringBuilder("Fresh");
                    sb = new StringBuilder(disjunction.size() * 10);
                    sb.append("MalformedIndex ");
                    for (String mid: disjunction) {
                        sb.append(mid);

                        Long rd  = currentMap.get(mid);
                        if (rd == null) {
                            rd = resMap.get(mid);
                        }

                        if (rd != null && rd * 1000 >= days5) {
                            freshSb.append(mid);
                            freshSb.append(" ");
                        }

                        if (rd != null) {
                            sb.append(" (");
                            sb.append(rd);
                            sb.append(") ");
                        } else {
                            sb.append(" ");
                        }
                    }

                    logger.info(sb.toString());
                    if (freshSb.length() > 6) {
                        server.malformedIndexFresh();
                        logger.info(freshSb.toString());
                    }
                }
            }

            super.completed(result);
        }

        @Override
        public void failed(final Exception e) {
            RequestErrorType errorType =
                RequestErrorType.ERROR_CLASSIFIER.apply(e);

            if (errorType == RequestErrorType.NON_RETRIABLE) {
                if (requests.getAndSet(0) > 0) {
                    callback.failed(e);
                }

                return;
            }

            if (requests.decrementAndGet() == 0) {
                callback.failed(new NoDecentBackendException(position));
            }
        }

        @Override
        public void cancelled() {
            if (requests.decrementAndGet() == 0) {
                callback.cancelled();
            }
        }
    }
}
