package ru.yandex.msearch.proxy.api.async.senders;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.nio.protocol.BasicAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.client.producer.ProducerClient;
import ru.yandex.function.CorpUidPredicate;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.FixedMultiFutureCallback;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.parser.email.MailAliases;
import ru.yandex.parser.mail.senders.SenderInfo;
import ru.yandex.parser.mail.senders.SenderType;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.proxy.SearchResultConsumerFactory;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.result.SearchResult;

public class SendersHandler
    implements HttpAsyncRequestHandler<JsonObject>
{
    private static final SearchResult EMPTY_RESULT = SearchResult.EMPTY;
    private static final String SENDERS_SEARCH_REQUEST =
        "/search?skip-nulls&json-type=dollar&get=url,"
        + "senders_sent_count,senders_received_count,"
        + "senders_last_contacted,senders_names,senders_from_read_count,"
        + "pfilters_last_type,pfilters_spams,pfilters_hams,"
        + "pfilters_last_timestamp,user_ml_features,user_ml_embeddings,"
        + "tabpf_last_timestamp,tabpf_last_tab";

    private final AsyncHttpServer server;
    private final Long failoverDelay;

    public SendersHandler(final AsyncHttpServer server) {
        this.server = server;
        failoverDelay = server.config().sendersFailoverDelay();
    }

    @Override
    public void handle(
        final JsonObject payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session =
            new BasicProxySession(server, exchange, context);
        List<SendersRequest> requests = parseRequests(payload);
        Long senderUid = session.params().getLong("sender-uid", null);
        DoubleFutureCallback<
            List<SendersRequestWithResponse>,
            SearchResult> resultPrinter =
                new DoubleFutureCallback<>(
                    new ResultPrinter(
                        session,
                        JsonTypeExtractor.NORMAL.extract(session.params())));
        int maxNamesCount = session.params().getInt("names-max", 10);
        int size = requests.size();
        FixedMultiFutureCallback<SearchResult> sendersCallback =
            new FixedMultiFutureCallback<>(
                new MergingCallback(
                    resultPrinter.first(),
                    requests,
                    server,
                    maxNamesCount),
                size * 3);
        AsyncClient searchClient = server.searchClient().adjust(context);
        Supplier<? extends HttpClientContext> contextGenerator =
            session.listener().createContextGeneratorFor(searchClient);
        ProducerClient producerClient =
            server.producerClient().adjust(context);
        Supplier<? extends HttpClientContext> producerContextGenerator =
            session.listener().createContextGeneratorFor(producerClient);
        for (int i = 0, pos = 0; i < size; ++i, pos += 3) {
            SendersRequest request = requests.get(i);
            String queue;
            if (request.corp()) {
                queue = server.config().pgCorpQueue();
            } else {
                queue = server.config().pgQueue();
            }
            Prefix prefix = new LongPrefix(request.uid());

            UniversalSearchProxyRequestContext requestContext =
                new PlainUniversalSearchProxyRequestContext(
                    new User(queue, prefix),
                    null,
                    true,
                    searchClient,
                    session.logger());

            server.sequentialRequest(
                session,
                requestContext,
                new BasicAsyncRequestProducerGenerator(
                    sendersQuery(request, prefix)),
                failoverDelay,
                true,
                SearchResultConsumerFactory.OK,
                contextGenerator,
                sendersCallback.callback(pos),
                producerClient,
                producerContextGenerator);

            List<String> inReplyTo = request.inReplyTo();
            if (inReplyTo.isEmpty()) {
                sendersCallback.callback(pos + 1).completed(EMPTY_RESULT);
            } else {
                server.sequentialRequest(
                    session,
                    requestContext,
                    new BasicAsyncRequestProducerGenerator(
                        msgIdQuery(inReplyTo, prefix)),
                    failoverDelay,
                    true,
                    SearchResultConsumerFactory.OK,
                    contextGenerator,
                    sendersCallback.callback(pos + 1),
                    producerClient,
                    producerContextGenerator);
            }

            List<String> references = request.references();
            if (references.isEmpty()) {
                sendersCallback.callback(pos + 2).completed(EMPTY_RESULT);
            } else {
                server.sequentialRequest(
                    session,
                    requestContext,
                    new BasicAsyncRequestProducerGenerator(
                        msgIdQuery(references, prefix)),
                    failoverDelay,
                    true,
                    SearchResultConsumerFactory.OK,
                    contextGenerator,
                    sendersCallback.callback(pos + 2),
                    producerClient,
                    producerContextGenerator);
            }
        }
        if (senderUid == null) {
            resultPrinter.second().completed(SearchResult.EMPTY);
        } else {
            long uid = senderUid.longValue();
            String queue;
            if (CorpUidPredicate.INSTANCE.test(uid)) {
                queue = server.config().pgCorpQueue();
            } else {
                queue = server.config().pgQueue();
            }

            UniversalSearchProxyRequestContext requestContext =
                new PlainUniversalSearchProxyRequestContext(
                    new User(queue, new LongPrefix(uid)),
                    null,
                    true,
                    searchClient,
                    session.logger());

            server.sequentialRequest(
                session,
                requestContext,
                new BasicAsyncRequestProducerGenerator(
                    "/search?length=1&json-type=dollar"
                    + "&get=user_ml_features,user_ml_embeddings"
                    + "&text=url:user_ml_features_uid_" + uid
                    + "&prefix=" + uid),
                failoverDelay,
                true,
                SearchResultConsumerFactory.OK,
                contextGenerator,
                resultPrinter.second(),
                producerClient,
                producerContextGenerator);
        }
    }

    private static List<SendersRequest> parseRequests(final JsonObject request)
        throws HttpException
    {
        try {
            JsonList requests = request.get("requests").asList();
            int size = requests.size();
            List<SendersRequest> parsed = new ArrayList<>(size);
            for (int i = 0; i < size; ++i) {
                JsonObject senderRequest = requests.get(i);
                try {
                    parsed.add(new SendersRequest(senderRequest.asMap()));
                } catch (JsonException e) {
                    throw new BadRequestException(
                        "Failed to parse request:\n"
                        + JsonType.NORMAL.toString(senderRequest),
                        e);
                }
            }
            return parsed;
        } catch (JsonException e) {
            throw new BadRequestException(
                "Failed to parse requests:\n"
                + JsonType.NORMAL.toString(request),
                e);
        }
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        if (request instanceof HttpEntityEnclosingRequest) {
            return new JsonAsyncTypesafeDomConsumer(
                ((HttpEntityEnclosingRequest) request).getEntity(),
                StringCollectorsFactory.INSTANCE,
                BasicContainerFactory.INSTANCE);
        } else {
            throw new BadRequestException("Payload expected");
        }
    }

    @Override
    public String toString() {
        return "Performs batch search in senders index";
    }

    private static String sendersQuery(
        final SendersRequest request,
        final Prefix prefix)
        throws BadRequestException
    {
        StringBuilder sb = new StringBuilder("url:(\"user_ml_features_uid_");
        prefix.toStringBuilder(sb);
        sb.append('"');
        String senderHost = request.senderHost();
        if (senderHost != null) {
            senderHost = SearchRequestText.quoteEscape(senderHost);
        }
        for (SenderInfo senderInfo: request.sendersAddrs()) {
            SenderType senderType = senderInfo.type();
            String email = senderInfo.email();
            Map.Entry<String, String> pair =
                MailAliases.INSTANCE.parseAndNormalize(senderInfo.email());
            String escapedEmail = SearchRequestText.quoteEscape(email);
            sb.append(" OR \"");
            sb.append(senderType.sendersPrefix());
            sb.append(prefix);
            sb.append('_');
            sb.append(escapedEmail);
            sb.append('"');
            if (senderHost != null) {
                sb.append(" OR \"");
                sb.append(senderType.pfiltersPrefix());
                sb.append(prefix);
                sb.append('_');
                sb.append(escapedEmail);
                sb.append('/');
                sb.append(senderHost);
                sb.append("\" OR \"");
                sb.append(senderType.tabPfPrefix());
                sb.append(prefix);
                sb.append('_');
                sb.append(escapedEmail);
                sb.append('/');
                sb.append(senderHost);
                sb.append('"');
            }
        }
        for (SenderInfo senderInfo: request.sendersDomains()) {
            sb.append(" OR \"");
            sb.append(senderInfo.type().sendersDomainPrefix());
            sb.append(prefix);
            sb.append('_');
            sb.append(SearchRequestText.quoteEscape(senderInfo.email()));
            sb.append('"');
        }
        sb.append(')');
        QueryConstructor query = new QueryConstructor(SENDERS_SEARCH_REQUEST);
        query.append("prefix", prefix.toString());
        query.append("text", new String(sb));
        return query.toString();
    }

    private static String msgIdQuery(
        final List<String> msgIds,
        final Prefix prefix)
        throws BadRequestException
    {
        StringBuilder sb = new StringBuilder("msg_id:(");
        boolean empty = true;
        for (String msgId: msgIds) {
            if (empty) {
                empty = false;
            } else {
                sb.append(" OR ");
            }
            sb.append('"');
            sb.append(SearchRequestText.quoteEscape(msgId));
            sb.append('"');
        }
        sb.append(") AND NOT folder_type:(spam OR trash)");
        QueryConstructor query =
            new QueryConstructor("/search?json-type=dollar&length=0");
        query.append("prefix", prefix.toString());
        query.append("text", new String(sb));
        return query.toString();
    }

    private static class MergingCallback
        extends AbstractFilterFutureCallback<
            List<SearchResult>,
            List<SendersRequestWithResponse>>
    {
        private final List<SendersRequest> requests;
        private final AsyncHttpServer server;
        private final int maxNamesCount;

        MergingCallback(
            final FutureCallback<List<SendersRequestWithResponse>> callback,
            final List<SendersRequest> requests,
            final AsyncHttpServer server,
            final int maxNamesCount)
        {
            super(callback);
            this.requests = requests;
            this.server = server;
            this.maxNamesCount = maxNamesCount;
        }

        @Override
        public void completed(final List<SearchResult> results) {
            int size = requests.size();
            List<SendersRequestWithResponse> responses = new ArrayList<>(size);
            for (int i = 0, pos = 0; i < size; ++i, pos += 3) {
                SendersDocs docs = new SendersDocs(
                    results.get(pos),
                    results.get(pos + 1),
                    results.get(pos + 2));
                SendersRequest request = requests.get(i);
                responses.add(
                    new SendersRequestWithResponse(
                        request,
                        new SendersResponse(
                            docs,
                            request,
                            maxNamesCount)));

                SenderType senderType = docs.senderType();
                if (senderType != null) {
                    server.senderType(senderType);
                }

                senderType = docs.domainSenderType();
                if (senderType != null) {
                    server.senderDomainType(senderType);
                }

                senderType = docs.pfiltersSenderType();
                if (senderType != null) {
                    server.pfilterSenderType(senderType);
                }

                senderType = docs.tabPfSenderType();
                if (senderType != null) {
                    server.tabPfSenderType(senderType);
                }
            }
            callback.completed(responses);
        }
    }
}
