package ru.yandex.msearch.proxy.api.async.senders;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import ru.yandex.function.CorpUidPredicate;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.parser.mail.senders.SenderInfo;
import ru.yandex.parser.mail.senders.SenderType;

public class SendersRequest implements JsonValue {
    private static final String FAILED_TO_PARSE = "Failed to parse ";

    private static final String SUID = "suid";
    private static final String UID = "uid";
    private static final String EMAIL = "email";
    private static final String DOMAIN = "domain";
    private static final String SENDER_HOST = "sender-host";
    private static final String IN_REPLY_TO = "in-reply-to";
    private static final String REFERENCES = "references";

    private final long uid;
    private final Long suid;
    private final String email;
    private final String domain;
    private final String senderHost;
    private final List<SenderInfo> sendersAddrs;
    private final List<SenderInfo> sendersDomains;
    private final List<String> inReplyTo;
    private final List<String> references;

    public SendersRequest(final JsonMap request)
        throws JsonException
    {
        uid = request.getLong(UID);
        suid = request.getLong(SUID, null);
        senderHost = request.getString(SENDER_HOST, null);
        sendersAddrs = new ArrayList<>();
        sendersDomains = new ArrayList<>();
        for (SenderType type: SenderType.values()) {
            String email = request.getString(type.emailField(), null);
            if (email != null && !email.isEmpty()) {
                sendersAddrs.add(new SenderInfo(type, email));
            }
            String domain = request.getString(type.domainField(), null);
            if (domain != null && !domain.isEmpty()) {
                sendersDomains.add(new SenderInfo(type, domain));
            }
        }

        if (!sendersAddrs.isEmpty()
            && sendersAddrs.get(0).type() == SenderType.FROM)
        {
            email = sendersAddrs.get(0).email();
        } else {
            email = null;
        }

        if (!sendersDomains.isEmpty()
            && sendersDomains.get(0).type() == SenderType.FROM)
        {
            domain = sendersDomains.get(0).email();
        } else {
            domain = null;
        }

        inReplyTo = parseStringList(request.getListOrNull(IN_REPLY_TO));
        references = parseStringList(request.getListOrNull(REFERENCES));
    }

    private static List<String> parseStringList(final JsonList list)
        throws JsonException
    {
        if (list == null) {
            return Collections.emptyList();
        }
        int size = list.size();
        if (size == 0) {
            return Collections.emptyList();
        } else {
            List<String> parsedList = new ArrayList<>(size);
            for (int i = 0; i < size; ++i) {
                parsedList.add(list.get(i).asString());
            }
            return parsedList;
        }
    }

    public Long suid() {
        return suid;
    }

    public long uid() {
        return uid;
    }

    public boolean corp() {
        return CorpUidPredicate.INSTANCE.test(uid);
    }

    public String email() {
        return email;
    }

    public String domain() {
        return domain;
    }

    public String senderHost() {
        return senderHost;
    }

    public List<SenderInfo> sendersAddrs() {
        return sendersAddrs;
    }

    public List<SenderInfo> sendersDomains() {
        return sendersDomains;
    }

    public List<String> inReplyTo() {
        return inReplyTo;
    }

    public List<String> references() {
        return references;
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.startObject();
        writer.key(SUID);
        writer.value(suid);
        writer.key(UID);
        writer.value(uid);
        if (senderHost != null) {
            writer.key(SENDER_HOST);
            writer.value(senderHost);
        }
        for (SenderInfo senderInfo: sendersAddrs) {
            writer.key(senderInfo.type().emailField());
            writer.value(senderInfo.email());
        }
        for (SenderInfo senderInfo: sendersDomains) {
            writer.key(senderInfo.type().domainField());
            writer.value(senderInfo.email());
        }

        int size = inReplyTo.size();
        if (size > 0) {
            writer.key(IN_REPLY_TO);
            writer.startArray();
            for (int i = 0; i < size; ++i) {
                writer.value(inReplyTo.get(i));
            }
            writer.endArray();
        }

        size = references.size();
        if (size > 0) {
            writer.key(REFERENCES);
            writer.startArray();
            for (int i = 0; i < size; ++i) {
                writer.value(references.get(i));
            }
            writer.endArray();
        }
        writer.endObject();
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder();
        sb.append("SendersRequest: suid: ");
        sb.append(suid);

        sb.append(", uid: ");
        sb.append(uid);

        sb.append(", sender-host: ");
        sb.append(senderHost);

        sb.append(", senders-addr: ");
        sb.append(sendersAddrs);

        sb.append(", senders-domains: ");
        sb.append(sendersDomains);

        sb.append(", in-reply-tos: ");
        sb.append(inReplyTo);

        sb.append(", references: ");
        sb.append(references);

        return new String(sb);
    }
}
