package ru.yandex.msearch.proxy.api.async.senders;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.parser.mail.senders.SenderType;

public class SendersResponse implements JsonValue {
    private final SendersRequest request;
    //senders
    private final long sentCount;
    private final long receivedCount;
    private final long readCount;
    private final long lastContacted;
    private final List<String> names;
    private final String senderType;
    //senders domain
    private final long domainSentCount;
    private final long domainReceivedCount;
    private final long domainReadCount;
    private final long domainLastContacted;
    private final String domainSenderType;
    //pfilters
    private final String pfiltersLastType;
    private final long pfiltersLastTime;
    private final long pfiltersSpams;
    private final long pfiltersHams;
    private final String pfiltersSenderType;
    //tabpf
    private final String tabPfLastTab;
    private final double tabPfLastTime;

    private final JsonObject userMlFeatures;
    private final JsonObject userMlEmbeddings;
    private final boolean inReplyToMatches;
    private final boolean referencesMatches;

    public SendersResponse(
        final SendersDocs docs,
        final SendersRequest request,
        final int maxNamesCount)
    {
        this.request = request;
        Map<String, String> attrs = docs.sendersDoc().attrs();
        sentCount = parseLong(attrs.get("senders_sent_count"), 0);
        receivedCount = parseLong(attrs.get("senders_received_count"), 0);
        readCount = parseLong(attrs.get("senders_from_read_count"), 0);
        lastContacted = parseLong(attrs.get("senders_last_contacted"), 0);
        names =
            parseNames(attrs.get("senders_names"), maxNamesCount);
        senderType = senderTypeName(docs.senderType());

        attrs = docs.domainSendersDoc().attrs();
        domainSentCount = parseLong(attrs.get("senders_sent_count"), 0);
        domainReceivedCount =
            parseLong(attrs.get("senders_received_count"), 0);
        domainReadCount =
            parseLong(attrs.get("senders_from_read_count"), 0);
        domainLastContacted =
            parseLong(attrs.get("senders_last_contacted"), 0);
        domainSenderType = senderTypeName(docs.domainSenderType());

        attrs = docs.pfiltersDoc().attrs();
        pfiltersLastType = attrs.get("pfilters_last_type");
        pfiltersLastTime = parseLong(attrs.get("pfilters_last_timestamp"), 0);
        pfiltersSpams = parseLong(attrs.get("pfilters_spams"), 0);
        pfiltersHams = parseLong(attrs.get("pfilters_hams"), 0);
        pfiltersSenderType = senderTypeName(docs.pfiltersSenderType());

        attrs = docs.tabPfDoc().attrs();
        tabPfLastTab = attrs.get("tabpf_last_tab");
        tabPfLastTime = parseDouble(attrs.get("tabpf_last_timestamp"), 0);

        userMlFeatures = docs.userMlFeatures();
        userMlEmbeddings = docs.userMlEmbeddings();

        inReplyToMatches = docs.inReplyToMatches();
        referencesMatches = docs.referencesMatches();
    }

    private static long parseLong(final String str, final long def) {
        if (str != null) {
            try {
                return Long.parseLong(str);
            } catch (NumberFormatException ignore) {
            }
        }
        return def;
    }

    private static double parseDouble(final String str, final double def) {
        if (str != null) {
            try {
                return Double.parseDouble(str);
            } catch (NumberFormatException ignore) {
            }
        }
        return def;
    }

    private static String senderTypeName(final SenderType type) {
        if (type == null) {
            return null;
        } else {
            return type.typeName();
        }
    }

    private static List<String> parseNames(
        final String str,
        final int maxNames)
    {
        List<String> names = null;
        if (str != null) {
            names = new ArrayList<>(maxNames);
            int prevSep = str.indexOf('\n') + 1;//skip firt line
            int sep = str.indexOf('\n', prevSep);
            while (sep < str.length() && sep != -1 && names.size() < maxNames) {
                names.add(str.substring(prevSep, sep));
                prevSep = sep + 1;
                sep = str.indexOf('\n', prevSep);
            }
            if (sep == -1 && prevSep != -1) {
                //add last token
                names.add(str.substring(prevSep));
            }
        }
        if (names == null || names.isEmpty()) {
            names = Collections.emptyList();
        }
        return names;
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.startObject();
        if (request.email() != null) {
            writer.key("sent_count");
            writer.value(sentCount);
            writer.key("received_count");
            writer.value(receivedCount);
            writer.key("received_read_count");
            writer.value(readCount);
            writer.key("last_contacted");
            writer.value(lastContacted);
            writer.key("names");
            writer.value(names);
            writer.key("sender_type");
            writer.value(senderType);
        }
        if (request.domain() != null) {
            writer.key("domain_send_count");
            writer.value(domainSentCount);
            writer.key("domain_received_count");
            writer.value(domainReceivedCount);
            writer.key("domain_received_read_count");
            writer.value(domainReadCount);
            writer.key("domain_sender_type");
            writer.value(domainSenderType);
        }
        if (request.senderHost() != null) {
            writer.key("pfilters_last_type");
            writer.value(pfiltersLastType);
            writer.key("pfilters_last_time");
            writer.value(pfiltersLastTime);
            writer.key("pfilters_spams");
            writer.value(pfiltersSpams);
            writer.key("pfilters_hams");
            writer.value(pfiltersHams);
            writer.key("pfilters_sender_type");
            writer.value(pfiltersSenderType);

            if (tabPfLastTab != null) {
                writer.key("tabpf_last_timestamp");
                writer.value(tabPfLastTime);
                writer.key("tabpf_last_tab");
                writer.value(tabPfLastTab);
            }
        }
        if (userMlFeatures != JsonNull.INSTANCE) {
            writer.key("user_ml_features");
            userMlFeatures.writeValue(writer);
        }
        if (userMlEmbeddings != JsonNull.INSTANCE) {
            writer.key("user_ml_embeddings");
            userMlEmbeddings.writeValue(writer);
        }
        if (inReplyToMatches) {
            writer.key("in_reply_to_matches");
            writer.value(true);
        }
        if (referencesMatches) {
            writer.key("references_matches");
            writer.value(true);
        }
        writer.endObject();
    }
}

