package ru.yandex.msearch.proxy.api.async.so;

import java.io.IOException;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.BasicAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.client.tvm2.UserAuthResult;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.proxy.StaticEntitySendingCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.BasicSession;
import ru.yandex.msearch.proxy.api.async.Session;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;

/**
 * 1) Get hunter stats
 * 2) If report_timeout passed from last update, count records
 * 3) Update hunter stats
 */
public class BugBountyGetScore implements HttpAsyncRequestHandler<HttpRequest>
{
    private static final long REPORT_TIMEOUT = 24 * 3600;

    private final AsyncHttpServer server;

    public BugBountyGetScore(AsyncHttpServer server) {
        this.server = server;
    }

    @Override
    public HttpAsyncRequestConsumer<HttpRequest> processRequest(
        final HttpRequest httpRequest,
        final HttpContext httpContext)
    {
        return new BasicAsyncRequestConsumer();
    }

    @Override
    public void handle(
        HttpRequest request,
        HttpAsyncExchange exchange,
        HttpContext context)
        throws HttpException, IOException
    {
        UserAuthResult authResult = server.checkTvmUserAuthorization(request);
        if (authResult.errorDescription() != null) {
            throw new BadRequestException(authResult.errorDescription());
        }
        long uid = authResult.ticket().getUids()[0];

        Session session = new BasicSession(server, exchange, context);

        QueryConstructor query = new QueryConstructor("/search?", false);
        query.append("prefix", uid);
        query.append("get", "*");
        query.append("skip-nulls", "true");
        query.append("json-type", "dollar");
        query.append("text", "url:" + "sobb_hunter_" + uid);
        makeQuery(
            uid,
            query.toString(),
            session,
            new HunterStatsParsingCallback(session, uid));
    }

    private void makeQuery(
        long uid,
        String query,
        ProxySession session,
        FutureCallback<? super JsonObject> callback)
    {
        AsyncClient client =
            server.searchClient().adjust(session.context());
        server.sequentialRequest(
            session,
            new PlainUniversalSearchProxyRequestContext(
                new User(
                        server.config().pgQueue(),
                    new LongPrefix(uid)),
                null,
                true,
                client,
                session.logger()),
            new BasicAsyncRequestProducerGenerator(query),
            server.config().failoverSearchDelay(),
            false,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            callback);
    }

    private class HunterStatsParsingCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final long uid;

        protected HunterStatsParsingCallback(ProxySession session, long uid) {
            super(session);
            this.uid = uid;
        }

        @Override
        public void completed(JsonObject docs) {
            try {
                JsonList hits = docs.get("hitsArray").asList();
                long ctime = System.currentTimeMillis() / 1000;
                if (hits.size() == 1) {
                    JsonMap record = hits.get(0).asMap();
                    long lastUpdate = record.getLong("sobb_report_time", 0L);
                    if (lastUpdate + REPORT_TIMEOUT > ctime) {
                        session.response(
                            HttpStatus.SC_OK,
                            new NStringEntity(
                                replyDoc(
                                    record.getLong("sobb_inbox"),
                                    lastUpdate),
                                ContentType.APPLICATION_JSON));
                        return;
                    }
                }
                QueryConstructor query =
                    new QueryConstructor("/search?", false);
                try {
                    query.append("prefix", uid);
                    query.append("get", "url");
                    query.append("skip-nulls", "true");
                    query.append("json-type", "dollar");
                    query.append("text", "url:sobb_record_" + uid + "_*");
                    query.append("postfilter", "sobb_inbox >= 1");
                    query.append("length", 0);
                } catch (BadRequestException e) {
                    failed(e);
                    return;
                }

                makeQuery(
                    uid,
                    query.toString(),
                    session,
                    new RecordsParsingCallback(session, ctime, uid));
            } catch (JsonException e) {
                failed(e);
            }
        }
    }


    private class RecordsParsingCallback
            extends AbstractProxySessionCallback<JsonObject> {

        private final long ctime;
        private final long uid;

        protected RecordsParsingCallback(
            ProxySession session,
            long ctime,
            long uid)
        {
            super(session);
            this.ctime = ctime;
            this.uid = uid;
        }

        @Override
        public void completed(JsonObject response) {
            long realInboxed;
            try {
                realInboxed = response.get("hitsCount").asLong();
            } catch (JsonBadCastException e) {
                failed(e);
                return;
            }

            AsyncClient client =
                server.producerStoreClient().adjust(session.context());
            client.execute(
                server.config().producerStoreConfig().host(),
                new BasicAsyncRequestProducerGenerator(
                    "/update?prefix=" + uid +
                        "&service=" + server.config().pgQueue(),
                    updateDoc(ctime, realInboxed),
                    ContentType.APPLICATION_JSON),
                EmptyAsyncConsumerFactory.OK,
                session.listener().createContextGeneratorFor(client),
                new StaticEntitySendingCallback<>(
                    session,
                    HttpStatus.SC_OK,
                    new NStringEntity(
                        replyDoc(realInboxed, ctime),
                        ContentType.APPLICATION_JSON)));
        }




        private String updateDoc(long ctime, long inboxed) {
            return String.format(
                "{\"prefix\":%1$d,\"AddIfNotExists\":true,\"docs\":[{"
                    + "\"sobb_report_time\":\"%2$d\","
                    + "\"sobb_inbox\":%3$d,"
                    + "\"url\":\"sobb_hunter_%1$d\"}]}", uid, ctime, inboxed);
        }
    }

    private static String replyDoc(long inboxed, long time) {
        return "{"
           + "\n  \"inboxed_targets\": " + inboxed
           + ",\n  \"report_time\": " + time + "\n}\n";
    }
}
