package ru.yandex.msearch.proxy.api.async.so;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.BasicAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.cache.async.AsyncCache;
import ru.yandex.cache.async.AsyncCacheResult;
import ru.yandex.cache.async.AsyncLoader;
import ru.yandex.cache.async.ConcurrentLinkedHashMapAsyncStorage;
import ru.yandex.cache.async.ConcurrentLinkedHashMapAsyncStorageStater;
import ru.yandex.cache.async.ConstAsyncCacheTtlCalculator;
import ru.yandex.cache.async.LongArrayWeigher;
import ru.yandex.cache.async.StringWeigher;
import ru.yandex.cache.async.UnwrappingFutureCallback;
import ru.yandex.collection.ComparableLongPair;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.config.ImmutableDkimStatsConfig;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.proxy.SearchResultConsumerFactory;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

public class GetDkimStats implements HttpAsyncRequestHandler<HttpRequest> {
    private final AsyncHttpServer server;
    private final AsyncCache<String, long[], Context, RuntimeException>
        cache;

    public GetDkimStats(
        final AsyncHttpServer server,
        final ImmutableDkimStatsConfig config)
    {
        this.server = server;
        ConcurrentLinkedHashMapAsyncStorage<String, long[]> storage =
            new ConcurrentLinkedHashMapAsyncStorage<>(
                config.cacheCapacity(),
                Runtime.getRuntime().availableProcessors(),
                StringWeigher.INSTANCE,
                LongArrayWeigher.INSTANCE);
        cache = new AsyncCache<>(
            storage,
            new ConstAsyncCacheTtlCalculator(config.cacheTtl()),
            new Loader(server, config.failoverDelay()));
        server.registerStater(
            new ConcurrentLinkedHashMapAsyncStorageStater(
                storage,
                false,
                "dkim-stats-",
                "dkim",
                null));
    }

    @Override
    public HttpAsyncRequestConsumer<HttpRequest> processRequest(
        final HttpRequest request,
        final HttpContext context)
    {
        return new BasicAsyncRequestConsumer();
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session =
            new BasicProxySession(server, exchange, context);
        Context requestContext = new Context(session);
        StringBuilder sb = new StringBuilder();
        sb.append(requestContext.selector.from());
        sb.append('/');
        sb.append(requestContext.selector.day());
        sb.append('/');
        sb.append(requestContext.topDomainsCount);
        sb.append('/');
        sb.append(requestContext.days);
        for (String dkimDomain: requestContext.selector.dkimDomains()) {
            sb.append('/');
            sb.append(dkimDomain);
        }
        cache.get(
            new String(sb),
            requestContext,
            new UnwrappingFutureCallback<>(
                new ResultPrinter(server, session),
                server.dkimStatsCacheHitType(),
                session.logger()));
    }

    private static class ResultPrinter
        extends AbstractProxySessionCallback<long[]>
    {
        private final AsyncHttpServer server;
        private final JsonType jsonType;

        ResultPrinter(
            final AsyncHttpServer server,
            final ProxySession session)
            throws HttpException
        {
            super(session);
            this.server = server;
            jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        }

        @Override
        public void completed(final long[] stats) {
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = jsonType.create(sbw)) {
                writer.startObject();
                writer.key("dkim_stats");
                writer.startObject();
                writer.key("total");
                writer.value(stats[0]);
                writer.key("dkimless");
                writer.value(stats[1]);
                long bestDomain = stats[2];
                if (bestDomain >= 0L) {
                    writer.key("best_domain");
                    writer.value(bestDomain);
                }
                writer.key("top_domains");
                writer.startArray();
                for (int i = 3; i < stats.length; ++i) {
                    writer.value(stats[i]);
                }
                writer.endArray();
                writer.endObject();
                writer.endObject();
            } catch (IOException e) {
                session.handleException(new ServiceUnavailableException(e));
                return;
            }
            session.response(
                HttpStatus.SC_OK,
                new NStringEntity(
                    sbw.toString(),
                    ContentType.APPLICATION_JSON.withCharset(
                        session.acceptedCharset())));
        }
    }

    private static class Loader
        implements AsyncLoader<String, long[], Context>
    {
        private final AsyncHttpServer server;
        private final Long failoverDelay;

        Loader(final AsyncHttpServer server, final Long failoverDelay) {
            this.server = server;
            this.failoverDelay = failoverDelay;
        }

        private void load(
            final Context context,
            final FutureCallback<? super long[]> callback)
            throws HttpException
        {
            DkimStatsSelector selector = context.selector;
            String from = SearchRequestText.fullEscape(selector.from(), false);
            StringBuilder sb = new StringBuilder("url:(");
            for (int i = 0; i < context.days; ++i) {
                if (i != 0) {
                    sb.append(" OR ");
                }
                sb.append("dkim_stat_");
                sb.append(from);
                sb.append('/');
                sb.append(selector.day() - i);
                sb.append('/');
                sb.append('*');
            }
            sb.append(')');

            QueryConstructor query =
                new QueryConstructor(
                    "/search?dkim-stats&json-type=dollar&get=url,dkim_stat_hams");
            query.append("from", selector.from());
            query.append("prefix", selector.prefix().prefix());
            query.append("text", new String(sb));

            ProxySession session = context.session;
            AsyncClient searchClient =
                server.searchClient().adjust(session.context());
            server.sequentialRequest(
                session,
                new PlainUniversalSearchProxyRequestContext(
                    new User(server.config().pgQueue(), selector.prefix()),
                    null,
                    true,
                    searchClient,
                    session.logger()),
                new BasicAsyncRequestProducerGenerator(query.toString()),
                failoverDelay,
                true,
                SearchResultConsumerFactory.OK,
                session.listener().createContextGeneratorFor(searchClient),
                new Callback(callback, selector, context.topDomainsCount));
        }

        @Override
        public void load(
            final String uri,
            final Context context,
            final FutureCallback<? super long[]> callback)
        {
            try {
                load(context, callback);
            } catch (HttpException e) {
                callback.failed(e);
            }
        }
    }

    private static class Callback
        extends AbstractFilterFutureCallback<SearchResult, long[]>
    {
        private final DkimStatsSelector selector;
        private final int topDomainsCount;

        Callback(
            final FutureCallback<? super long[]> callback,
            final DkimStatsSelector selector,
            final int topDomainsCount)
        {
            super(callback);
            this.selector = selector;
            this.topDomainsCount = topDomainsCount;
        }

        private void processResult(final SearchResult result)
            throws HttpException
        {
            long total = 0L;
            long dkimless = 0L;
            Map<String, long[]> counters = new HashMap<>();
            for (SearchDocument doc: result.hitsArray()) {
                String url = doc.attrs().getOrDefault("url", "");
                int idx = url.lastIndexOf('/');
                if (idx == -1) {
                    throw new ServiceUnavailableException(
                        "Malformed doc url: " + doc.attrs());
                }
                long count;
                try {
                    count = Long.parseLong(doc.attrs().get("dkim_stat_hams"));
                } catch (RuntimeException e) {
                    throw new ServiceUnavailableException(
                        "Malformed dkim_stat_hams: " + doc);
                }
                String domain = url.substring(idx + 1);
                if (domain.equals("none")) {
                    dkimless += count;
                } else if (domain.equals("total")) {
                    total += count;
                } else {
                    long[] counter =
                        counters.computeIfAbsent(domain, x -> new long[1]);
                    counter[0] += count;
                }
            }
            int size = counters.size();
            List<ComparableLongPair<String>> topDomains =
                new ArrayList<>(size);
            for (Map.Entry<String, long[]> entry: counters.entrySet()) {
                topDomains.add(
                    new ComparableLongPair<>(
                        entry.getValue()[0],
                        entry.getKey()));
            }
            topDomains.sort(Comparator.reverseOrder());
            Set<String> dkimDomains = selector.dkimDomains();
            long bestDomain = 0L;
            for (int i = 0; i < size; ++i) {
                ComparableLongPair<String> topDomain = topDomains.get(i);
                if (dkimDomains.contains(topDomain.second())) {
                    bestDomain = topDomain.first();
                    break;
                }
            }
            int topDomainsLen = Math.min(topDomainsCount, size);
            long[] stats = new long[topDomainsLen + 3];
            stats[0] = total;
            stats[1] = dkimless;
            if (dkimDomains.isEmpty()) {
                stats[2] = -1L;
            } else {
                stats[2] = bestDomain;
            }
            for (int i = 0; i < topDomainsLen; ++i) {
                stats[i + 3] = topDomains.get(i).first();
            }
            callback.completed(stats);
        }

        @Override
        public void completed(final SearchResult result) {
            try {
                processResult(result);
            } catch (HttpException e) {
                super.failed(e);
            }
        }
    }

    private static class Context {
        private final ProxySession session;
        private final DkimStatsSelector selector;
        private final int topDomainsCount;
        private final int days;

        Context(final ProxySession session) throws HttpException {
            this.session = session;
            selector = new DkimStatsSelector(session);
            topDomainsCount = session.params().get(
                "top-domains-count",
                DkimStatsSelector.DEFAULT_TOP_DOMAINS_COUNT,
                NonNegativeIntegerValidator.INSTANCE);
            days = session.params().get(
                "days",
                DkimStatsSelector.DEFAULT_DAYS_COUNT,
                PositiveIntegerValidator.INSTANCE);
        }
    }
}

