package ru.yandex.msearch.proxy.api.async.so;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.BinaryOperator;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.protocol.BasicAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.concurrent.ConcurrentAccumulator;
import ru.yandex.executor.concurrent.ConcurrentExecutor;
import ru.yandex.executor.concurrent.ConcurrentExecutorStats;
import ru.yandex.executor.concurrent.CountersTasksProvider;
import ru.yandex.executor.concurrent.TasksExecutor;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.DuplexFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.stater.GolovanChart;
import ru.yandex.stater.GolovanChartGroup;
import ru.yandex.stater.GolovanPanel;
import ru.yandex.stater.GolovanSignal;
import ru.yandex.stater.ImmutableGolovanPanelConfig;
import ru.yandex.stater.ImmutableStatersConfig;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;
import ru.yandex.util.timesource.TimeSource;

public class UpdateDkimStats
    extends CountersTasksProvider<String, UpdateDkimStatsTask>
    implements
        GenericAutoCloseable<RuntimeException>,
        HttpAsyncRequestHandler<HttpRequest>,
        Stater,
        TasksExecutor<UpdateDkimStatsTask>
{
    private static final BinaryOperator<UpdateDkimStatsTask> COUNTERS_MERGER =
        (newTask, oldTask) ->
            oldTask == null ? null : oldTask.updateWith(newTask);

    private final LongAdder failures = new LongAdder();
    private final LongAdder hardFailures = new LongAdder();
    private final ConcurrentAccumulator<UpdateStat> updateStats =
        new ConcurrentAccumulator<>(
            UpdateStatAccumulator.INSTANCE,
            UpdateStat.ZERO);
    private final AsyncHttpServer server;
    private final Logger logger;
    private final String service;
    private final HttpHost host;
    private final AsyncClient client;
    private final ImmutableStatersConfig statersConfig;
    private final ConcurrentExecutor executor;
    private final long retryInterval;

    public UpdateDkimStats(final AsyncHttpServer server) {
        super(COUNTERS_MERGER, server.config().workers() << 1);
        this.server = server;
        logger = server.logger();
        service = server.config().pgQueue();
        host = server.config().producerStoreConfig().host();
        client = server.producerStoreClient();
        statersConfig = client.statersConfig();
        int updateConcurrency =
            server.config().dkimStatsConfig().updateConcurrency();
        executor = new ConcurrentExecutor(
            server.getThreadGroup(),
            server.getName() + "-DkimUpdate",
            this,
            this,
            updateConcurrency,
            updateConcurrency * 2);
        retryInterval = server.config().dkimStatsConfig().retryInterval();
        server.registerStater(this);
    }

    public void start() {
        executor.start();
    }

    @Override
    public void close() {
        executor.close();
        super.close();
    }

    // HttpAsyncRequestHandler implementation
    @Override
    public HttpAsyncRequestConsumer<HttpRequest> processRequest(
        final HttpRequest request,
        final HttpContext context)
    {
        return new BasicAsyncRequestConsumer();
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session =
            new BasicProxySession(server, exchange, context);
        UpdateDkimStatsTask task = new UpdateDkimStatsTask(session);
        String key = task.key();
        addCounters(key, task);
        session.response(HttpStatus.SC_OK);
    }

    // TasksExecutor implementation
    @Override
    public long retryIntervalFor(final Exception e) {
        return retryInterval;
    }

    @Override
    public void execute(
        final UpdateDkimStatsTask task,
        final FutureCallback<Void> callback)
    {
        logger.info(
            "Executing task " + task + " ( " + task.tasksCount() + ' ' + ')');
        try {
            processTask(
                task,
                new DuplexFutureCallback<>(
                    callback,
                    new StatingFutureCallback(task)));
        } catch (Throwable t) {
            logger.log(Level.WARNING, "Failed to process task " + task, t);
            callback.completed(null);
            hardFailures.increment();
        }
    }

    // Stater implementation
    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        UpdateStat stat = updateStats.get();
        ConcurrentExecutorStats executorStats = executor.stats();
        int size = size();
        long failures = this.failures.longValue();
        long hardFailures = this.hardFailures.longValue();
        statsConsumer.stat("dkim-stats-update-queue_ammm", size);
        statsConsumer.stat("dkim-stats-update-count_dmmm", stat.updatesCount);
        statsConsumer.stat(
            "dkim-stats-update-tasks-count_dmmm",
            stat.tasksCount);
        statsConsumer.stat("dkim-stats-update-failures_dmmm", failures);
        statsConsumer.stat(
            "dkim-stats-update-hard-failures_dmmm",
            hardFailures);

        statsConsumer.stat(
            "dkim-stats-update-on-air_ammm",
            executorStats.onAir());
        statsConsumer.stat(
            "dkim-stats-update-retry-tasks_ammm",
            executorStats.retryTasks());
    }

    @Override
    public void addToGolovanPanel(
        final GolovanPanel panel,
        final String statsPrefix)
    {
        String chartsPrefix = statsPrefix + "dkim-stats";
        GolovanChartGroup group =
            new GolovanChartGroup(chartsPrefix, statsPrefix);
        ImmutableGolovanPanelConfig config = panel.config();

        GolovanChart chart = new GolovanChart(
            "-update-queue",
            " average update queue size per instance",
            false,
            false,
            0d);
        chart.addSplitSignal(
            config,
            "div(" + chartsPrefix + "-update-queue_ammm,"
            + statsPrefix + "instance-alive_ammm)",
            0,
            true,
            false);
        group.addChart(chart);

        chart = new GolovanChart(
            "-updates",
            " updates (rps)",
            false,
            true,
            0d);
        chart.addSignal(
            new GolovanSignal(
                chartsPrefix + "-update-count_dmmm",
                config.tag(),
                "updates",
                null,
                1,
                false));
        chart.addSignal(
            new GolovanSignal(
                chartsPrefix + "-update-tasks-count_dmmm",
                config.tag(),
                "tasks",
                null,
                1,
                false));
        group.addChart(chart);

        chart = new GolovanChart(
            "-failures",
            " failures (rps)",
            false,
            true,
            0d);
        chart.addSignal(
            new GolovanSignal(
                chartsPrefix + "-update-failures_dmmm",
                config.tag(),
                "soft failures",
                null,
                1,
                false));
        chart.addSignal(
            new GolovanSignal(
                chartsPrefix + "-update-hard-failures_dmmm",
                config.tag(),
                "hard failures",
                null,
                1,
                false));
        group.addChart(chart);

        chart = new GolovanChart(
            "-executor-tasks",
            " average executor tasks",
            false,
            false,
            0d);
        chart.addSignal(
            new GolovanSignal(
                "div(" + chartsPrefix + "-update-on-air_ammm,"
                + statsPrefix + "instance-alive_ammm)",
                config.tag(),
                "on air tasks",
                null,
                1,
                false));
        chart.addSignal(
            new GolovanSignal(
                "div(" + chartsPrefix + "-update-retry-tasks_ammm,"
                + statsPrefix + "instance-alive_ammm)",
                config.tag(),
                "retry tasks queue size",
                null,
                1,
                false));
        group.addChart(chart);

        panel.addCharts("dkim", null, group);
    }

    private void processTask(
        final UpdateDkimStatsTask task,
        final FutureCallback<Void> callback)
        throws HttpException, IOException
    {
        String commonPrefix =
            "dkim_stat_" + task.from() + '/' + task.day() + '/';
        long timestamp = task.day() * DkimStatsSelector.TIMESTAMP_GRANULARITY;
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.DOLLAR.create(sbw)) {
            writer.startObject();
            writer.key("prefix");
            task.prefix().writeValue(writer);
            writer.key("AddIfNotExists");
            writer.value(true);
            writer.key("docs");
            writer.startArray();

            // Increment `total' counter
            writer.startObject();
            writer.key("url");
            writer.value(commonPrefix + "total");
            writer.key("dkim_stat_day");
            writer.value(timestamp);
            writer.key("dkim_stat_hams");
            writer.startObject();
            writer.key("function");
            writer.value("inc");
            writer.key("args");
            writer.startArray();
            writer.value(task.total());
            writer.endArray();
            writer.endObject();
            writer.endObject();

            long dkimless = task.dkimless();
            if (dkimless > 0) {
                // Increment `none' counter
                writer.startObject();
                writer.key("url");
                writer.value(commonPrefix + "none");
                writer.key("dkim_stat_day");
                writer.value(timestamp);
                writer.key("dkim_stat_hams");
                writer.startObject();
                writer.key("function");
                writer.value("inc");
                writer.key("args");
                writer.startArray();
                writer.value(dkimless);
                writer.endArray();
                writer.endObject();
                writer.endObject();
            }
            for (Map.Entry<String, long[]> entry
                : task.dkimDomains().entrySet())
            {
                writer.startObject();
                writer.key("url");
                writer.value(commonPrefix + entry.getKey());
                writer.key("dkim_stat_day");
                writer.value(timestamp);
                writer.key("dkim_stat_hams");
                writer.startObject();
                writer.key("function");
                writer.value("inc");
                writer.key("args");
                writer.startArray();
                writer.value(entry.getValue()[0]);
                writer.endArray();
                writer.endObject();
                writer.endObject();
            }

            writer.endArray();
            writer.endObject();
        }

        QueryConstructor query =
            new QueryConstructor("/update?dkim-stats");
        query.append("service", service);
        query.append("prefix", task.prefix().prefix());
        query.append("from", task.from());

        AsyncClient client;
        if (statersConfig == null) {
            client = this.client;
        } else {
            client =
                this.client.adjustStater(statersConfig, task.requestInfo());
        }
        client.execute(
            host,
            new BasicAsyncRequestProducerGenerator(
                query.toString(),
                sbw.toString(),
                ContentType.APPLICATION_JSON),
            EmptyAsyncConsumerFactory.OK,
            client.httpClientContextGenerator(),
            callback);
    }

    private class StatingFutureCallback implements FutureCallback<Void> {
        private final long start = TimeSource.INSTANCE.currentTimeMillis();
        private final UpdateDkimStatsTask task;

        StatingFutureCallback(final UpdateDkimStatsTask task) {
            this.task = task;
        }

        @Override
        public void cancelled() {
            logger.warning("Task " + task + " cancelled");
        }

        @Override
        public void completed(final Void result) {
            int tasksCount = task.tasksCount();
            updateStats.accept(new UpdateStat(1, tasksCount));
            long now = TimeSource.INSTANCE.currentTimeMillis();
            logger.info(
                "Task " + task + " ( " + tasksCount
                + " ) completed in " + (now - start)
                + " ms, lag: " + (now - task.start()) + " ms");
        }

        @Override
        public void failed(final Exception e) {
            failures.increment();
            logger.log(
                Level.WARNING,
                "Task " + task + " ( " + task.tasksCount()
                + " ) completed in "
                + (TimeSource.INSTANCE.currentTimeMillis() - start)
                + " ms",
                e);
        }
    }

    private static class UpdateStat {
        public static final UpdateStat ZERO = new UpdateStat(0L, 0L);

        private final long updatesCount;
        private final long tasksCount;

        UpdateStat(final long updatesCount, final long tasksCount) {
            this.updatesCount = updatesCount;
            this.tasksCount = tasksCount;
        }
    }

    private enum UpdateStatAccumulator implements BinaryOperator<UpdateStat> {
        INSTANCE;

        @Override
        public UpdateStat apply(final UpdateStat lhs, UpdateStat rhs) {
            return new UpdateStat(
                lhs.updatesCount + rhs.updatesCount,
                lhs.tasksCount + rhs.tasksCount);
        }
    }
}

