package ru.yandex.msearch.proxy.api.async.suggest;

import java.util.Collections;
import java.util.List;

import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;

import java.util.logging.Level;

import org.apache.http.HttpHost;

import org.apache.http.concurrent.FutureCallback;

import org.apache.http.nio.protocol.HttpAsyncRequestProducer;

import ru.yandex.http.proxy.ProxySession;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.nio.HttpAsyncResponseConsumerFactory;

import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.proxy.AsyncHttpServer;

import ru.yandex.msearch.proxy.api.async.mail.searcher.AbstractPlainSearcher;

import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.CgiParams;

public abstract class AbstractSuggestPlainSearcher<T>
    extends AbstractPlainSearcher<T>
{
    protected final AsyncHttpServer server;
    protected final User user;
    protected final long deadline;
    protected final Function<? super Exception, RequestErrorType> errorClassifier
        = RequestErrorType.ERROR_CLASSIFIER;

    protected final PrefixedLogger logger;

    public AbstractSuggestPlainSearcher(
        final AsyncHttpServer server,
        final SuggestRequest<?> request,
        final User user)
        throws BadRequestException
    {
        super(
            (ProxySession) request.session(),
            searchClient(server, request),
            server.producerClient());

        this.server = server;
        this.user = user;

        this.logger = request.logger();

        long timeout = request.session().params().getLong("timeout", -1L);

        if (timeout > 0) {
            deadline = request.session().requestStartTime() + timeout;
        } else {
            deadline = -1L;
        }
    }

    private static AsyncClient searchClient(
        final AsyncHttpServer server,
        final SuggestRequest<?> request)
        throws BadRequestException
    {
        if (request.cgiParams().getBoolean("fast", false)) {
            return server.fastSearchClient();
        }

        return server.searchClient();
    }

    protected abstract HttpAsyncResponseConsumerFactory<T> consumerFactory();

    @Override
    public void search(
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
            request,
        final FutureCallback<T> callback)
    {
        List<HttpHost> hosts = hosts(server.searchMap(), user);

        DeadlineAnyOfFutureCallback<T> metaCallback =
            new DeadlineAnyOfFutureCallback<>(callback, hosts.size());

        if (deadline > 0) {
            logger.info(
                "Request will expire in "
                    + (System.currentTimeMillis() - deadline) + " ms");

            for (int i = hosts.size(); i-- > 0;) {
                proxySession.subscribeForCancellation(
                    searchClient.execute(
                        Collections.singletonList(hosts.get(i)),
                        request,
                        deadline,
                        consumerFactory(),
                        listener().createContextGeneratorFor(searchClient),
                        metaCallback));
            }
        } else {
            for (int i = hosts.size(); i-- > 0;) {
                proxySession.subscribeForCancellation(
                    searchClient.execute(
                        hosts.get(i),
                        request,
                        consumerFactory(),
                        listener().createContextGeneratorFor(searchClient),
                        metaCallback));
            }
        }
    }

    private class DeadlineAnyOfFutureCallback<T> implements FutureCallback<T> {
        private final FutureCallback<? super T> callback;

        private final AtomicInteger requests;
        private Exception e = null;

        public DeadlineAnyOfFutureCallback(
            final FutureCallback<? super T> callback,
            final int requests)
        {
            this.callback = callback;
            this.requests = new AtomicInteger(requests);
        }

        @Override
        public void cancelled() {
            if (requests.decrementAndGet() == 0) {
                callback.cancelled();
            }
        }

        @Override
        public void completed(final T result) {
            if (requests.getAndSet(0) > 0) {
                callback.completed(result);
            }
        }

        @Override
        public void failed(final Exception e) {
            RequestErrorType errorType = errorClassifier.apply(e);
            if (errorType == RequestErrorType.NON_RETRIABLE) {
                if (requests.getAndSet(0) > 0) {
                    callback.failed(e);
                }
            } else {
                Exception ex = null;
                long now = System.currentTimeMillis();
                synchronized (this) {
                    if (deadline > 0
                        && now >= deadline
                        && requests.getAndSet(0) > 0)
                    {
                        requests.set(0);
                        ex = e;
                    } else {
                        if (this.e == null) {
                            this.e = e;
                        } else {
                            this.e.addSuppressed(e);
                        }
                        if (requests.decrementAndGet() == 0) {
                            ex = this.e;
                            this.e = null;
                        }
                    }
                }

                if (ex != null) {
                    callback.failed(ex);
                }
            }
        }
    }
}
