package ru.yandex.msearch.proxy.api.async.suggest;

import java.io.IOException;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.msearch.proxy.api.async.suggest.united.Target;

public class BasicSuggest implements Suggest {
    public static final String TARGET = "target";
    public static final String SHOW_TEXT = "show_text";
    public static final String SEARCH_TEXT = "search_text";
    public static final String SEARCH_PARAMS = "search_params";

    protected final Target target;
    protected final Map<String, String> searchParams;
    protected final String showText;
    protected final String searchText;
    protected final double[] factors;

    public BasicSuggest(final Target target, final String text) {
        this(target, text, text);
    }

    public BasicSuggest(
        final Target target,
        final String showText,
        final String searchText)
    {
        this(target, showText, searchText, Collections.emptyMap());
    }

    public BasicSuggest(
        final Target target,
        final String showText,
        final String searchText,
        final Map<String, String> params)
    {
        this.target = target;
        this.showText = showText;
        this.searchText = searchText;
        this.factors = SuggestFactors.create();
        if (params.size() <= 0) {
            this.searchParams = Collections.emptyMap();
        } else {
            this.searchParams = new LinkedHashMap<>(params);
        }
    }

    public double[] factors() {
        return factors;
    }

    @Override
    public Target target() {
        return target;
    }

    @Override
    public String showText() {
        return showText;
    }

    @Override
    public String searchText() {
        return searchText;
    }

    @Override
    public Suggest withPrefix(final String prefix) {
        return new BasicSuggest(
            target(),
            showText(),
            prefix + searchText(),
            searchParams);
    }

    @Override
    public void appendToWriter(final JsonWriterBase writer)
        throws IOException
    {
        writer.key(TARGET);
        writer.value(target);
        writer.key(SHOW_TEXT);
        writer.value(showText);
        writer.key(SEARCH_TEXT);
        writer.value(searchText);
        writer.key(SEARCH_PARAMS);
        writer.value(searchParams);
    }

    @Override
    public Map<String, String> searchParams() {
        return searchParams;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        BasicSuggest that = (BasicSuggest) o;

        return searchText.equals(that.searchText);

    }

    @Override
    public int hashCode() {
        return searchText.hashCode();
    }

    @Override
    public String toString() {
        return "BasicSuggest{" +
            "target=" + target +
            ", showText='" + showText + '\'' +
            ", searchText='" + searchText +
            '}';
    }

    public static String toString(final Suggest suggest) {
        StringBuilderWriter sbw =
            new StringBuilderWriter(new StringBuilder(""));
        try (JsonWriter writer = new JsonWriter(sbw)) {
            writer.value(suggest);
        } catch (IOException ioe) {
            throw new RuntimeException(ioe);
        }

        return sbw.toString();
    }
}
