package ru.yandex.msearch.proxy.api.async.suggest;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import org.joda.time.DateTimeZone;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.http.util.YandexHeaders;
import ru.yandex.msearch.proxy.MsearchProxyExperiment;
import ru.yandex.msearch.proxy.api.async.mail.Product;
import ru.yandex.msearch.proxy.api.async.mail.Side;
import ru.yandex.msearch.proxy.api.async.suggest.lang.EnSuggestLanguagePack;
import ru.yandex.msearch.proxy.api.async.suggest.lang.RuSuggestLanguagePack;
import ru.yandex.msearch.proxy.api.async.suggest.lang.SuggestLanguagePack;
import ru.yandex.msearch.proxy.api.async.suggest.lang.SuggestMultiLanguagePack;
import ru.yandex.msearch.proxy.api.async.suggest.united.Target;

import ru.yandex.msearch.proxy.experiment.MalformedUserSplitException;
import ru.yandex.msearch.proxy.experiment.UserSplit;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeIntegerValidator;

import ru.yandex.parser.uri.CgiParams;

public class BasicSuggestRequestParams implements SuggestRequestParams {
    private static final CollectionParser<String, List<String>, Exception>
        ORDER_FIELDS_PARSER = new CollectionParser<>(
            NonEmptyValidator.INSTANCE,
            ArrayList::new);
    private static final CollectionParser<String, Set<String>, Exception>
        SORT_FIELDS_PARSER = new CollectionParser<>(
            NonEmptyValidator.INSTANCE,
            LinkedHashSet::new);
    private static final DateTimeZone MSK_TIMEZONE =
        DateTimeZone.forID("Europe/Moscow");

    private final Target target;
    private final int numdoc;
    private final int page;
    private final Set<String> fields;
    private final List<String> order;
    private final String request;
    private final SuggestLanguagePack language;
    private final DateTimeZone timezone;
    private final Side side;
    private final Product product;
    private final Set<MsearchProxyExperiment> experiments;

    public BasicSuggestRequestParams(final Target target) {
        this.target = target;
        this.fields = Collections.emptySet();
        this.order = Collections.emptyList();
        this.request = null;
        this.numdoc = 0;
        this.page = 0;
        this.language = null;
        this.timezone = DateTimeZone.UTC;
        this.product = Product.UNKNOWN;
        this.side = Side.UNKNOWN;
        this.experiments = Collections.emptySet();
    }

    public BasicSuggestRequestParams(
        final Target target,
        final CgiParams params,
        final Set<MsearchProxyExperiment> experiments,
        final int defaultLimit)
        throws BadRequestException
    {
        this(target, params, experiments, 0, false, defaultLimit);
    }

    public BasicSuggestRequestParams(
        final Target target,
        final CgiParams params,
        final Set<MsearchProxyExperiment> experiments,
        final boolean orderEnabled,
        final int defaultLimit)
        throws BadRequestException
    {
        this(target, params, experiments, 0, orderEnabled, defaultLimit);
    }

    public BasicSuggestRequestParams(
        final Target target,
        final CgiParams params,
        final String exps,
        final int oldLimit,
        final boolean orderEnabled,
        final int defaultLimit)
        throws BadRequestException
    {
        this(
            target,
            params,
            parseExperiments(exps),
            oldLimit,
            orderEnabled,
            defaultLimit);
    }

    public BasicSuggestRequestParams(
        final Target target,
        final CgiParams params,
        final Set<MsearchProxyExperiment> experiments,
        final int oldLimit,
        final boolean orderEnabled,
        final int defaultLimit)
        throws BadRequestException
    {
        this.target = target;
        this.experiments = experiments;

        page = params.get(
            "p",
            0,
            NonNegativeIntegerValidator.INSTANCE);

        if (oldLimit > 0) {
            numdoc = oldLimit;
        } else {
            Integer limit = params.get(
                "limit", null, NonNegativeIntegerValidator.INSTANCE);

            if (limit == null) {
                numdoc = params.get(
                    "numdoc",
                    defaultLimit,
                    NonNegativeIntegerValidator.INSTANCE);
            } else {
                numdoc = limit;
            }
        }

        String langId =
            params.getString(
                "lang",
                RuSuggestLanguagePack.INSTANCE.id())
                .toLowerCase(Locale.ROOT).trim();
        SuggestLanguagePack langPack =
            SuggestMultiLanguagePack.LANGUAGES.get(langId);

        if (langPack == null) {
            langPack = EnSuggestLanguagePack.INSTANCE;
        }

        language = langPack;

        request = params.getString("request", "");

        if (orderEnabled) {
            List<String> orderParams = params.get(
                "order",
                Collections.singletonList("desc"),
                ORDER_FIELDS_PARSER);

            fields = params.get(
                "sort_fields",
                Collections.emptySet(),
                SORT_FIELDS_PARSER);


            if (orderParams.size() == 1) {
                order = Collections.nCopies(fields.size(), orderParams.get(0));
            } else {
                if (orderParams.size() != fields.size()) {
                    throw new BadRequestException(
                        "Order param not correspond sort_fields param");
                }

                order = orderParams;
            }
        } else {
            order = Collections.emptyList();
            fields = Collections.emptySet();
        }
        timezone = params.get(
            "tzoffset",
            MSK_TIMEZONE,
            x -> DateTimeZone.forOffsetMillis(Integer.parseInt(x) * 60000));

        this.side = Side.parse(params);
        this.product = Product.parse(params);
    }

    public BasicSuggestRequestParams(
        final SuggestRequestParams requestParams,
        final CgiParams params,
        final boolean orderEnabled,
        final int defaultLimit)
        throws BadRequestException
    {
        this(
            requestParams.target(),
            params,
            requestParams.experiments(),
            requestParams.length(),
            orderEnabled,
            defaultLimit);
    }

    public BasicSuggestRequestParams(
        final SuggestRequestParams requestParams,
        final CgiParams params,
        final int defaultLimit)
        throws BadRequestException
    {
        this(requestParams, params, false, defaultLimit);
    }

    @Override
    public synchronized int offset() {
        return page * numdoc;
    }

    @Override
    public synchronized int length() {
        return numdoc;
    }

    @Override
    public synchronized Set<String> fields() {
        return fields;
    }

    public synchronized List<String> order() {
        return order;
    }

    @Override
    public String request() {
        return request;
    }

    @Override
    public Target target() {
        return target;
    }

    @Override
    public SuggestLanguagePack language() {
        return language;
    }

    @Override
    public DateTimeZone timezone() {
        return timezone;
    }

    @Override
    public Side side() {
        return side;
    }

    @Override
    public Product product() {
        return product;
    }

    @Override
    public Set<MsearchProxyExperiment> experiments() {
        return experiments;
    }

    @Override
    public boolean experiment(final MsearchProxyExperiment experiment) {
        return experiments.contains(experiment);
    }

    public static Set<MsearchProxyExperiment> parseExperiments(
        final String boxes)
    {
        Set<MsearchProxyExperiment> result = Collections.emptySet();

        if (boxes != null) {
            UserSplit split;
            try {
                split = new UserSplit("none", boxes, "");
            } catch (MalformedUserSplitException mue) {
                return result;
            }

            result = new LinkedHashSet<>();
            for (String exp: split.tests()) {
                MsearchProxyExperiment experiment =
                    MsearchProxyExperiment.find(exp);
                if (experiment != null) {
                    result.add(experiment);
                }
            }
        }

        return result;
    }
}
