package ru.yandex.msearch.proxy.api.async.suggest;

import java.util.ArrayList;
import java.util.List;
import java.util.function.UnaryOperator;
import java.util.regex.Pattern;

import ru.yandex.search.request.util.SearchRequestText;

import ru.yandex.util.string.StringUtils;

public class SuggestRequestText {

    public enum Trim {
        LEFT,
        BOTH,
        NO
    }

    public static final String OR = " OR ";
    public static final String AND = " AND ";
    public static final String ASTERISK = "*";
    protected static final String KEY_VALUE_SEP = ":";
    private static final Pattern SPACES = Pattern.compile(" +");
    private static final String BASIC_CHARS = "#\\\\(){}\\[\\]'?~+:;!^\"\\-,=/";
    private static final String NON_WORD_CHARS = BASIC_CHARS + ASTERISK;
    private static final Pattern NON_WORD_CHARS_PATTERN =
        Pattern.compile(StringUtils.concat('[', NON_WORD_CHARS, ']'));
    public static final Pattern LETTERS_OR_NUMS = Pattern.compile(
        "[^\\p{Alnum}]", Pattern.UNICODE_CHARACTER_CLASS);

    private final List<List<String>> parts;

    public SuggestRequestText(final List<String> texts) {
        this(texts, UnaryOperator.identity());
    }

    public SuggestRequestText(
        final List<String> texts,
        final UnaryOperator<String> wordsModifier)
    {
        parts = new ArrayList<>();

        for (String text: texts) {
            String[] words = buildParts(text);
            List<String> wordsList = new ArrayList<>(words.length);
            for (String word: words) {
                String modified = wordsModifier.apply(word);
                if (modified != null) {
                    wordsList.add(modified);
                }
            }
            if (!wordsList.isEmpty()) {
                parts.add(wordsList);
            }
        }
    }

    public static String ltrim(String s) {
        int i = 0;
        while (i < s.length() && (Character.isWhitespace(s.charAt(i))
            || Character.isSpaceChar(s.charAt(i))))
        {
            i++;
        }

        return s.substring(i);
    }

    public static String[] buildParts(final String text) {
        return buildParts(text, Trim.LEFT);
    }

    public static String[] buildParts(final String text, Trim trim) {
        String normalized = NON_WORD_CHARS_PATTERN.matcher(
            SearchRequestText.normalize(text)).replaceAll(" ");
        String trimmed = normalized;
        if (trim == Trim.LEFT) {
            trimmed = ltrim(normalized);
        } else if (trim == Trim.BOTH) {
            trimmed = normalized.trim();
        }

        if (trimmed.isEmpty()) {
            return new String[0];
        }

        return SPACES.split(trimmed, -1);
    }

    public boolean isEmpty() {
        return parts.isEmpty();
    }

    public List<List<String>> getParts() {
        return parts;
    }

    public StringBuilder buildFields(
        final List<? extends RequestTextField> fields)
    {
        StringBuilder sb = new StringBuilder("(");
        sb = buildFields(sb, fields);
        sb.append(")");
        return sb;
    }

    public StringBuilder buildFields(
        final StringBuilder sb,
        final List<? extends RequestTextField> fields)
    {
        return buildFields(sb, fields, OR);
    }

    public StringBuilder buildFields(
        final StringBuilder sb,
        final List<? extends RequestTextField> fields,
        final String operator)
    {
        sb.append("(");
        for (int i = 0; i < parts.size(); i++) {
            List<String> request = parts.get(i);
            for (int j = 0; j < fields.size(); j++) {
                if (j != 0) {
                    sb.append(operator);
                }

                sb.append(fields.get(j).name());
                sb.append(KEY_VALUE_SEP);
                fields.get(j).processForRequest(sb, request);
            }

            if (i != parts.size() - 1) {
                sb.append(")");
                sb.append(operator);
                sb.append("(");
            }
        }
        sb.append(")");

        return sb;
    }
}
