package ru.yandex.msearch.proxy.api.async.suggest.contact;

import java.util.Collection;

import org.apache.james.mime4j.codec.DecodeMonitor;
import org.apache.james.mime4j.codec.DecoderUtil;

import ru.yandex.msearch.proxy.api.async.mail.PerfieldScorer;

import ru.yandex.search.result.SearchDocument;

public class ContactParser {
    protected static String removeQuotes(final String str) {
        if (str.indexOf('"') == -1) {
            return str;
        }
        StringBuilder sb = new StringBuilder();
        boolean addNext = false;
        for (int i = 0; i < str.length(); i++) {
            char c = str.charAt(i);
            if (addNext) {
                sb.append(c);
                addNext = false;
            }
            if (c == '\\') {
                addNext = true;
                continue;
            } else if (c == '"') {
                continue;
            }
            sb.append(c);
        }
        return sb.toString();
    }

    private static int scanEnd(
        final String str,
        final int from,
        final int upto)
    {
        for (int i = from; i < upto; i++) {
            if (str.charAt(i) == ' '
                || str.charAt(i) == '\n'
                || str.charAt(i) == '\t'
                || str.charAt(i) == '\r'
                || str.charAt(i) == '\"'
                || str.charAt(i) == '\''
                || str.charAt(i) == '\\'
                || str.charAt(i) == ','
                || str.charAt(i) == '<'
                || str.charAt(i) == '>')
            {
                return i;
            }
        }

        return upto;
    }

    private static int scanStart(
        final String str,
        final int from,
        final int to)
    {
        for (int i = from; i >= to; i--) {
            if (str.charAt(i) == ' '
                || str.charAt(i) == '\n'
                || str.charAt(i) == '\t'
                || str.charAt(i) == '\r'
                || str.charAt(i) == '\"'
                || str.charAt(i) == '\''
                || str.charAt(i) == '\\'
                || str.charAt(i) == ','
                || str.charAt(i) == '<'
                || str.charAt(i) == '>')
            {
                return i + 1;
            }
        }

        return 0;
    }

    private static Email createEmail(
        final String str,
        final int emailStart,
        final int emailEnd)
    {
        String name;
        String address;
        if (emailStart > 0) {
            String possibleName = str.substring(0, emailStart);
            name = removeQuotes(possibleName)
                .replace(',', ' ')
                .replace('\"', ' ')
                .replace('\'', ' ')
                .replace('\\', ' ')
                .replace('<', ' ')
                .replace('>', ' ')
                .trim();
        } else {
            name = "";
        }

        address = str.substring(emailStart, emailEnd)
            .replace('\"', ' ')
            .replace('\'', ' ')
            .replace('\\', ' ')
            .replace(',', ' ')
            .replace('<', ' ')
            .replace('>', ' ')
            .replace(')', ' ')
            .replace('(', ' ')
            .trim();

        return new Email(name, address);
    }

    public static Email parse(
        final SearchDocument doc,
        final String header,
        final Collection<String> excludeSet)
    {
        String headerStr = doc.attrs().get(header);
        if (headerStr == null) {
            return null;
        }

        headerStr = headerStr.trim();
        if (headerStr.length() == 0 || excludeSet.contains(headerStr)) {
            return null;
        }

        Email email = parseEmail(headerStr);

        if (email == null || excludeSet.contains(email.address)) {
            return null;
        }

        return email;
    }

    public static Email parseEmail(final String str) {
        if (str.startsWith("=?") && str.endsWith("?=")
            || str.startsWith("<=?") && str.endsWith("?=>"))
        {
            String decoded =
                DecoderUtil.decodeEncodedWords(str, DecodeMonitor.SILENT);
            if (!decoded.equals(str)) {
                return parseEmail(decoded);
            }
        }

        int firstAt = str.indexOf('@');
        if (firstAt == -1) {
            return new Email("", str);
        }

        int emailStart = scanStart(str, firstAt, 0);
        int emailEnd = scanEnd(str, firstAt, str.length());

        Email email = createEmail(str, emailStart, emailEnd);
        if (emailStart > 0 && email.name.length() == 0) {
            //"email@address" email@address.com
            int secondAt = str.indexOf('@', firstAt + 1);
            if (secondAt != -1) {
                emailStart = scanStart(str, secondAt, firstAt);
                emailEnd = scanEnd(str, secondAt, str.length());
                return createEmail(str, emailStart, emailEnd);
            }
        }

        return email;
    }

    public static class Email {
        private String address;
        private String name;

        public Email(final String name, final String address) {
            this.name = name;
            this.address = address;
        }

        public String address() {
            return address;
        }

        public String name() {
            return name;
        }

        @Override
        public String toString() {
            return "Email{" +
                "address='" + address + '\'' +
                ", name='" + name + '\'' +
                '}';
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }

            Email email = (Email) o;

            return address.equals(email.address);
        }

        @Override
        public int hashCode() {
            return address.hashCode();
        }
    }
}
