package ru.yandex.msearch.proxy.api.async.suggest.contact;

import java.util.Collection;
import java.util.Locale;

import ru.yandex.msearch.proxy.api.async.suggest.highlight
    .HighlightedContactSuggest;
import ru.yandex.msearch.proxy.highlight.HtmlHighlighter;
import ru.yandex.msearch.proxy.api.async.suggest.contact.ContactParser.Email;
import ru.yandex.msearch.proxy.api.async.suggest.united.Target;
import ru.yandex.parser.email.MailAliases;

public class ContactSuggestBuilder {
    protected Target target = Target.CONTACT;
    protected String rawEmail;
    protected String rawDisplayName;
    protected long timestamp;
    protected String prefix = "";
    protected String postfix = "";
    protected int boost = 0;
    protected int unreadCnt = 0;
    protected Collection<String> requests;
    protected boolean highlight = false;

    protected ContactSuggestBuilder() {
    }

    public ContactSuggestBuilder email(final Email email) {
        return user(email.name(), email.address());
    }

    public ContactSuggestBuilder user(
        final String displayName,
        final String email)
    {
        this.rawEmail = email;
        this.rawDisplayName = displayName;

        return this;
    }

    public ContactSuggestBuilder prefix(final String prefix) {
        this.prefix = prefix;
        return this;
    }

    public ContactSuggestBuilder postfix(final String postfix) {
        this.postfix = postfix;
        return this;
    }

    public ContactSuggestBuilder ts(final long timestamp) {
        this.timestamp = timestamp;
        return this;
    }

    public ContactSuggestBuilder boost(final int boost) {
        this.boost = boost;
        return this;
    }

    public ContactSuggestBuilder unreadCount(final int uc) {
        this.unreadCnt = uc;
        return this;
    }

    protected String buildShowText(final String displayName, final String email) {
        return "\"" + displayName + "\" " + email;
    }

    public ContactSuggestBuilder highlight(final Collection<String> requests) {
        this.highlight = true;
        this.requests = requests;
        return this;
    }

    protected ContactSuggest doHighlight(
        final String displayName,
        final String email,
        final ContactSuggest suggest)
    {
        String shinyName = null;
        String shinyEmail = null;

        String lowName = displayName.toLowerCase(Locale.ROOT);
        String lowEmail = email.toLowerCase(Locale.ROOT);

        for (String request : requests) {
            if (request == null || request.isEmpty()) {
                continue;
            }

            String lowRequest = request.toLowerCase(Locale.ROOT);

            if (shinyName == null) {
                int index = lowName.indexOf(lowRequest);
                if (index != -1) {
                    shinyName = HtmlHighlighter.INSTANCE.highlightString(
                        displayName,
                        index,
                        index + request.length());
                }
            }

            if (shinyEmail == null) {
                int index = lowEmail.indexOf(lowRequest);
                if (index != -1) {
                    shinyEmail = HtmlHighlighter.INSTANCE.highlightString(
                        email,
                        index,
                        index + request.length());
                }
            }
        }
        if (shinyName == null) {
            shinyName = HtmlHighlighter.HTML_TRANSLATOR.translate(displayName);
        }

        if (shinyEmail == null) {
            shinyEmail = HtmlHighlighter.HTML_TRANSLATOR.translate(email);
        }

        return new HighlightedContactSuggest(suggest, shinyEmail, shinyName);
    }

    public ContactSuggest build() {
        String displayName = formName(rawDisplayName, rawEmail);
        String searchTxt = prefix + rawEmail + postfix;
        String email = rawEmail;

        String showText = buildShowText(displayName, email);
        ContactSuggest suggest = new BasicContactSuggest(
            target,
            displayName,
            email,
            showText,
            searchTxt,
            prefix,
            formHash(rawDisplayName, rawEmail),
            timestamp,
            unreadCnt,
            boost);

        if (highlight) {
            suggest = doHighlight(displayName, email, suggest);
        }

        return suggest;
    }

    protected static String formHash(
        final String nameRaw,
        final String address)
    {
        String norm = MailAliases.INSTANCE.normalizeEmail(address);

        String name = nameRaw.trim();

        if (!name.isEmpty() && !name.equalsIgnoreCase(address)) {
            return "\"" + name + "\" " + norm;
        }

        int at = norm.indexOf("@");
        if (at == -1) {
            return "\"" + norm + "\" " + norm;
        }

        return "\"" + norm.substring(0, at) + "\" " + norm;
    }

    public static String formName(
        final String nameRaw,
        final String address)
    {
        String name = nameRaw.trim();

        if (name.equals("")) {
            name = address;
        } else if (!name.equalsIgnoreCase(address)) {
            return name;
        }

        String[] parts = name.split("@");
        if (parts.length > 0) {
            name = parts[0];
        }

        return name;
    }

    public static ContactSuggestBuilder create(final Email email) {
        ContactSuggestBuilder builder = new ContactSuggestBuilder();
        return builder.email(email);
    }
}
