package ru.yandex.msearch.proxy.api.async.suggest.contact;

import java.util.List;
import java.util.Locale;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.ProxyParams;
import ru.yandex.msearch.proxy.api.async.mail.PerfieldScorer;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestLuceneRequest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequest;

import ru.yandex.parser.searchmap.User;

import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;

import ru.yandex.search.document.mail.FolderType;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.PrefixType;

import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.util.string.StringUtils;

public class ContactSuggestLuceneRequest implements SuggestLuceneRequest {
    private static final String GET =
        "received_date,message_type,uid,hdr_from,hdr_to,hdr_cc";
    private static final String CORP_ADDITIONAL_GET = "reply_to_normalized";

    private static final String RECEIVED = "senders_received_count";
    private static final String SENT = "senders_sent_count";
    private static final String LAST_CONTACTED = "senders_last_contacted";
    private static final String CUT = "_cut";
    private static final String JOIN = "_join_req";

    private static final int MULTIPLIER = 2;

    private final  User user;
    private final String luceneRequest;
    private final long failoverSearchDelay;

    public ContactSuggestLuceneRequest(
        final AsyncHttpServer server,
        final SuggestRequest<ContactSuggests> request,
        final PrefixedLogger logger)
        throws BadRequestException
    {
        String queryText;

        CgiParams params = request.cgiParams();
        String luceneRequest = params.getString("groupRequest");
        List<String> headers = params.getAll("headers");
        String filterByContacts = params.getString("filterByContacts", "");

        String mdb = params.getString("mdb");
        PrefixType prefixType = server.searchMap().prefixType(mdb);
        Prefix suid;
        Prefix uid;
        Prefix prefix;

        if (mdb.equals(ProxyParams.PG)) {
            suid = params.get(ProxyParams.SUID, null, prefixType);
            uid = params.get(ProxyParams.UID, prefixType);
            prefix = uid;
        } else {
            suid = params.get(ProxyParams.SUID, prefixType);
            uid = params.get(ProxyParams.UID, null, prefixType);
            prefix = suid;
        }

        boolean corp = params.getBoolean("corp", false);
        Integer fid;
        if (corp) {
            fid = params.getInt("fid", 1);
        } else {
            fid = params.getInt("fid", null);
        }
        user = new User(server.resolveService(mdb, prefix), prefix);


        QueryConstructor query =
            new QueryConstructor("/search-async-mail-suggest?");

        query.append("sort", "received_date");
        query.append("format", "json");
        query.append("merge_func", "none");
        query.append("prefix", user.prefix().toString());
        query.append("service", user.service());

        StringBuilder get = new StringBuilder(GET);
        if (corp) {
            get.append(',').append(CORP_ADDITIONAL_GET);
        }

        queryText = luceneRequest;
        if (fid != null) {
            if (queryText.isBlank()) {
                queryText = "fid:" + fid;
            } else {
                queryText = "fid:" + fid + " AND " + StringUtils.concat("(", queryText, ")");
            }
        }

        if (queryText.isBlank()) {
            if (mdb.equals("pg")) {
                queryText = "uid:" + uid;
            } else {
                queryText = "suid:" + suid;
            }
        }

        if (!corp) {
            boolean excludeSpam = params.getBoolean("excludeSpam", true);
            boolean excludeTrash = params.getBoolean("excludeTrash", true);
            FolderType folder = params.getEnum(FolderType.class, "folder", null);

            if (folder != null) {
                queryText = StringUtils.concat("(", queryText, ")");
                queryText += " AND folder_type:" + SearchRequestText.fullEscape(folder.name().toLowerCase(Locale.ENGLISH), false);
            } else {
                if (excludeSpam || excludeTrash) {
                    queryText = StringUtils.concat("(", queryText, ")");

                    if (excludeSpam) {
                        queryText += " AND NOT folder_type:spam";
                    }

                    if (excludeTrash) {
                        queryText += " AND NOT folder_type:trash";
                    }
                }
            }
        }

        logger.info("query text: " + queryText);
        query.append("text", queryText);

        if (headers.size() > 1) {
            query.append("group", "thread_id");
        } else if(headers.size() == 1) {
            String header = headers.get(0);
            switch(header) {
                case "hdr_to":
                case "hdr_cc":
                case "hdr_from":
                case "reply_to":
                    query.append("pruning-group-field", header + "_keyword");
                    query.append("force-pruning-group-field", "true");
                    break;
                default:
                    break;
            }
            query.append("group", headers.get(0));
        } else {
            query.append("group", "received_date");
        }

        int limit = request.requestParams().length() * MULTIPLIER;
        query.append("length", String.valueOf(limit));
        query.append("offset", String.valueOf(request.requestParams().offset()));

        if (!filterByContacts.isEmpty()) {
            // TODO remove scorer
            PerfieldScorer scorer = new PerfieldScorer(headers);
            scorer.applyGetFields(get);
            query.append("dp", filterByContacts);
        }
        addDpForGetSendersMailCount(query, headers, get);
        query.append("get", get.toString());

        request.logger().info("Lucene request: " + query.toString());

        this.luceneRequest = query.toString();
        this.failoverSearchDelay = server.config().failoverSearchDelay();
    }

    private void addDpForGetSendersMailCount(
        final QueryConstructor query,
        final List<String> headers,
        final StringBuilder get)
        throws BadRequestException
    {
        query.append("dp", "const(senders_uid_ join_prefix)");
        query.append("dp", "const(_ underscore)");
        for (String header: headers) {
            String norm = header + "_normalized";
            query.append(
                "dp",
                "replace_all(" + norm + ",\n, " + norm + CUT + ')');
            query.append(
                "dp",
                "concat(join_prefix,uid,underscore," + norm + CUT + ' '
                    + header + JOIN + ')');
            query.append(
                "dp",
                "left_join(" + header + JOIN + ",url,,"
                    + RECEIVED + ' ' + RECEIVED + '_' + header + ','
                    + SENT + ' ' + SENT + '_' + header + ','
                    + LAST_CONTACTED + ' ' + LAST_CONTACTED + '_' + header
                    + ')');

            get.append(',').append(SENT).append('_').append(header);
            get.append(',').append(RECEIVED).append('_').append(header);
            get.append(',').append(LAST_CONTACTED).append('_').append(header);
        }
    }

    @Override
    public User user() {
        return user;
    }

    @Override
    public String luceneRequest() {
        return luceneRequest;
    }

    @Override
    public long failoverSearchDelay() {
        return failoverSearchDelay;
    }
}
