package ru.yandex.msearch.proxy.api.async.suggest.contact.rules;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.SequentialCompleteMultiFutureCallback;
import ru.yandex.logger.PrefixedLogger;

import ru.yandex.msearch.proxy.Synonyms;
import ru.yandex.msearch.proxy.api.async.ProxyParams;
import ru.yandex.msearch.proxy.api.async.mail.Side;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRule;
import ru.yandex.msearch.proxy.api.async.suggest.contact.ContactSuggest;
import ru.yandex.msearch.proxy.api.async.suggest.contact.ContactSuggests;
import ru.yandex.msearch.proxy.api.async.suggest.contact.TokenizeRule;
import ru.yandex.msearch.proxy.api.async.suggest.contact.Util;
import ru.yandex.msearch.proxy.api.async.suggest.lang.SuggestLanguagePack.QL;

import ru.yandex.parser.uri.CgiParams;

public class AggregateContactSuggestRule
    implements SuggestRule<ContactSuggests>
{
    private static final List<String> SEARCH_HEADERS =
        Arrays.asList("hdr_to", "hdr_from", "hdr_cc");
    private static final List<String> CORP_SEARCH_HEADERS =
        Arrays.asList("hdr_to", "hdr_from", "hdr_cc", "reply_to");

    private static final int MAX_LENGTH_ONLY_BY_LOGIN_SEARCH = 3;

    private final Synonyms synonyms;
    private final SuggestRule<ContactSuggests> next;

    public AggregateContactSuggestRule(
        final Synonyms synonyms,
        final SuggestRule<ContactSuggests> next)
    {
        this.synonyms = synonyms;
        this.next = next;
    }

    @Override
    public void execute(final SuggestRequest<ContactSuggests> request)
        throws HttpException
    {
        CgiParams params = request.cgiParams();
        PrefixedLogger logger = request.logger();

        String requestStr = params.getString(ProxyParams.REQUEST);
        requestStr = requestStr.replaceAll("[\\\\:\"*]", "");
        requestStr = requestStr.replace('ё', 'e').replace('Ё', 'Е');
        boolean ignoreDomains = !requestStr.isEmpty()
            && requestStr.length() <= MAX_LENGTH_ONLY_BY_LOGIN_SEARCH
            && !requestStr.startsWith("@");

        Side side = request.requestParams().side();
        List<TokenizeRule> rulesArray = Util.parseRequestString(
            synonyms,
            requestStr,
            logger,
            side);
        String filterByContacts = "";
        Collection<String> headers;

        QL ql = params.getEnum(QL.class, "ql", null);

        if (ql == null) {
            if (params.getBoolean("corp", false)) {
                headers = CORP_SEARCH_HEADERS;
            } else {
                headers = SEARCH_HEADERS;
            }
        } else {
            if (ql.equals(QL.BCC) || ql.equals(QL.CC)
                || ql.equals(QL.FROM) || ql.equals(QL.TO))
            {
                headers = Collections.singletonList("hdr_" + ql.toString());
            } else {
                logger.warning("Unsupported ql param " + ql);
                request.callback().completed(new ContactSuggests());
                return;
            }
        }

        AggregateSessionWithPriority session =
            new AggregateSessionWithPriority(
                headers.size(),
                request,
                logger,
                rulesArray,
                ignoreDomains);
        SequentialCompleteMultiFutureCallback<ContactSuggests> callback =
            new SequentialCompleteMultiFutureCallback<>(
                session,
                session.logger,
                headers.size());

        for (int i = 0; i < headers.size(); i++) {
            String header = ((List<String>) headers).get(i);
            String groupRequest =
                Util.generateRequest(rulesArray, header);

            if (ignoreDomains) {
                filterByContacts =
                    Util.generateFilterByContactsDP(rulesArray, header);
            }

            CgiParams groupRequestParams = new CgiParams(params);
            groupRequestParams.replace("groupRequest", groupRequest);
            groupRequestParams.replace("headers", header);
            groupRequestParams.replace("filterByContacts", filterByContacts);

            if (requestStr.isEmpty() && i == 0) {
                groupRequestParams.replace("boost", "1");
            }

            this.next.execute(
                request
                    .withCgiParams(groupRequestParams)
                    .withCallback(callback.next()));
        }
    }

    private static void collectContactSuggests(
        final ContactSuggests source,
        final ContactSuggests target,
        final List<TokenizeRule> rules,
        final boolean ignoreDomains)
    {
        for (ContactSuggest suggest: source) {
            // hook for corp
            if (!suggest.filterable()) {
                target.add(suggest);
                continue;
            }

            String name =
                suggest.displayName().toLowerCase().replace('ё', 'e');
            String address = suggest.emailLow();
            String login;
            String domain;
            int atIndex = address.indexOf('@');
            if (atIndex == -1) {
                login = address;
                domain = "";
            } else {
                login = address.substring(0, atIndex);
                domain = address.substring(atIndex + 1);
            }

            if (ignoreDomains) {
                name = Util.removeTLD(name);
                domain = Util.removeTLD(
                    Util.removePopularDomains(domain));
            }

            Set<String> parts = new HashSet<>(tokenize(name));
            parts.add(name);
            List<String> loginParts = tokenize(login);
            parts.addAll(loginParts);
            if (!loginParts.isEmpty()) {
                parts.add(loginParts.get(loginParts.size() - 1) + "@");
            }
            List<String> domainParts = tokenize(domain);
            parts.addAll(domainParts);
            parts.add(domain);

            if (!ignoreDomains) {
                parts.add(address);
                if (!domainParts.isEmpty()) {
                    parts.add("@" + domainParts.get(0));
                }
                parts.add("@" + domain);
            }
            parts.remove("");
            for (TokenizeRule rule: rules) {
                if (rule.searchable()) {
                    continue;
                }
                if (Util.containsAllTokens(parts, rule.tokens())) {
                    target.add(suggest);
                    break;
                }
            }
        }
    }

    private static List<String> tokenize(final String addr) {
        return Arrays.asList(addr.split("[._\\-@, \\t]"));
    }

    private class AggregateSessionWithPriority
        implements FutureCallback<List<ContactSuggests>>
    {
        private final List<TokenizeRule> rules;
        private final PrefixedLogger logger;
        private final int limit;
        private final FutureCallback<? super ContactSuggests> callback;
        private final boolean ignoreDomains;
        private final long startTs;
        private final int size;

        private volatile boolean done = false;

        private AggregateSessionWithPriority(
            final int size,
            final SuggestRequest<ContactSuggests> request,
            final PrefixedLogger logger,
            final List<TokenizeRule> rules,
            final boolean ignoreDomains)
        {
            this.size = size;
            this.rules = rules;
            this.logger = logger;
            this.limit = request.requestParams().length();
            this.callback = request.callback();
            this.ignoreDomains = ignoreDomains;
            this.startTs = System.currentTimeMillis();
        }

        @Override
        public void failed(final Exception ex) {
            synchronized (this) {
                if (done) {
                    return;
                }
                done = true;
            }
            callback.failed(ex);
        }

        @Override
        public void cancelled() {
            synchronized (this) {
                if (done) {
                    return;
                }
                done = true;
            }
            callback.cancelled();
        }

        @Override
        public void completed(final List<ContactSuggests> resps) {
            if (done) {
                return;
            }
            ContactSuggests result = new ContactSuggests();
            int i = 0;
            for (ContactSuggests resp: resps) {
                if (resp == null) {
                    failed(
                        new Exception(
                            "Contact Request failed with index " + i));
                    return;
                }
                this.logger.info(
                    "Searching suggests callback " + i
                        + " completed in "
                        + (System.currentTimeMillis() - startTs)
                        + " ms, size " + resp.size());

                collect(resp, result);
                this.logger.info("After collection " + result.size());

                if (result.size() >= limit) {
                    logger.info("No need to wait other suggests");
                    synchronized (this) {
                        if (done) {
                            return;
                        }
                        done = true;
                    }
                    callback.completed(result.head(limit));
                    return;
                }
                i++;
            }
            if (resps.size() == size) {
                synchronized (this) {
                    if (done) {
                        return;
                    }
                    done = true;
                }
                logger.info("All suggests callback finished");
                callback.completed(result.head(limit));
            }
        }

        private void collect(
            final ContactSuggests source,
            final ContactSuggests result)
        {
            collectContactSuggests(
                source,
                result,
                rules,
                ignoreDomains);
        }
    }
}
