package ru.yandex.msearch.proxy.api.async.suggest.contact.rules;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.ServerException;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRule;
import ru.yandex.msearch.proxy.api.async.suggest.contact.ContactSuggests;
import ru.yandex.msearch.proxy.config.ImmutableContactSuggestConfig;
import ru.yandex.parser.uri.CgiParams;

public class ContactSuggestRule implements SuggestRule<ContactSuggests> {
    public static final String[] EXCLUDE_NO_EMAILS =
        {
            "No address",
            "no_address",
            "undisclosed-recipients:;",
            "\"No_address\" <>",
            "\"No address\" <>",
            "\"no_address\" <>",
            "\"no address\" <>"
        };

    private final ImmutableContactSuggestConfig config;
    private final SuggestRule<ContactSuggests> next;

    public ContactSuggestRule(
        final ImmutableContactSuggestConfig config,
        final SuggestRule<ContactSuggests> next)
    {
        this.config = config;
        this.next = next;
    }

    @Override
    public void execute(
        final SuggestRequest<ContactSuggests> request)
        throws HttpException
    {
        if (!config.enabled()) {
            request.callback().failed(new ServerException(
                HttpStatus.SC_FORBIDDEN,
                "Suggest api is disabled")
            );

            return;
        }

        CgiParams params = request.cgiParams();

        String requestRaw = params.getString("request", null);

        boolean ignoreEmpty =
            config.showOnEmptyRequest() || params.containsKey("ql");

        if (requestRaw == null) {
            requestRaw = params.getString("q", null);
        }

        if (requestRaw == null) {
            // preserving old alias for now
            requestRaw = params.getString("text", null);

            if (ignoreEmpty && (requestRaw == null || requestRaw.isEmpty())) {
                requestRaw = "";
            } else {
                if (requestRaw == null) {
                    throw new BadRequestException("missing parameter q");
                }

                if (requestRaw.equals("")) {
                    throw new BadRequestException("empty request");
                }
            }
        }

        String db = params.getString("mdb", null);
        if (db == null) {
            db = params.getString("maildb", null);
        }

        if (db == null || db.isEmpty()) {
            // preserving old alias for now
            db = params.getString("db", null);
        }

        params.replace("mdb", db);

        String kps = params.getString("kps", null);
        if (kps != null) {
            params.replace("suid", kps);
        }

        HashSet<String> excludeSet = new HashSet<>();
        HashSet<String> selfEmails = new HashSet<>();
        List<String> emailParams = params.getAll("emails");
        if( emailParams.size() > 0 ) {
            for(String emailParam: emailParams) {
                emailParam = emailParam.trim();
                if (emailParam.isEmpty()) {
                    continue;
                }

                String[] emails = emailParam.split(",");
                for (String email : emails) {
                    email = email.trim();
                    if (!email.isEmpty()) {
                        selfEmails.add(email);
                    }
                }
            }
        }

        excludeSet.addAll(Arrays.asList(EXCLUDE_NO_EMAILS));
        excludeSet.addAll(selfEmails);

        params.replace("self_emails", selfEmails);
        params.replace("exclude", excludeSet);
        params.replace("request", requestRaw);

        if (requestRaw.trim().isEmpty() && !ignoreEmpty) {
            FutureCallback<? super ContactSuggests> callback =
                request.callback();
            this.next.execute(
                request.withCallback(EmptyFutureCallback.INSTANCE));
            callback.completed(new ContactSuggests());
        } else {
            this.next.execute(request);
        }
    }
}
