package ru.yandex.msearch.proxy.api.async.suggest.contact.rules;

import java.util.Collection;
import java.util.List;
import java.util.logging.Level;

import org.joda.time.DateTime;

import ru.yandex.blackbox.BlackboxUserinfo;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;

import ru.yandex.msearch.proxy.AsyncHttpServer;

import ru.yandex.msearch.proxy.api.async.suggest.SuggestLuceneRequest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequest;
import ru.yandex.msearch.proxy.api.async.suggest.contact.ContactParser.Email;
import ru.yandex.msearch.proxy.api.async.suggest.contact.ContactParserUpdated;
import ru.yandex.msearch.proxy.api.async.suggest.contact.ContactSuggestBuilder;
import ru.yandex.msearch.proxy.api.async.suggest.contact.ContactSuggests;
import ru.yandex.msearch.proxy.api.async.suggest.contact
    .WordsStartContactSuggestBuilder;
import ru.yandex.msearch.proxy.api.async.suggest.rules
    .AbstractPlainSuggestCallback;

import ru.yandex.msearch.proxy.api.suggest.Translit;

import ru.yandex.parser.uri.CgiParams;

import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

public class PlainContactSuggestCallback
    extends AbstractPlainSuggestCallback<ContactSuggests>
{
    private static final double APPROX_SECONDS_PER_MONTH = 2628000;

    protected final List<String> headers;
    protected final List<String> excludeSet;
    protected final int boost;
    protected final String prefix;
    protected final boolean highlight;
    protected final Collection<String> requests;
    protected final List<String> selfEmails;

    public PlainContactSuggestCallback(
        final AsyncHttpServer server,
        final SuggestLuceneRequest searchRequest,
        final SuggestRequest<ContactSuggests> suggestRequest)
        throws BadRequestException
    {
        super(server, searchRequest, suggestRequest);

        CgiParams params = suggestRequest.cgiParams();

        excludeSet = params.getAll("exclude");
        headers = params.getAll("headers");
        boost = params.getInt("boost", 0);
        prefix = params.getString("requestPrefix", "");
        highlight = params.getBoolean(
            "highlight",
            server.config().suggestConfig().highlight());
        selfEmails = params.getAll("selfEmails");

        requests = Translit.suggestSet(
            suggestRequest.requestParams().request(),
            suggestRequest.requestParams().side());
    }

    protected void build(
        final ContactSuggests suggests,
        final SearchDocument doc,
        final long yearAgoTs)
    {
        String dateStr = doc.attrs().get("received_date");

        if (dateStr == null) {
            suggestRequest.logger()
                .log(Level.WARNING, "Empty received_date " + doc);
            return;
        }

        long ts;
        try {
            ts = Long.parseLong(dateStr);
        } catch (NumberFormatException nfe) {
            suggestRequest.logger()
                .log(Level.WARNING, "Bad received_date " + doc, nfe);
            return;
        }

        for (String header: headers) {
            List<Email> emails =
                ContactParserUpdated.parse(doc, header, excludeSet);

            for (Email email: emails) {
                if (email == null) {
                    continue;
                }

                ContactSuggestBuilder builder =
                    getBuilder(email, doc, header, ts);
                builder.boost(calcBoost(doc, header, ts, yearAgoTs, email));

                if (highlight) {
                    builder.highlight(requests);
                }
                suggests.add(builder.build());
            }
        }
    }

    @Override
    public void completed(final SearchResult result) {
        ContactSuggests suggests = new ContactSuggests();

        long yearAgoTs = new DateTime().minusYears(1).getMillis() / 1000;
        for (SearchDocument doc: result.hitsArray()) {
            build(suggests, doc, yearAgoTs);
        }

        suggestRequest.logger().info(
            "Search completed, size " + suggests.size());

        suggestRequest.callback().completed(suggests);
    }

    @Override
    protected ContactSuggests empty() {
        return new ContactSuggests();
    }

    protected ContactSuggestBuilder getBuilder(
        final Email email,
        final SearchDocument doc,
        final String header,
        final long ts)
    {
        return WordsStartContactSuggestBuilder.create(email)
            .prefix(prefix)
            .ts(ts);
    }

    protected int calcBoost(
        final SearchDocument doc,
        final String hdr,
        final long receivedDate,
        final long yearAgoTs,
        final Email email)
    {
        if (selfEmails != null && selfEmails.contains(email.address())) {
            return 0;
        }
        int boost = 0;
        long received = parseSendersField(doc, "senders_received_count", hdr);
        long sent = parseSendersField(doc, "senders_sent_count", hdr);
        long lastContacted =
            parseSendersField(doc, "senders_last_contacted", hdr);

        if (lastContacted == 0L) {
            lastContacted = receivedDate;
        }
        long diff = lastContacted - yearAgoTs;
        if (diff <= 0) {
            return 0;
        }

        boost += Math.round((double) diff / APPROX_SECONDS_PER_MONTH);

        // two-way communication
        if (received > 0 && sent > 0) {
            boost += 6;
        }

        String msgType = doc.attrs().get("message_type");
        String uidStr = doc.attrs().get("uid");
        Long uid = null;
        try {
            uid = ValueUtils.asLongOrNull(uidStr);
        } catch (JsonUnexpectedTokenException e) {
            suggestRequest.logger().log(
                Level.WARNING, "Incorrect uid: " + uidStr, e);
        }

        if (msgType != null && uid != null) {
            if (!BlackboxUserinfo.pdd(uid)
                && !BlackboxUserinfo.corp(uid)
                && msgType.contains("5 eticket")
                && msgType.contains("16 s_aviaeticket"))
            {
                boost += 4;
            }
        }
        return Math.min(boost, 99);
    }

    private long parseSendersField(
        final SearchDocument doc,
        final String field,
        final String hdr)
    {
        String fieldStr;
        if (hdr == null) {
            fieldStr = doc.attrs().get(field);
        } else {
            fieldStr = doc.attrs().get(field + '_' + hdr);
        }
        if (fieldStr != null) {
            try {
                long result = Long.parseLong(fieldStr);
                if (result < 0) {
                    result = 0L;
                }
                return result;
            } catch (NumberFormatException e) {
                suggestRequest.logger().log(
                    Level.WARNING,
                    "Bad field " + field + ": " + doc,
                    e);
            }
        }
        return 0L;
    }
}
