package ru.yandex.msearch.proxy.api.async.suggest.folder;

import java.util.Locale;

import java.util.function.Function;
import java.util.regex.Pattern;

import ru.yandex.collection.IntList;
import ru.yandex.util.string.StringUtils;

public class Folder {
    public static final Function<String, String> NORMALIZER =
        s -> s.toLowerCase(Locale.ROOT)
            .replace('ё', 'е')
            .replaceAll("\\s", " ")
            //nbsp
            .replaceAll("\\p{javaSpaceChar}", " ");

    private final Pattern WS = Pattern.compile("\\w+");

    private final String SEP_REPLACE = " › ";

    private final String name;
    private final String path;
    private final String pathHl;
    private final String normalized;
    // index of name part in full path part
    private final int nameIndex;
    private IntList indicies;

    public Folder(final String fPath) {
        this(fPath, null);
    }

    public Folder(final String fPath, final Folder parent) {
        int idx = fPath.lastIndexOf('|');
        String localPath = fPath;
        if (idx > 0 && parent != null) {
            localPath =
                StringUtils.concat(
                    parent.path,
                    "|",
                    fPath.substring(idx + 1).trim());
            idx = parent.path.length();
        } else if (idx > 0 && parent == null) {
            localPath = fPath.substring(idx + 1);
            idx = -1;
        }

        if (idx >= 0) {
            StringBuilder pathSb =
                new StringBuilder(localPath.length() + SEP_REPLACE.length() * 2);
            StringBuilder pathHlSb =
                new StringBuilder(localPath.length() + SEP_REPLACE.length() * 2);
            int nameExtraOffset = 0;
            for (int i = 0; i < localPath.length(); i++) {
                char c = localPath.charAt(i);
                if (c == '|') {
                    pathSb.append(SEP_REPLACE);
                    pathHlSb.append(SEP_REPLACE);
                    nameExtraOffset += SEP_REPLACE.length() - 1;
                } else {
                    pathHlSb.append(c);
                    pathSb.append(c);
                }
            }

            name = localPath.substring(idx + 1);

            this.path = pathSb.toString();
            this.pathHl = pathHlSb.toString();
            this.nameIndex = idx + nameExtraOffset + 1;
        } else {
            this.nameIndex = 0;
            this.name = localPath;
            this.path = localPath;
            this.pathHl = localPath;
        }

        this.normalized = NORMALIZER.apply(name);
        this.indicies = normalizeFolder(name);
    }

    private IntList normalizeFolder(final String name) {
        IntList indexes = new IntList(name.length() >> 1);

        boolean ws = true;
        for (int i = 0; i < name.length(); i++) {
            char c = name.charAt(i);
            if (Character.isWhitespace(c)
                || Character.isSpaceChar(c)
                || Character.isISOControl(c))
            {
                if (!ws) {
                    ws = true;
                }
            } else if (ws) {
                indexes.add(i);
                ws = false;
            }
        }

        return indexes;
    }

    public int match(final String request) {
        for (int i = 0; i < indicies.size(); i++) {
            if (normalized.startsWith(request, indicies.get(i))) {
                return indicies.get(i);
            }
        }

        return -1;
    }

    public String name() {
        return name;
    }

    public String path() {
        return path;
    }

    public String pathHighlight() {
        return pathHl;
    }

    public int nameIndex() {
        return nameIndex;
    }

    public int words() {
        return indicies.size();
    }

    @Override
    public String toString() {
        return "Folder{" +
            "name='" + name + '\'' +
            ", path='" + path + '\'' +
            ", pathHl='" + pathHl + '\'' +
            ", normalized='" + normalized + '\'' +
            ", nameIndex=" + nameIndex +
            ", indicies=" + indicies +
            '}';
    }
}
