package ru.yandex.msearch.proxy.api.async.suggest.folder;

import java.net.URISyntaxException;
import java.util.Collections;

import org.apache.commons.lang3.text.translate.CharSequenceTranslator;
import org.apache.commons.lang3.text.translate.LookupTranslator;
import org.apache.http.HttpException;
import org.apache.http.HttpResponse;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.message.BasicHeader;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;

import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.HttpAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.msearch.proxy.AsyncHttpServer;
import ru.yandex.msearch.proxy.api.async.ProxyParams;
import ru.yandex.msearch.proxy.api.async.mail.SearchRequest;
import ru.yandex.msearch.proxy.api.async.suggest.BasicSuggest;
import ru.yandex.msearch.proxy.api.async.suggest.BasicSuggests;
import ru.yandex.msearch.proxy.api.async.suggest.Suggest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRequest;
import ru.yandex.msearch.proxy.api.async.suggest.SuggestRule;
import ru.yandex.msearch.proxy.api.async.suggest.Suggests;
import ru.yandex.msearch.proxy.api.async.suggest.highlight.HighlightedSuggest;
import ru.yandex.msearch.proxy.api.async.suggest.lang.SuggestLanguagePack;
import ru.yandex.msearch.proxy.api.async.suggest.united.Target;
import ru.yandex.msearch.proxy.highlight.HtmlHighlighter;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.PrefixType;
import ru.yandex.util.string.StringUtils;

public class FolderSuggestRule implements
    SuggestRule<Suggests<? extends Suggest>>
{
    private static final CharSequenceTranslator RSAQUO_TRANSLATOR
        = new LookupTranslator(
        new String[][] {{
            "&rsaquo;",
            "<span class=\"msearch-folder-separator\">&rsaquo;</span>"}});

    private static final String WMI_SUFFIX = "caller=msearch";

    private final AsyncHttpServer server;

    public FolderSuggestRule(final AsyncHttpServer server) {
        this.server = server;
    }

    @Override
    public void execute(
        final SuggestRequest<Suggests<? extends Suggest>> request)
        throws HttpException
    {
        CgiParams params = request.cgiParams();
        if (!params.containsKey("ql")
            && params.getString("request", "").trim().isEmpty())
        {
            request.callback().completed(
                new BasicSuggests(
                    Target.FOLDER,
                    request.requestParams().length()));
            return;
        }

        String mdb = params.getString(ProxyParams.MDB);
        PrefixType prefixType = server.searchMap().prefixType(mdb);
        Prefix suid;
        Prefix uid;
        Prefix prefix;
        if (mdb.equals("pg")) {
            suid = params.get(ProxyParams.SUID, null, prefixType);
            uid = params.get(ProxyParams.UID, prefixType);
            prefix = uid;
        } else {
            suid = params.get(ProxyParams.SUID, prefixType);
            uid = params.get(ProxyParams.UID, null, prefixType);
            prefix = suid;
        }

        boolean corp = SearchRequest.corp(prefix);

        ImmutableURIConfig foldersConfig;
        if (corp) {
            foldersConfig = server.config().corpFoldersConfig();
        } else {
            foldersConfig = server.config().foldersConfig();
        }

        QueryConstructor query = new QueryConstructor(
            new StringBuilder(foldersConfig.uri().toASCIIString())
                .append(foldersConfig.firstCgiSeparator())
                .append(WMI_SUFFIX));
        query.append("mdb", mdb);
        if (uid != null) {
            query.append("uid", uid.toString());
        }

        if (suid != null) {
            query.append("suid", suid.toString());
        }

        String foldersRequest = query.toString();
        AsyncClient foldersClient = server.foldersClient(corp);

        long timeout = params.getLong("timeout", -1L);
        try {
            if (timeout < 0) {
                foldersClient.execute(
                    new HeaderAsyncRequestProducerSupplier(
                        new AsyncGetURIRequestProducerSupplier(foldersRequest),
                        new BasicHeader(YandexHeaders.X_YA_SERVICE_TICKET,
                            server.filterSearchTvm2Ticket(corp))),
                    NormalizedFoldersConsumerFactory.OK,
                    request.session().listener()
                        .createContextGeneratorFor(foldersClient),
                    new FoldersCallback(request));
            } else {
                BasicAsyncRequestProducerGenerator producerGenerator =
                    new BasicAsyncRequestProducerGenerator(foldersRequest);
                producerGenerator.addHeader(
                    YandexHeaders.X_YA_SERVICE_TICKET,
                    server.filterSearchTvm2Ticket(corp));
                foldersClient.execute(
                    Collections.singletonList(foldersConfig.host()),
                    producerGenerator,
                    request.session().requestStartTime() + timeout,
                    NormalizedFoldersConsumerFactory.OK,
                    request.session().listener()
                        .createContextGeneratorFor(foldersClient),
                    new FoldersCallback(request));
            }
        } catch (URISyntaxException ue) {
            request.callback().failed(ue);
        }
    }

    private Suggest buildSuggest(
        final String fid,
        final String prefix,
        final Folder folder,
        final int index,
        final int highlightLen)
    {
        String searchText;
        if (folder.words() > 1) {
            searchText = StringUtils.concat(prefix, "\"", folder.name());
            searchText += "\"";
        } else {
            searchText = StringUtils.concat(prefix, folder.name());
        }

        Suggest suggest =
            new FolderSuggest(
                folder.path(),
                searchText,
                fid
            );

        if (fid != null && highlightLen >= 0) {
            suggest =
                new HighlightedSuggest(
                    suggest,
                    RSAQUO_TRANSLATOR.translate(
                        HtmlHighlighter.INSTANCE.highlightString(
                            folder.pathHighlight(),
                            folder.nameIndex() + index,
                            folder.nameIndex() + index + highlightLen)));
        }

        return suggest;
    }

    private Suggest buildSuggest(
        final String key,
        final String prefix,
        final int highlightLen)
    {
        Suggest suggest =
            new BasicSuggest(
                Target.FOLDER,
                key,
                StringUtils.concat(prefix, key)
            );

        if (key != null && highlightLen >= 0) {
            suggest =
                new HighlightedSuggest(
                    suggest,
                    RSAQUO_TRANSLATOR.translate(
                        HtmlHighlighter.INSTANCE.highlightString(key, 0, highlightLen)));
        }

        return suggest;
    }

    private void suggest(
        final SuggestRequest<Suggests<? extends Suggest>> suggestRequest,
        final Folders folders,
        final boolean pure,
        final boolean highlight)
    {
        CgiParams params = suggestRequest.cgiParams();
        String requestPrefix = params.getString("requestPrefix", "");
        BasicSuggests suggests =
            new BasicSuggests(
                Target.FOLDER,
                suggestRequest.requestParams().length());

        SuggestLanguagePack langPack =
            suggestRequest.requestParams().language();

        StringBuilder sb = new StringBuilder(requestPrefix);
        if (!pure) {
            sb.append(langPack.folder());
            sb.append(':');
        }

        final String prefix = sb.toString();

        for (String request: params.getAll("request")) {
            String normRequest = Folder.NORMALIZER.apply(request);
            final int highlightLen = highlight ? normRequest.length(): -1;

            folders.user().forEach((fid, folder) -> {
                int index = folder.match(normRequest);
                if (index >= 0) {
                    suggests.add(
                        buildSuggest(
                            fid,
                            prefix,
                            folder,
                            index,
                            highlightLen));
                }
            });

            folders.system().forEach((fid, folder) -> {
                String langName = langPack.systemFolders().get(folder);
                if (langName != null) {
                    Folder normalizedFolder = new Folder(langName);

                    int index = normalizedFolder.match(normRequest);

                    if (index >= 0) {
                        suggests.add(
                            buildSuggest(
                                fid,
                                prefix,
                                normalizedFolder,
                                index,
                                highlightLen));
                    }
                }
            });
        }

        suggestRequest.callback().completed(suggests);
    }

    private final class FoldersCallback implements FutureCallback<Folders> {
        final SuggestRequest<Suggests<? extends Suggest>> request;

        private final boolean pure;
        private final boolean highlight;

        public FoldersCallback(
            final SuggestRequest<Suggests<? extends Suggest>> request)
            throws HttpException
        {
            this.request = request;
            this.pure = request.cgiParams().getBoolean("pure", false);
            this.highlight =
                request.cgiParams().getBoolean(
                    "highlight",
                    server.config().suggestConfig().highlight());
        }

        @Override
        public void completed(final Folders folders) {
            suggest(request, folders, pure, highlight);
        }

        @Override
        public void failed(final Exception e) {
            request.callback().failed(e);
        }

        @Override
        public void cancelled() {
            request.callback().cancelled();
        }
    }

    private enum NormalizedFoldersConsumerFactory
        implements HttpAsyncResponseConsumerFactory<Folders>
    {
        INSTANCE;

        public static final StatusCheckAsyncResponseConsumerFactory<Folders> OK =
            new StatusCheckAsyncResponseConsumerFactory<>(
                HttpStatusPredicates.OK,
                INSTANCE);

        @Override
        public SuggestFolderConsumer create(
            final HttpAsyncRequestProducer producer,
            final HttpResponse response)
            throws HttpException
        {
            return new SuggestFolderConsumer(response.getEntity());
        }
    }
}
